/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "backend.h"

Backend *Backend::_instance = 0;

/*!
 * \brief Gets the current instance of the backend object.
 * \return If there is a instance returns it, else creates a new instance then returns it.
 *
 *  Singleton Design Pattern.
 */
Backend *Backend::getInstance(void)
{
    if (!Backend::_instance) {
        Backend::_instance = new Backend();
    }
    return Backend::_instance;
}

/*! 
 * \brief TwitterButton constructor.
 */
Backend::Backend()
    : twitterapi( new TwitterAPI ),
    friendsPage(1),
    isUpdate(false)
{
        QSettings settings("Zagaia","TweeGo");
//        settings.beginGroup("PROXY");
//        if(settings.value("useProxy").toBool()) {
//            m_twitLib->SetProxy(SERVER::HTTP_PROXY, settings.value("ip").toString(),
//                                settings.value("port").toInt(),
//                                settings.value("user").toString(),
//                                settings.value("pass").toString());
//        }
//        settings.endGroup();

        twitterapi->setServiceUrl("https://twitter.com");
        twitterapi->setUsingOAuth(true);

        connect(twitterapi, SIGNAL(credentials(EntryList)), SLOT(_credentials(EntryList)));
        connect(twitterapi, SIGNAL(newEntries(EntryList)), SLOT(_parseEntries(EntryList)));
        connect(twitterapi, SIGNAL(deleteEntry(quint64)), SLOT(_deleteEntry(quint64)));
        connect(twitterapi, SIGNAL(errorMessage(QString)), SLOT(_error(QString)));
}

void Backend::_deleteEntry(quint64 id)
{
    emit deletedTweet(id);
}

void Backend::_credentials(EntryList entries)
{
    userInformations = entries.at(0).userInfo;
    emit signedOn();
}

void Backend::_parseEntries(EntryList entries)
{
    foreach( const Entry &entry, entries )
        if(entry.isOwn)
            userInformations = entry.userInfo;

    if(!isUpdate)
        emit entryList(entries);
    else {
        emit updatePosted(entries);
        isUpdate = false;
    }
}

/*!
 * \brief Try to post a new update on the twitter.
 * \param status Status message.
 */
void Backend::update(QString status, QString lat, QString lon, quint64 inReplyToStatusId)
{
    twitterapi->postUpdate(status/*,inReplyToStatusId*/);
    isUpdate = true;
//    m_isUpdate = true;

//    m_twitLib->PostNewStatus(status, lat, lon, in_reply_to_status_id);

//    if(m_twitLib->head.statusCode() == 200) {
//        emit updatePosted();
//        return true;
//    }
//    else {
//        emit updateError("Update Failed. Retry?");
//        return false;
//    }
}

/*!
 * \brief Try to get a timeline from the twitter.
 * \param type The type of timeline that is requested.
 *
 * The app can download the user, friends and public timelines.
 */
void Backend::getTimeline(Timeline::TimelineType type)
{
    switch(type)
    {
        case Timeline::Users:
            // TODO: implement twitterapi->userTimeline().
            break;
        case Timeline::Friends:
            twitterapi->friendsTimeline(friendsPage);
            friendsPage++;
            break;
        case Timeline::Public:
            twitterapi->publicTimeline();
            break;
    }
}

/*!
 * \brief Verify the user informations.
 *
 * This function is used when the app needs to get the user informations
 */
void Backend::verifyCredentials(void)
{
    twitterapi->verifyCredentials();
}

/*!
 * \brief This function returns a list with the user informations.
 * \return A list with the user informations.
 */
UserInfo Backend::userInfo()
{
    return userInformations;
}

/*!
 * \brief Sets which friends timeline page to download.
 * \param page Page number.
 */
void Backend::setFriendsPage(int page)
{
    friendsPage = page;
}

/*!
 * \brief Gets friend's page number that was last downloaded.
 * \return Number of the last page downloaded.
 */
int Backend::getFriendsPage(void)
{
//    return option1->page;
}

/*!
 * \brief Sets which user timeline page to download.
 * \param page Page number.
 */
void Backend::setUsersPage(int page)
{
//    option2->page = page;
//    option2->screen_name = userInformations.at(0);
}

/*!
 * \brief Gets user's page number that was last downloaded.
 * \return Number of the last downloaded page.
 */
int Backend::getUsersPage(void)
{
//    return option2->page;
}


/*!
 * \brief Adds the status(tweet) as favorite.
 * \param id Tweet's id.
 * \param isFavorite String that check if the tweet is already favorite or not.
 * \return True if successful or false otherwise.
 */
bool Backend::addFavorite(unsigned long long int id, bool isFavorite)
{
//    if(!isFavorite)
//        m_twitLib->AddFavorite(id);
//    else
//        m_twitLib->RemoveFavorite(id);

//    if(m_twitLib->head.statusCode() == 200)
//        return true;
//    else
//        return false;
}

/*!
 * \brief Removes the status(tweet).
 * \param id Status id.
 */
void Backend::removeStatus(quint64 id)
{
    twitterapi->deleteUpdate(id);
}

/*!
 * \brief Removes the user from the following list.
 * \param user Username.
 * \return true if successful or false otherwise.
 */
bool Backend::removeFriendship(QString user)
{
//    m_twitLib->RemoveFriendship(user);

//    if(m_twitLib->head.statusCode() == 200)
//        return true;
//    else
//        return false;
}

/*!
 * \brief Adds the user in the following list.
 * \param user Username.
 * \param follow Condition to follow.
 * \return true if successful or false otherwise.
 */
bool Backend::addFriendship(QString user)
{
//    m_twitLib->AddFriendship(user,true);

//    if(m_twitLib->head.statusCode() == 200)
//        return true;
//    else
//        return false;
}

/*!
 * \brief Send the user a message.
 * \param user Username.
 * \param text The message.
 * \return true if successful or false otherwise.
 */
bool Backend::sendDirectMessage(QString user, QString text)
{
//    m_twitLib->SendDirectMessage(user, text);

//    if(m_twitLib->head.statusCode() == 200)
//        return true;
//    else
//        return false;
}

/*!
 * \brief Change user's profile image.
 * \param image The image.
 * \return true if successful or false otherwise.
 */
bool Backend::updateProfileImage(QString image)
{
//    m_twitLib->UpdateProfileImage(image);

//    if(m_twitLib->head.statusCode() == 200)
//        return true;
//    else
//        return false;
}

/*!
 * \brief Try to get a mentions timeline from the twitter.
 */
void Backend::getMentions()
{
//    m_twitLib->GetRecentMentions(option3);

//    if(m_twitLib->head.statusCode() != 200)
//        emit connectionError("Failed to get mentions. Retry?");
}

/*!
 * \brief Sets which mentions timeline page to download.
 * \param page Page number.
 */
void Backend::setMentionsPage(int page)
{
//    option3->page = page;

//    changeListOrder = true;
}

void Backend::setConsumerKey(const QByteArray &consumerKey)
{
    twitterapi->setConsumerKey(consumerKey);
}

void Backend::setConsumerSecret(const QByteArray &consumerSecret)
{
    twitterapi->setConsumerSecret(consumerSecret);
}

void Backend::setLogin(const QString &login)
{
    twitterapi->setLogin(login);
}

void Backend::setPassword(const QString &password)
{
    twitterapi->setPassword(password);
}

void Backend::abort(void)
{
    twitterapi->abort();
}

void Backend::_error(QString message)
{
    qDebug() << message;
}
