/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "tweet.h"

/*!
 * \brief Class Constructor
 * \param username The username
 * \param text The tweet message
 * \param imageUrl The user's avatar url
 * \param statusId The tweet id
 * \param createdAt When the tweet was created
 * \param source Which app was used when creating the message
 * \param favorite If the tweet is favorite
 * \param coordinates Coordinates where the tweet was made
 * \param isMyTweet If the tweet belongs to the user
 * \param posInList The tweet position in the list
 * \param parent the Tweet parent
 */
Tweet::Tweet(const QString &username, const QString &text, const QString &imageUrl, const QString &statusId,
             const QString &createdAt, const QString &source, const bool &favorite, const QString &coordinates,
             bool isMyTweet, int posInList, QGraphicsWidget *parent)
    : TwitterApplicationItem(parent),
    m_username(username)
{
    paintImage = false;
    this->setMinimumSize(776,74);

    m_username = username;
    m_text = text;
    m_imageUrl = imageUrl;
    m_statusId = statusId;
    m_source = source;
    m_createdAt = createdAt;
    m_favorite = favorite;
    m_coordinates = coordinates;
    m_isMyTweet = isMyTweet;
    m_posInList = posInList;
    m_following = true;

    createTweetMessage(m_username, parseText(m_text), parseCreatedAt(createdAt), parseSource(m_source));

    connect(&m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(handleNetworkRequest(QNetworkReply*)));
    download(imageUrl);
}

/*!
 * \brief Reimplements mouse press event
 * \param event Used to get the mouse click position
 */
void Tweet::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

/*!
 * \brief Reimplements mouse press event
 * \param event Used to get the mouse click position
 */
void Tweet::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if(this->contains(event->pos()))
        emit this->createTweetInfo(this);
}

/*!
 * \brief Writes The whole tweet message
 * \param username The user that made the tweet
 * \param text The tweet message
 * \param creatAt Information about the time of the tweet
 * \param source Information about the origin of the tweet
 */
void Tweet::createTweetMessage(QString username, QString text, QString creatAt, QString source)
{
    QString messageHtml = "<p style=\" font-size:18px; padding-bottom:0px; margin-bottom:0px; font-family:'Nokia Sans';\"> ";
    messageHtml += "<span style=\" color:#0f95da;\">" + username + "</span>";
    messageHtml += "<span style=\" color:#000000;\"> " + text + "</span>";
    messageHtml += "</p>";

    QString messageInfo = "<p align=\"right\"; style=\" padding-top:5px; margin-top:5px; font-size:14px; font-family:'Nokia Sans';\"> ";
    messageInfo += "<span style=\" color=#757575;\">" + creatAt + " via " + source + "</span>";
    messageInfo += "</p>";

    message = new QGraphicsTextItem(this);
    message->setPos(74,0);
    message->setTextWidth(702);
    message->setHtml(messageHtml + messageInfo);

    if(message->document()->size().height() < 56) {
        messageHtml.replace("</p>","<br></p>");
        message->setHtml(messageHtml + messageInfo);
    }

    if(message->document()->size().height() > 74)
        this->setMinimumHeight(message->document()->size().height());
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void Tweet::download(QString Url)
{
    QNetworkRequest request;
    request.setUrl(Url);
    m_manager.get(request);

}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void Tweet::handleNetworkRequest(QNetworkReply *reply)
{
    if(!m_image.load(reply,0))
        m_image = QImage();

    emit imageDownloaded();

    reply->deleteLater();
    paintImage = true;
    update();
}

/*!
 * \brief Paints the Tweet white rectangle and the user avatar
 * \param event Unused
 */
void Tweet::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    QPointF origin = rect().topLeft();

    // Transparent Rect
    painter->setPen(Qt::NoPen);

    if(m_isMyTweet)
        painter->setBrush(QColor(255, 240, 204));
    else
        painter->setBrush(QColor(246, 253, 255));

    painter->drawRoundedRect(origin.x(), origin.y(), rect().width(), (origin.y() + rect().height()), 2, 2);

    // Avatar Rect
    painter->setBrush(Qt::white);
    if(paintImage)
        painter->drawImage(QRect(13,(this->boundingRect().height() - 48)/2,48,48),m_image);
    else
        painter->drawRect(13,(this->boundingRect().height() - 48)/2,51,51);
}

/*!
 * \brief This funcion is a parser for writting the user tweet's source app
 * \param source The source list
 * \return A list with only the app name
 */
QString Tweet::parseSource(QString source)
{
    return source.remove(QRegExp("<[^<]*>"));
}

/*!
 * \brief This funcion is a parser for writting the time of user tweet
 * \param source The list with the time of the tweets
 * \return A list with only the time of the tweets
 */
QString Tweet::parseCreatedAt(QString createdAt)
{
    createdAt.remove(QRegExp("\\+0{4} "));
    QDateTime tweetDate = QDateTime::fromString(createdAt,"ddd MMM dd HH:mm:ss yyyy");
    tweetDate.setTimeSpec(Qt::UTC);

    QString newCreatedAt;
    int seconds = tweetDate.toUTC().secsTo(QDateTime::currentDateTime().toUTC());

    if (seconds < 20) {
        newCreatedAt = "Less than " + QString::number(seconds) + " seconds ago";
    } else if(seconds < 30) {
        newCreatedAt = "About half minute ago";
    } else if(seconds < 60) {
        newCreatedAt = "Less than 1 minute ago";
    } else if(seconds < 3600) {
        newCreatedAt = QString::number(seconds/60) + " minutes ago";
    } else if(seconds < 7200) {
        newCreatedAt = "About 1 hour ago";
    } else if(seconds < 86400) {
        newCreatedAt = "About " + QString::number(seconds/3600) + " hours ago";
    } else {
        newCreatedAt = QString::number(tweetDate.time().hour()) + ":" +
                       QString::number(tweetDate.time().minute()) + " " +
                       QDate::shortMonthName(tweetDate.date().month()) + " " +
                       QString::number(tweetDate.date().day());

        switch(tweetDate.date().day()) {
            case 1:
            case 21:
                newCreatedAt += "st "; break;
            case 2:
            case 22:
                newCreatedAt += "nd "; break;
            case 3:
                newCreatedAt += "rd "; break;
            default:
                newCreatedAt += "th "; break;
        }
    }

    return newCreatedAt;
}

/*!
 * \brief This funcion is a parser for writting the mentions, urls and trends with blue color
 * \param text The tweet message
 * \return The text with the mentions, urls and trends with blue color
 */
QString Tweet::parseText(QString text)
{  
    // Paint #Trends
    QRegExp tag( "#([\\w\\d-]+)( ?)", Qt::CaseInsensitive );
    text.replace( tag, "<font color=#0f95da>#\\1</font>\\2");

    // Paint Links
    QRegExp ahref( "((https?|ftp)://[^ ]+)( ?)", Qt::CaseInsensitive );
    text.replace( ahref, "<font color=#0f95da>\\1</font>\\3" );

    // Paint @Mentions
    text.replace( QRegExp( "(^|[^a-zA-Z0-9])@([\\w\\d_]+(/[\\w\\d-]+)?)" ),
                  QString( "\\1@<font color=#0f95da>\\2</font>"));
    return text;
}

/*!
 * \brief Method to get the username from tweet
 * \return The username
 */
QString Tweet::getUsername(void)
{
    return this->m_username;
}

/*!
 * \brief Method to get the tweet's message from tweet
 * \return The message
 */
QString Tweet::getMessage(void)
{
    return this->m_text;
}

/*!
 * \brief Method to get the avatar's image from tweet
 * \return The avatar image
 */
QImage Tweet::getImage(void)
{
    return this->m_image;
}

/*!
 * \brief Method to get the tweet's id from tweet
 * \return The tweet's id
 */
QString Tweet::getStatusId(void)
{
    return this->m_statusId;
}

/*!
 * \brief Method to get the favorite flag from tweet
 * \return The favorite flag
 */
bool Tweet::getFavorite(void)
{
    return this->m_favorite;
}

/*!
 * \brief Method to get the tweet's coordinates from tweet
 * \return The tweet's coordinates
 */
QString Tweet::getCoordinates(void)
{
    return this->m_coordinates;
}

/*!
 * \brief Method to get the tweet's ownership flag
 * \return The tweet's ownership flag
 */
bool Tweet::getIsMyTweet(void)
{
    return this->m_isMyTweet;
}
/*!
 * \brief Method to get the tweet's position in the list from tweet
 * \return The tweet's position
 */
int Tweet::getPosInList(void)
{
    return this->m_posInList;
}

/*!
 * \brief Method to get the tweet's coordinates from tweet
 * \return The tweet's coordinates
 */
bool Tweet::getFollowing(void)
{
    return this->m_following;
}

/*!
 * \brief Method to get the Tweet message in html format
 * \return The message in html format
 */
QTextDocument* Tweet::getTweetDocument()
{
    return this->message->document();
}

/*!
 * \brief Method to set the tweet's follow flag from tweet
 * \param username The username
 * \param foloow The follow flag
 */
void Tweet::setFollowing(QString username, bool follow)
{
    if(m_username == username)
        m_following = follow;
}

/*!
 * \brief Method to set the favorite flag in tweet
 * \param favorite The new favorite flag
 */
void Tweet::setFavorite(bool favorite)
{
    this->m_favorite = favorite;
}

/*!
 * \brief Method to set the position of the tweet on the list
 * \param position Tweet's position
 */
void Tweet::setPosition(int position)
{
    m_posInList = position;
}
