/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loginview.h"

/*!
 * \brief LoginView constructor.
 */
LoginView::LoginView(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _exitAnim = 0;
}

/*!
 * \brief LoginView destructor.
 */
LoginView::~LoginView()
{
    qDebug() << "DELETED LoginView object : ";
}

/*!
 * \brief Creates the LoginView objects and animations.
 */
void LoginView::createUi(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _createLoginBackground();
    _createEntries();
    _createLabels();
    _createCheckbox();
    _createButtons();
    _createLogo();

    /* Animations */
    _entryAnim = _createEntryAnimation();
    _entryAnim->start();
    _exitAnim = _createExitAnimation();


    connect(qobject_cast<QObject *>(_exitAnim), SIGNAL(finished()), this,
            SLOT(_exitAnimationFinished()));
    connect(_exitButton, SIGNAL(released()), this, SIGNAL(exitLogin()));
    connect(_loginButton, SIGNAL(released()), this, SLOT(_emitLoginAccount()));
    connect(checkbox,SIGNAL(pressed()),this,SLOT(_checked()));
    connect(_entryPassword, SIGNAL(enterPressed()), this, SLOT(_emitLoginAccount()));
    connect(_cancelButton, SIGNAL(released()), this, SLOT(_endLogin()));
}

/*!
 * \brief Emits exit animation signal.
 */
void LoginView::_exitAnimationFinished(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit done();
}

/*!
 * \brief This method creates the entry animation for all the objects in the screen.
 * \return anim the animation entry animation.
 */
QAbstractAnimation *LoginView::_createEntryAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = new QParallelAnimationGroup(this);
    /* Delay time for each animation (milliseconds) */
    int delay = 100;

    QEasingCurve easingCurve;
    easingCurve.setType(QEasingCurve::OutCirc);

    /* Buttons */
    _addOpacityAnim(anim, _loginButton, 0);
    _addPosAnim(anim, _loginButton, QPointF(640, 380 + 30), QPointF(640, 380),
                easingCurve, 0);
    _addOpacityAnim(anim, _exitButton, 0);
    _addPosAnim(anim, _exitButton, QPointF(460, 380 + 30), QPointF(460, 380),
                easingCurve, 0);

    easingCurve.setType(QEasingCurve::OutBack);
    easingCurve.setOvershoot(2.5);

    /* Checkbox */
    _addOpacityAnim(anim, _checkboxLabel, 0);
    _addPosAnim(anim, _checkboxLabel, QPointF(465,300 + 30), QPointF(465,300),
                easingCurve, 0);
    _addOpacityAnim(anim, checkbox, 0);
    _addPosAnim(anim, checkbox, QPointF(670, 295 + 30), QPointF(670,295),
                easingCurve, 0);

    /* Password Entry */
    _addOpacityAnim(anim, _labelPassword, delay * 2);
    _addPosAnim(anim, _labelPassword, QPointF(465,150 + 30), QPointF(465,150), easingCurve, delay * 2);
    _addOpacityAnim(anim, _entryPassword, delay * 2);
    _addPosAnim(anim, _entryPassword, QPointF(465,190 + 30), QPointF(465,190),
                easingCurve, delay * 2);

    /* ID Entry */
    _addOpacityAnim(anim, _labelID, delay * 3);
    _addPosAnim(anim, _labelID, QPointF(465,40 + 30), QPointF(465,40),easingCurve, delay * 3);
    _addOpacityAnim(anim, _entryID, delay * 3);
    _addPosAnim(anim, _entryID, QPointF(465,75 + 30), QPointF(465,75),
                easingCurve, delay * 3);
   
    /* Logo */
    _addPosAnim(anim, _logo, QPointF(0,0), QPointF(-200,0),easingCurve, 0);

    /* Background */
    _addOpacityAnim(anim, _background, 0);

    return anim;
}

/*!
 * \brief This method creates the exit animation for all the objects in the screen.
 * \return anim the exit animation.
 * This method takes the entry animation and sets it backward for the exit animation.
 */
QAbstractAnimation *LoginView::_createExitAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = qobject_cast<QParallelAnimationGroup *>(_createEntryAnimation());

    anim->setDirection(QAbstractAnimation::Backward);
    return anim;
}

/*!
 * \brief Creates all text entries.
 */
void LoginView::_createEntries(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    /* Add the entries to the scene */
    // ID entry
    _entryID = new LineEdit(LineEdit::Short,this);
    _entryID->setOpacity(0);
    _entryID->setPos(QPointF(800, 480));

    // Password entry
    _entryPassword = new LineEdit(LineEdit::Short,this);
    _entryPassword->setOpacity(0);
    _entryPassword->setEchoMode(QLineEdit::Password);
    _entryPassword->setPos(QPointF(800, 480));

    /* Creating a focus focus order */
//    connect(_entryID->getEntry()->widget(), SIGNAL(returnPressed()),
//            _entryPassword, SLOT(selectAll()));

}

/*!
 * \brief Creates all text labels.
 */
void LoginView::_createLabels(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(18);

    /* Add the labels to the scene */
    // ID label
    _labelID = new TwitterApplicationItem(this);
    QGraphicsTextItem *labelIDItem =
        new QGraphicsTextItem(_labelID);
    labelIDItem->setFont(font);
    labelIDItem->setDefaultTextColor(QColor(84,83,83,255));
    labelIDItem->setPlainText(tr("Username or email"));
    _labelID->setOpacity(0);

    // Password label
    _labelPassword = new TwitterApplicationItem(this);
    QGraphicsTextItem *labelPasswordItem =
        new QGraphicsTextItem(_labelPassword);
    labelPasswordItem->setFont(font);
    labelPasswordItem->setDefaultTextColor(QColor(84,83,83,255));
    labelPasswordItem->setPlainText(tr("Password"));
    _labelPassword->setOpacity(0);
}

/*!
 * \brief Creates password checkbox.
 */
void LoginView::_createCheckbox(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    /* Add the checkbox to the scene */
    checkbox = new TwitterCheckBox(TwitterCheckBox::RememberMe, QPointF(0, 0), false, this);
    checkbox->setOpacity(0);

    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(18);

    // Checkbox label
    _checkboxLabel = new TwitterApplicationItem(this);
    QGraphicsTextItem *checkboxLabelItem =
        new QGraphicsTextItem(_checkboxLabel);
    checkboxLabelItem->setFont(font);
    checkboxLabelItem->setDefaultTextColor(QColor(84,83,83,255));
    checkboxLabelItem->setPlainText(tr("Remember Me"));
    _checkboxLabel->setOpacity(0);
}

/*!
 * \brief Creates login background.
 */
void LoginView::_createLoginBackground(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _background = new TwitterApplicationItem(this);
    QGraphicsPixmapItem *background = new QGraphicsPixmapItem(_background);
    background->setPixmap(QPixmap(":images/resources/loginBackground.png"));
    background->setPos(392,20);
}

/*!
 * \brief Creates all buttons.
 */
void LoginView::_createButtons(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _exitButton = new TwitterButton(TwitterButton::Generic, "Exit", this);
    _exitButton->setOpacity(0);

    _loginButton = new TwitterButton(TwitterButton::Generic, "Login", this);
    _loginButton->setOpacity(0);

    _cancelButton = new TwitterButton(TwitterButton::Generic, "Cancel", this);
    _cancelButton->setOpacity(0);
    _cancelButton->setPos(QPoint(640, 380));
    _cancelButton->setZValue(2.0);
}

/*!
 * \brief Creates logo.
 */
void LoginView::_createLogo(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _logo = new TwitterApplicationItem(this);
    QGraphicsPixmapItem *logo = new QGraphicsPixmapItem(_logo);
    logo->setPixmap(QPixmap(":images/resources/logo.png"));
    logo->setPos((800 - _logo->childrenBoundingRect().width())/2,50);
 }

/*!
 * \brief Add the opacity Animation
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 */
void LoginView::_addOpacityAnim(QParallelAnimationGroup *anim, TwitterApplicationItem *item, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("opacity");
    animation->setStartValue(0.0);
    animation->setEndValue(1.0);
    animation->setDuration(500);
    animation->setEasingCurve(QEasingCurve::Linear);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Add the position Animation
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void LoginView::_addPosAnim(QParallelAnimationGroup *anim,
                            TwitterApplicationItem *item,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(1000);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*! \brief Gets text from username textbox
 *  \return username username
*/
QString LoginView::getUsername(void)
{
    QWidget *widget;

    widget = _entryID->getEntry()->widget();
    return (dynamic_cast<QLineEdit*>(widget))->text();
}

/*! \brief Sets the username.
 * \param[in] username account's username.
 */
void LoginView::setUsername(const QString &username)
{
    QWidget *widget;

    widget = _entryID->getEntry()->widget();
    (dynamic_cast<QLineEdit*>(widget))->setText(username);
}

/*! \brief Gets password from password text box.
 *  \return password returns the password
 */
QString LoginView::getPassword(void)
{
    QWidget *widget;

    widget = _entryPassword->getEntry()->widget();
    return (dynamic_cast<QLineEdit*>(widget))->text();
}

/*! \brief Set the password.
 * \param[in] passwd account's password.
 */
void LoginView::setPassword(const QString &passwd)
{
    QWidget *widget;

    widget = _entryPassword->getEntry()->widget();
    (dynamic_cast<QLineEdit*>(widget))->setText(passwd);
}

/*! \brief Checks if the remember me checkbox is checked or not.
 * \return isChecked True if remember me is checked or false otherwise.
 */
bool LoginView::getRememberMe(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    return checkbox->isChecked();
}

/*! \brief Sets the remember me checkbox.
 * \param[in] remember True to remember or False otherwise.
 */
void LoginView::setRememberMe(bool remember)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    checkbox->setChecked(remember);
}

/*!
 * \brief Emits exitLogin signal if user presses exit button.
 */
void LoginView::_emitExitLogin(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit exitLogin();
}

/*!
 * \brief Emits login signal if user presses login button.
 */
void LoginView::_emitLoginAccount(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _connecting();
    _loginButton->setOpacity(0);
    _cancelButton->setOpacity(1.0);
    emit loginAccount();
}

/*!
 * \brief This slot is called when there is a connection error to delete the loader object.
 */
void LoginView::connectionError()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _deleteLoader();
    _exitButton->setEnabled(true);
    _entryID->setEnabled(true);
    _entryPassword->setEnabled(true);
    _loginButton->setEnabled(true);
    checkbox->setEnabled(true);
    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0);
}

/*!
 * \brief This slot starts the exit animation when user is successfully connected
 * or emits done signal if the animation could not be created.
 */
void LoginView::connected(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _loginButton->setOpacity(1.0);
    delete _cancelButton;
    _loader->startExitAnimation();
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief Deletes loader animation.
 */
void LoginView::_deleteLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    delete _loader;
    setEnabled(true);
}

/*!
 * \brief Calls loader after user presses the login button.
 */
void LoginView::_connecting(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _loader = new Loader(this);
    connect(_loader, SIGNAL(finished()), this, SLOT(_deleteLoader()));

    _loader->setZValue(1.0);
    QGraphicsRectItem *bg = _loader->getBackground();
    bg->setOpacity(0.6);

//    setEnabled(false);
    _exitButton->setEnabled(false);
    _entryID->setEnabled(false);
    _entryPassword->setEnabled(false);
    _loginButton->setEnabled(false);
    checkbox->setEnabled(false);

    _loader->startEntryAnimation();
}

/*!
 * \brief Emit the signal to load the name from the file or save the name to a file.
 */
void LoginView::_checked()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit rememberMe();
}

/*!
 * \brief When user hits cancel, this method deletes loader, emits a signal to
 * abort the login, and puts back login button on the screen.
 */
void LoginView::_endLogin()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    emit abort();
    _deleteLoader();
    _exitButton->setEnabled(true);
    _entryID->setEnabled(true);
    _entryPassword->setEnabled(true);
    _loginButton->setEnabled(true);
    checkbox->setEnabled(true);
    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0);
}
