/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "lineedit.h"

/*!
 * \brief LineEdit constructor.
 */
LineEdit::LineEdit(LineEdit::TypeLineEdit type ,QGraphicsWidget *parent)
        : TwitterApplicationItem(parent),
        _type(type)

{
    size = 0;
    setFlags(ItemIsFocusable);

    /* Add a background image */
    createBackground();

    /* Focus line */
    createFocus();

    /* The entry */
    createEntry();

    /* Entry widget' signals */
//    connect(_entryWidget, SIGNAL(clicked()), this, SLOT(selectAll()));
    connect(_entryWidget, SIGNAL(doubleClicked()), this, SLOT(selectAll()));
    connect(_entryWidget, SIGNAL(gainedFocus()), this, SLOT(gainedFocus()));
    connect(_entryWidget, SIGNAL(lostFocus()), this, SLOT(lostFocus()));
//    connect(_entryWidget, SIGNAL(returnPressed()), this, SLOT(lostFocus()));
    connect(this, SIGNAL(key()), this, SLOT(keepImage()));
}

/*!
 * \brief LineEdit destructor.
 */
LineEdit::~LineEdit()
{
}

/*!
 * \brief Sets the entry widget current echo mode.
 */
void LineEdit::setEchoMode(QLineEdit::EchoMode echo)
{
    _entryWidget = (LineEditBox *) _entry->widget();
    _entryWidget->setEchoMode(echo);
}

/*!
 * \brief Called when the item is clicked.
 */
void LineEdit::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
    selectAll();
    update();
}

/*!
 * \brief Called when the item is double-clicked.
 */
void LineEdit::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
    selectAll();
    update();
}

/*!
 * \brief Called when a key is pressed and emits a signal if enter was pressed.
 */
void LineEdit::keyPressEvent(QKeyEvent *event)
{
    if(event->key() == Qt::Key_Return || event->key() == Qt::Key_Enter)
        emit enterPressed();
}

/*!
 * \brief Creates the background image.
 */
void LineEdit::createBackground(void)
{
    _image = new QGraphicsPixmapItem(this);

    switch(_type){
        case Short:
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short.png"));
            break;
        case Long:
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox.png"));
            break;
    }

    _image->setZValue(-1);
}

/*!
 * \brief Creates the focus outilne. By default,
 * the outline is not visible.
 */
void LineEdit::createFocus(void)
{
    _focusLine = new FocusLine(_image->pixmap().width(), _image->pixmap().height()  , this);
    _focusLine->setPos(0, 0);
    _focusLine->setVisible(false);
}

/*!
 * \brief Creates the entry box. A QLineEdit object(QWidget)
 * is added to a QGraphicsProxyWidget.
 */
void LineEdit::createEntry(void)
{
    /* Create a frameless transparent entry box */
    _entryWidget = new LineEditBox();
    QPalette palette(_entryWidget->palette());
    palette.setColor(_entryWidget->backgroundRole(), Qt::transparent);
    palette.setColor(QPalette::Disabled, QPalette::Background, Qt::transparent);
    palette.setBrush(QPalette::Disabled, QPalette::Text, Qt::black);
    _entryWidget->setPalette(palette);
    _entryWidget->setAttribute(Qt::WA_NoSystemBackground);
    _entryWidget->setContextMenuPolicy(Qt::NoContextMenu);
    _entryWidget->setFrame(false);

    switch(_type){
        case Short:
            _entryWidget->setGeometry(5, 15, 255, 32);
            break;
        case Long:
            _entryWidget->setGeometry(50, 10, 572, 32);
            break;
    }

    switch(_type){
        case Long:
            _entryWidget->setMaxLength(1139);
            break;
    }

    /* Widget font */
    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(18);

    /* Proxy widget to be added to the scene */
    _entry = new QGraphicsProxyWidget(_image);
    _entry->setFlag(ItemIgnoresParentOpacity);
    _entry->setFont(font);
    _entry->setWidget(_entryWidget);
    _entry->setZValue(1);
}

/*!
 * \brief Sets the cursor position.
 * \param pos Cursor's position
 */
void LineEdit::setCursorPosition(int pos)
{
    _entryWidget->setCursorPosition(pos);
}

/*!
 * \brief Gets the length of the text.
 * \return Text length
 */
int LineEdit::getTextLength()
{
    return _entryWidget->text().length();
}

/*!
 * \brief Sets the 1ength of the text.
 */
void LineEdit::setTextLength(int newSize)
{
    size = newSize;
}

/*!
 * \brief Returns the QGraphicsProxyWidget object
 * that contains the entry.
 */
QGraphicsProxyWidget *LineEdit::getEntry(void)
{
    return _entry;
}

/*!
 * \brief Selects the entire line.
 */
void LineEdit::selectAll(void)
{
    if (!_entryWidget->text().isEmpty()) {
        _entryWidget->selectAll();
    }

    _entryWidget->setFocus();
}

/*!
 * \brief Called when the QLineEdit object has acquired focus.
 */
void LineEdit::gainedFocus(void)
{
    if(_type == Long) {
        _focusLine->setVisible(false);
        if(size > 140)
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_red.png"));
        else
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_pressed.png"));
    }
    else
        _focusLine->setVisible(true);
    update();
}

/*!
 * \brief Called when the QLineEdit object has lost focus.
 */
void LineEdit::lostFocus(void)
{
    _focusLine->setVisible(false);
    if(_type == Long)
        _image->setPixmap(QPixmap(":images/resources/textboxes/textbox.png"));
    else
        _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short.png"));


    _entryWidget->deselect();
    _entryWidget->clearFocus();
    update();
}

/*!
 * \brief Makes sure textbox color is the same when typing.
 */
void LineEdit::keepImage()
{
    if(_type == Long){
        if(size > 140)
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_red.png"));
        else
           _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_pressed.png"));
    }
    else
        _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short.png"));
}
