/* Tower Toppler - Nebulus
 * Copyright (C) 2000-2006  Andreas R�ver
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "menu.h"

#include "game.h"
#include "points.h"
#include "bonus.h"
#include "sprites.h"
#include "archi.h"
#include "screen.h"
#include "event.h"
#include "decl.h"
#include "level.h"
#include "sound.h"
#include "leveledit.h"
#include "stars.h"
#include "robots.h"
#include "configuration.h"
#include "highscore.h"

#include <SDL_endian.h>

#include <stdlib.h>
#include <string.h>

#define NUMHISCORES 10
#define HISCORES_PER_PAGE 5


// TODO:  We should remove hardcoded keys, as other languages probably don't
// use 'y' and 'n' as their yes/no.  

static const char *men_exitmenu(Menusystem *m);

static unsigned short menupicture, titledata;
static unsigned char currentmission = 0;

static void men_reload_sprites(Uint8 what) {
  Uint8 pal[3*256];

  if (what & 1) {
    file fi(dataarchive, menudat);

    scr_read_palette(&fi, pal);
    menupicture = scr_loadsprites(&restsprites, &fi, 1, 640, 480, false, pal, 0);
  }

  if (what & 2) {
    file fi(dataarchive, titledat);

    scr_read_palette(&fi, pal);
    titledata = scr_loadsprites(&fontsprites, &fi, 1, SPR_TITLEWID, SPR_TITLEHEI, true, pal, config.use_alpha_font());
  }
}

#ifdef GAME_DEBUG_KEYS
static char *debug_menu_extralife() {
  if (ms) lives_add();
  return _("Extra Life");
}

static char *debug_menu_extrascore(s) {
  if (ms) pts_add(200);
  return _("+200 Points");
}
#endif /* GAME_DEBUG_KEYS */

static const char *
men_main_background_proc(Menusystem *m)
{
  scr_blit(restsprites.data(menupicture), 0, 0);
  scr_blit(fontsprites.data(titledata), (SCREENWID - fontsprites.data(titledata)->w) / 2, 20);
  return NULL;
}

static const char *redefine(Menusystem *parent,const char *label,event_t event); 

static const char *redefine_menu_up(Menusystem *parent) {
  return(redefine(parent,_("Up"),up_ev));
}

static const char *redefine_menu_down(Menusystem *parent) {
  return(redefine(parent,_("Down"),down_ev));
}

static const char *redefine_menu_left(Menusystem *parent) {
  return(redefine(parent,_("Left"),left_ev));
}

static const char *redefine_menu_right(Menusystem *parent) {
  return(redefine(parent,_("Right"),right_ev));
}

static const char *redefine_menu_fire(Menusystem *parent) {
  return(redefine(parent,_("Fire"),fire_ev));
}

static const char *redefine(Menusystem *parent,const char *label,event_t event) {
  static bool in_opt=false;
  static int blink_cnt=0;
  std::string s=label;
  s+=": ";
  if(parent&&(!in_opt)) {
    in_opt=true;
    parent->capture();
    parent->clear_last_event();
  } else if(parent&&in_opt) {
    blink_cnt++;
    if(parent->get_last_event()&&(parent->get_last_event()->get_keysym()!=SDLK_UNKNOWN)) {
      events.set_keydef(event,parent->get_last_event()->get_keysym());
      parent->release();
      in_opt=false;
      blink_cnt=0;
    }
  }
  if(!in_opt || (blink_cnt & 4)) {
    s+=events.get_keydef(event);
  }
  return(s.c_str());
}





static const char *game_options_menu_password(Menusystem *prevmenu) {
  static char buf[50];
  char pwd[PASSWORD_LEN+1];

  if (prevmenu) {
    /* one more character to also copy the termination */
    strncpy(pwd, config.curr_password(), PASSWORD_LEN+1);
    while (!men_input(pwd, PASSWORD_LEN, -1, -1, PASSWORD_CHARS)) ;
    config.curr_password(pwd);
    /* FIXME: change -1, -1 to correct position; Need to fix menu system
     first... */
  }
  snprintf(buf, 50, _("Password: %s"), config.curr_password());
  return buf;
}

static const char *game_options_menu_statustop(Menusystem *prevmenu) {
  static char txt[30];
  if (prevmenu) {
    config.status_top(!config.status_top());
  }
  if (config.status_top()) snprintf(txt, sizeof(txt), "%s %c", _("Status on top"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Status on top"), 3);

  return txt;
}

static const char *game_options_menu_lives(Menusystem *prevmenu) {
  static char buf[50];
  int i;
  if (prevmenu) {
    switch(prevmenu->get_last_event()->get_type()) {
      case right_ev:
	config.start_lives(config.start_lives() + 1);
	if (config.start_lives() > 3) config.start_lives(3);
	break;
      case left_ev:
	config.start_lives(config.start_lives() - 1);
	if (config.start_lives() < 1) config.start_lives(1);
	break;
      default: return NULL;
    }
  }
  snprintf(buf, sizeof(buf)-config.start_lives(),_("Lives: "));
  for (i = 0; i < config.start_lives(); i++)
    sprintf(buf + strlen(buf), "%c", fonttoppler);
  return buf;
}

static const char *
game_options_menu_speed(Menusystem *prevmenu)
{
  // Changing game_speed during a game has no effect until a
  // a new game is started.
  static char buf[50];
  if (prevmenu) {
    switch(prevmenu->get_last_event()->get_type()) {
      case right_ev:
	config.game_speed(config.game_speed() + 1);
	if (config.game_speed() > MAX_GAME_SPEED) config.game_speed(MAX_GAME_SPEED);
	break;
      case left_ev:
	config.game_speed(config.game_speed() - 1);
	if (config.game_speed() < 0) config.game_speed(0);
	break;
      case fire_ev:
	config.game_speed((config.game_speed() + 1) % (MAX_GAME_SPEED+1));
	break;
      default: return NULL;
    }
  }
  snprintf(buf, 50, _("Game Speed: %i"), config.game_speed());
  return buf;
}

static const char *
game_options_bonus(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    config.nobonus(!config.nobonus());
  }
  if (config.nobonus()) snprintf(txt, sizeof(txt),"%s %c", _("Bonus"), 3);
  else snprintf(txt, sizeof(txt),"%s %c", _("Bonus"), 4);

  return txt;
}


static const char *men_game_options_menu(Menusystem *prevmenu) {
  static const char * s = _("Game Options");
  if (prevmenu) {
    Menusystem ms(s, NULL, 0, fontsprites.data(titledata)->h+30);
    ms.add_option(NULL,game_options_menu_password, SDLK_p, MOF_LEFT);
    ms.add_option(NULL, game_options_menu_lives, SDLK_l,
                         (menuoptflags)((int)MOF_PASSKEYS|(int)MOF_LEFT));
    ms.add_option(NULL, game_options_menu_statustop,SDLK_s);
    ms.add_option(NULL, game_options_menu_speed, SDLK_g,
                         (menuoptflags)((int)MOF_PASSKEYS|(int)MOF_LEFT));
    ms.add_option(NULL, game_options_bonus,SDLK_b);
    ms.add_spacer();
    ms.add_option(_("Back"), men_exitmenu);

    ms.run(prevmenu);
  }
  return s;
}

static const char *run_redefine_menu(Menusystem *prevmenu) {
  if (prevmenu) {
    Menusystem ms(_("Redefine Keys"), NULL, 0, fontsprites.data(titledata)->h+30);

    ms.add_option(NULL,redefine_menu_up,SDLK_UNKNOWN, MOF_LEFT);
    ms.add_option(NULL, redefine_menu_down, SDLK_UNKNOWN, MOF_LEFT);
    ms.add_option(NULL, redefine_menu_left, SDLK_UNKNOWN, MOF_LEFT);
    ms.add_option(NULL, redefine_menu_right, SDLK_UNKNOWN, MOF_LEFT);
    ms.add_option(NULL, redefine_menu_fire, SDLK_UNKNOWN, MOF_LEFT);
    ms.add_option(_("Back"), men_exitmenu);

    ms.run(prevmenu);

  }
  return _("Redefine Keys");
}

static const char *
men_options_windowed(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    config.fullscreen(!config.fullscreen());
    scr_reinit();
    SDL_ShowCursor(config.fullscreen() ? 0 : 1);
  }
  if (config.fullscreen()) snprintf(txt, sizeof(txt),"%s %c", _("Fullscreen"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Fullscreen"), 3);

  return txt;
}

static const char *
men_options_sounds(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    if (config.nosound()) {
      config.nosound(false);
      snd_init();
      if (!config.nomusic())
        snd_playTitle();
    } else {
      if (!config.nomusic())
        snd_stopTitle();
      snd_done();
      config.nosound(true);
    }
  }
  if (config.nosound()) snprintf(txt, sizeof(txt),"%s %c", _("Sounds"), 3);
  else snprintf(txt, sizeof(txt),"%s %c", _("Sounds"), 4);

  return txt;
}

  static const char *
men_options_music(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    if (config.nomusic()) {
      config.nomusic(false);
      snd_playTitle();
    } else {
      snd_stopTitle();
      config.nomusic(true);
    }
  }
  if (config.nomusic()) snprintf(txt, sizeof(txt),"%s %c", _("Music"), 3);
  else snprintf(txt, sizeof(txt),"%s %c", _("Music"), 4);

  return txt;
}

  static const char *
men_options_accel(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    if (config.accelerometer_enabled()) {
      config.accelerometer_enabled(false);
      config.proximity_sensor_enabled(false);
    } else {
      config.accelerometer_enabled(true);
      config.proximity_sensor_enabled(true);
    }
  }
  if (config.accelerometer_enabled()) snprintf(txt, sizeof(txt),"%s %c", _("Accelerometer"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Accelerometer"), 3);

  return txt;
}


static void
reload_font_graphics(void) {
  fontsprites.freedata();

  scr_reload_sprites(RL_FONT);
  men_reload_sprites(2);
}

static void
reload_robot_graphics(void) {
  objectsprites.freedata();
  scr_reload_sprites(RL_OBJECTS);
}

static void
reload_layer_graphics(void) {
  layersprites.freedata();
  scr_reload_sprites(RL_SCROLLER);
}

static const char *
men_alpha_font(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    config.use_alpha_font(!config.use_alpha_font());
    reload_font_graphics();
  }
  if (config.use_alpha_font()) snprintf(txt, sizeof(txt),"%s %c", _("Font alpha"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Font alpha"), 3);

  return txt;
}

static const char *
men_alpha_sprites(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    config.use_alpha_sprites(!config.use_alpha_sprites());
    reload_robot_graphics();
  }
  if (config.use_alpha_sprites()) snprintf(txt, sizeof(txt),"%s %c", _("Sprites alpha"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Sprites alpha"), 3);

  return txt;
}

static const char *
men_alpha_layer(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    config.use_alpha_layers(!config.use_alpha_layers());
    reload_layer_graphics();
  }
  if (config.use_alpha_layers()) snprintf(txt, sizeof(txt),"%s %c", _("Scroller alpha"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Scroller alpha"), 3);

  return txt;
}

static const char *
men_alpha_menu(Menusystem *ms)
{
  static char txt[30];
  if (ms) {
    config.use_alpha_darkening(!config.use_alpha_darkening());
  }
  if (config.use_alpha_darkening()) snprintf(txt, sizeof(txt),"%s %c", _("Shadowing"), 4);
  else snprintf(txt, sizeof(txt),"%s %c", _("Shadowing"), 3);

  return txt;
}

static const char *
men_waves_menu(Menusystem *ms)
{
  if (ms) {
    switch(ms->get_last_event()->get_type()) {
      case fire_ev:
	config.waves_type((config.waves_type() + 1) % configuration::num_waves);
	break;
      case right_ev:
	config.waves_type(config.waves_type() + 1);
	if (config.waves_type() >= configuration::num_waves) config.waves_type(configuration::num_waves - 1);
	break;
      case left_ev:
	config.waves_type(config.waves_type() - 1);
	if (config.waves_type() < 0) config.waves_type(0);
	break;
      default: return NULL;
      }
  }
  switch(config.waves_type()) {
    case configuration::waves_nonreflecting: return _("Nonreflecting waves");
    case configuration::waves_simple: return _("Simple waves");
    case configuration::waves_expensive: return _("Expensive waves");
    default: return _("Error");
  }
}

static const char *
men_full_scroller(Menusystem *ms)
{
  if (ms) {
    config.use_full_scroller(!config.use_full_scroller());
  }
  if (config.use_full_scroller()) return _("Complete Scroller");
  else return _("2 layers Scoller");
}


static const char *
men_alpha_options(Menusystem *mainmenu) {
  static const char * s = _("Alpha Options");
  if (mainmenu) {

    Menusystem ms(s, NULL, 0, fontsprites.data(titledata)->h+30);

    ms.add_spacer();
    ms.add_option(NULL, men_alpha_font, SDLK_UNKNOWN, MOF_RIGHT);
    ms.add_option(NULL, men_alpha_sprites, SDLK_UNKNOWN, MOF_RIGHT);
    ms.add_option(NULL, men_alpha_layer, SDLK_UNKNOWN, MOF_RIGHT);
    ms.add_option(NULL, men_alpha_menu, SDLK_UNKNOWN, MOF_RIGHT);
    ms.add_spacer();
    ms.add_option(_("Back"), men_exitmenu);

    ms.run(mainmenu);
  }
  return s;
}

static const char *
men_options_graphic(Menusystem *mainmenu) {
  static const char *s = _("Graphics");
  if (mainmenu) {

    Menusystem ms(s, NULL, 0, fontsprites.data(titledata)->h+30);

    ms.add_option(NULL, men_options_windowed,SDLK_f);
    ms.add_option(NULL, men_alpha_options,SDLK_a);
    ms.add_option(NULL, men_waves_menu, SDLK_w, MOF_PASSKEYS);
    ms.add_option(NULL, men_full_scroller, SDLK_s);
    ms.add_spacer();
    ms.add_option(_("Back"), men_exitmenu);

    ms.run(mainmenu);

  }
  return s;
}

static const char *
men_options(Menusystem *mainmenu) {
  static const char * s = _("Options");
  if (mainmenu) {

    Menusystem ms(s, NULL, 0, fontsprites.data(titledata)->h+30);

    ms.add_option(NULL, men_game_options_menu, SDLK_o);
    ms.add_option(NULL, run_redefine_menu, SDLK_r);
    ms.add_option(NULL, men_options_graphic, SDLK_g );
    ms.add_option(NULL, men_options_sounds, SDLK_s);
    ms.add_option(NULL, men_options_music,SDLK_m);
    ms.add_option(NULL, men_options_accel,SDLK_a);
    ms.add_option(_("Back"), men_exitmenu);

    ms.run(mainmenu); 
  }
  return s;
}

static int hiscores_timer = 0;
static int hiscores_pager = 0;
static int hiscores_state = 0;
static int hiscores_xpos = SCREENWID;
static int hiscores_hilited = -1;
static int hiscores_maxlen_pos = 0;
static int hiscores_maxlen_points = 0;
static int hiscores_maxlen_name = 0;
static int hiscores_maxlen = 0;

static void
  get_hiscores_string(int p, char **pos, char **points, char **name)
{
  Uint32 pt;
  Uint8 tw;

  static char buf1[SCORENAMELEN + 5];
  static char buf2[SCORENAMELEN + 5];
  static char buf3[SCORENAMELEN + 5];

  buf1[0] = buf2[0] = buf3[0] = '\0';

  hsc_entry(p, buf3, &pt, &tw);

  snprintf(buf1, SCORENAMELEN + 5, "%i.", p + 1);
  snprintf(buf2, SCORENAMELEN + 5, "%i", pt);

  *pos = buf1;
  *points = buf2;
  *name = buf3;
}

static void
calc_hiscores_maxlen(int *max_pos, int * max_points, int *max_name)
{
  for (int x = 0; x < hsc_entries(); x++) {
    char *a, *b, *c;
    int clen;

    get_hiscores_string(x, &a, &b, &c);

    clen = scr_textlength(a);
    if (clen > *max_pos) *max_pos = clen;

    clen = scr_textlength(b);
    if (clen < 64) clen = 64;
    if (clen > *max_points) *max_points = clen;

    clen = scr_textlength(c);
    if (clen > *max_name) *max_name = clen;
  }
}

static const char *
men_hiscores_background_proc(Menusystem *ms)
{
  static int blink_r = 120, blink_g = 200, blink_b = 40;
  static int next_page = 0;

  if (ms) {

    if(ms->get_last_event()) {
      ms->set_exit();
    }

    scr_blit(restsprites.data(menupicture), 0, 0);
    scr_blit(fontsprites.data(titledata), (SCREENWID - fontsprites.data(titledata)->w) / 2, 20);

    switch (hiscores_state) {
    case 0: /* bring the scores in */
      if (hiscores_xpos > ((SCREENWID - hiscores_maxlen) / 2)) {
        hiscores_xpos -= 10;
        break;
      } else hiscores_state = 1;
    case 1: /* hold the scores on screen */
      if (hiscores_timer < 100) {
        hiscores_timer++;
        break;
      } else {
        bool filled_page = false;
        bool firstpage = (hiscores_pager == 0);
        int pager = (hiscores_pager + 1) % (NUMHISCORES / HISCORES_PER_PAGE);
        for (int tmp = 0; tmp < HISCORES_PER_PAGE; tmp++) {
          //          int cs = tmp + (pager * HISCORES_PER_PAGE);
          //          if (scores[cs].points || strlen(scores[cs].name)) {
          filled_page = true;
          break;
          //          }
        }
        if (!filled_page && firstpage) {
          hiscores_timer = 0;
          break;
        } else {
          hiscores_state = 2;
          next_page = pager;
        }
      }
    case 2: /* move the scores out */
      if (hiscores_xpos > -(hiscores_maxlen + 40)) {
        hiscores_timer = 0;
        hiscores_xpos -= 10;
        break;
      } else {
        hiscores_state = 0;
        hiscores_xpos = SCREENWID;
        hiscores_pager = next_page;
      }
    default: break;
    }
    for (int t = 0; t < HISCORES_PER_PAGE; t++) {
      int cs = t + (hiscores_pager * HISCORES_PER_PAGE);
      int ypos = (t*(FONTHEI+1)) + fontsprites.data(titledata)->h + FONTHEI*2;
      char *pos, *points, *name;
      get_hiscores_string(cs, &pos, &points, &name);
      if (cs == hiscores_hilited) {
        int clen = hiscores_maxlen_pos + hiscores_maxlen_points + hiscores_maxlen_name + 20 * 2 + 20;
        scr_putbar(hiscores_xpos - 5, ypos - 3,
                   clen, FONTHEI + 3, blink_r, blink_g, blink_b, (config.use_alpha_darkening())?128:255);
      }
      scr_writetext(hiscores_xpos + hiscores_maxlen_pos - scr_textlength(pos), ypos, pos);
      scr_writetext(hiscores_xpos + hiscores_maxlen_pos + 20 + hiscores_maxlen_points - scr_textlength(points), ypos, points);
      scr_writetext(hiscores_xpos + hiscores_maxlen_pos + 20 + 20 + hiscores_maxlen_points, ypos, name);
    }
    scr_color_ramp(&blink_r, &blink_g, &blink_b);
  }
  return _("HighScores");
}

static void show_scores(bool back = true, int mark = -1) {
  static char buf[50];
  snprintf(buf, 50, _("Scores for %s"), lev_missionname(currentmission));
  Menusystem ms(buf, men_hiscores_background_proc, 0, fontsprites.data(titledata)->h + 30);

  hsc_select(lev_missionname(currentmission));

  hiscores_timer = 0;
  if ((mark >= 0) && (mark < NUMHISCORES))
    hiscores_pager = (mark / HISCORES_PER_PAGE);
  else
    hiscores_pager = 0;
  hiscores_state = 0;
  calc_hiscores_maxlen(&hiscores_maxlen_pos, &hiscores_maxlen_points, &hiscores_maxlen_name);
  hiscores_maxlen = hiscores_maxlen_pos + hiscores_maxlen_points + hiscores_maxlen_name + 20;
  hiscores_xpos = SCREENWID;
  hiscores_hilited = mark;

  /* fake options; the empty lines are used by the background proc */
  for (int tmpz = 0; tmpz < HISCORES_PER_PAGE; tmpz++) ms.add_spacer();

  if (back)
    ms.add_option(_("Back"), NULL);
  else
    ms.add_option( _("OK"), NULL);

  ms.run();
}

static void
congrats_background_proc(void)
{
  scr_blit(restsprites.data(menupicture), 0, 0);
  scr_blit(fontsprites.data(titledata), (SCREENWID - fontsprites.data(titledata)->w) / 2, 20);

  /* you can use up to 4 lines of text here, but please check
   * if the text fits onto the screen
   */
  const char * text = _("Congratulations! You are\n"
                        "probably good enough to\n"
                        "enter the highscore table!");

  int ypos = 210;

  for (int pos = 0; text[pos]; pos++)
    if (text[pos] == '\n') ypos -= 40;

  char line[200];
  int pos = 0;
  int linepos = 0;

  while (text[pos]) {

    if (text[pos] == '\n') {
      line[linepos] = 0;
      scr_writetext_center(ypos, line);
      linepos = 0;
      ypos += 40;
    } else {
      if (linepos < 198) {
        line[linepos] = text[pos];
        linepos++;
      }
    }
    pos++;
  }

  line[linepos] = 0;
  scr_writetext_center(ypos, line);

  scr_writetext_center(270, _("Please enter your name"));
}

/* highscores, after the game
 * pt = points,
 * twr = tower reached, -1 = mission finished
 */
static void men_highscore(unsigned long pt, int twr) {

  Uint8 pos = 0xff;

#ifndef GAME_DEBUG_KEYS

  hsc_select(lev_missionname(currentmission));

  /* check, if there is a chance at all to get into the list,
   * if not we don't need to lock the highscoretable
   */
  if (hsc_canEnter(pt)) {

    Menusystem::set_bgproc(congrats_background_proc);

    char name[SCORENAMELEN+1];

#ifndef WIN32
    /* copy the login name into the name entered into the highscore table */
    strncpy(name, getenv("LOGNAME"), SCORENAMELEN);
    name[SCORENAMELEN] = 0; // to be sure we have a terminated string
#else
    /* on systems without login we have no name */
    name[0] = 0;
#endif

    while (!men_input(name, SCORENAMELEN)) ;

    pos = hsc_enter(pt, twr, name);
  }

#endif /* GAME_DEBUG_KEYS */

  show_scores(false, pos);
}

static void
main_game_loop()
{
  unsigned char tower;
  Uint8 anglepos;
  Uint16 resttime;
  int demo = 0;
  int gameresult;
  Uint16 *tmpbuf = NULL;

  if (!lev_loadmission(currentmission)) {
    if (!men_yn(_("This mission contains\n"
                "unknown building blocks.\n"
                "You probably need a new\n"
                "version of Tower Toppler.\n"
                "Do you want to continue?"), false, men_main_background_proc))
      return;
  }

  tower = lev_tower_passwd_entry(config.curr_password());

  gam_newgame();
  bns_restart();
    
  do {
    ttsounds::instance()->startsound(SND_WATER);
    do {
      gam_loadtower(tower);
      scr_settowercolor(lev_towercol_red(), lev_towercol_green(), lev_towercol_blue());
      ttsounds::instance()->setsoundvol(SND_WATER, 128);
      gam_arrival();
      gameresult = gam_towergame(anglepos, resttime, demo, &tmpbuf);
    } while ((gameresult == GAME_DIED) && pts_lifesleft());

    if (gameresult == GAME_FINISHED) {
      gam_pick_up(anglepos, resttime);

      ttsounds::instance()->stopsound(SND_WATER);
      tower++;

      if (tower < lev_towercount()) {

        // load next tower, because its colors will be needed for bonus game
        gam_loadtower(tower);

        if (!config.nobonus())
          if (!bns_game())
            gameresult = GAME_ABORTED;
      }
    } else {
      ttsounds::instance()->stopsound(SND_WATER);
    }
  } while (pts_lifesleft() && (tower < lev_towercount()) && (gameresult != GAME_ABORTED));

  if (gameresult != GAME_ABORTED)
    men_highscore(pts_points(), (tower >= lev_towercount()) ? tower : -1);
}

#ifdef HUNT_THE_FISH
static const char *
men_main_bonusgame_proc()
{
  if (ms) {
    gam_newgame();
    scr_settowercolor(rand() % 256, rand() % 256, rand() % 256);
    lev_set_towercol(rand() % 256, rand() % 256, rand() % 256);
    bns_game();
  }
  return _("Hunt the Fish");
}
#endif /* HUNT_THE_FISH */

static const char *
men_main_startgame_proc(Menusystem *ms)
{
  if (ms) {
    int missioncount = lev_missionnumber();
    switch(ms->get_last_event()->get_type()) {
      case fire_ev: 
	dcl_update_speed(config.game_speed());
	snd_musicVolume(MIX_MAX_VOLUME/4);
	main_game_loop();
	snd_musicVolume(MIX_MAX_VOLUME);
	dcl_update_speed(MENU_DCLSPEED);
	break;
      case right_ev: currentmission = (currentmission + 1) % missioncount; 
	break;
      case left_ev: currentmission = (currentmission + missioncount - 1) % missioncount; 
        break;
      default:
	return NULL;  
    }  
  }
  static char s[30];
  snprintf(s, 30, _("%c Start: %s %c"), fontptrleft, _(lev_missionname(currentmission)), fontptrright);
  return s; 
}

static const char *
men_main_highscore_proc(Menusystem *ms)
{
  if (ms) {
   show_scores();
  }
  return _("Highscores");
}

static const char *
men_main_leveleditor_proc(Menusystem *ms)
{
  if (ms) {
    //snd_stoptitle();
    le_edit();
    //(void)get_nextkey();
    //snd_playtitle();
  }
  return _("Level Editor");
}

static const char *
men_main_timer_proc(Menusystem *ms)
{
  if (ms) {
    Uint8 num_demos = 0;
    Uint8 demos[256];
    Uint16 miss = rand() % lev_missionnumber();
    Uint8 num_towers;

    int demolen;
    Uint16 *demobuf;
    Uint8 anglepos;
    Uint16 resttime;

    for (int tmpm = 0; (tmpm < lev_missionnumber()) && (num_demos == 0); tmpm++) {
      Uint16 tmiss = (miss + tmpm) % lev_missionnumber();

      if (lev_loadmission(tmiss)) {
        num_towers = lev_towercount();

        for (Uint8 idx = 0; (idx < num_towers) && (num_demos < 255); idx++) {
          lev_selecttower(idx);
          lev_get_towerdemo(demolen, demobuf);
          if (demolen) demos[num_demos++] = idx;
        }
      }
    }

    if (num_demos < 1) return NULL;

    lev_selecttower(demos[rand() % num_demos]);
    lev_get_towerdemo(demolen, demobuf);

    dcl_update_speed(config.game_speed());
    gam_newgame();
    ttsounds::instance()->startsound(SND_WATER);
    scr_settowercolor(lev_towercol_red(), lev_towercol_green(), lev_towercol_blue());
    ttsounds::instance()->setsoundvol(SND_WATER, 128);
    rob_initialize();
    (void)gam_towergame(anglepos, resttime, demolen, &demobuf);
    ttsounds::instance()->stopsound(SND_WATER);
    dcl_update_speed(MENU_DCLSPEED);
  }
  return NULL;
}

static const char *men_exitmenu(Menusystem *m) {
  if(m) {
    m->set_exit();
  }
  return(0);
}

static const char *
men_game_return2game(Menusystem *tms)
{
  if (tms) {
    tms->set_exit();
    tms->set_state(0);
  }
  return _("Return to Game");
}

static const char *
men_game_leavegame(Menusystem *tms)
{
  if (tms) {
    tms->set_exit();
    tms->set_state(1);
  }
  return _("Quit Game");
}



#ifdef GAME_DEBUG_KEYS
void run_debug_menu(void) {
  Menusystem ms(_("DEBUG MENU"), NULL, 0, SCREENHEI / 5);

  ms.add_option(NULL, debug_menu_extralife);
  ms.add_option(NULL, debug_menu_extrascore);
  ms.add_option(NULL, NULL);
  ms.add_option(_("Back to Game"), men_exitmenu);

  ms.run(); 
}
#endif

void men_init(void) {
  men_reload_sprites(3);
}

void men_main() {
  Menusystem ms(NULL,men_main_background_proc,0, fontsprites.data(titledata)->h + 30);

  ms.set_timeproc(200, men_main_timer_proc);

  ms.add_option(NULL, men_main_startgame_proc, SDLK_s, MOF_PASSKEYS);
  ms.add_spacer();
  ms.add_option(NULL,men_main_highscore_proc,SDLK_h);
  ms.add_option(NULL,men_options,SDLK_o);
  ms.add_option(NULL,men_main_leveleditor_proc,SDLK_e);
#ifdef HUNT_THE_FISH
  ms.add_option( NULL, men_main_bonusgame_proc);
#endif
  ms.add_spacer();
  ms.add_option(_("Quit"), men_exitmenu, SDLK_q);
  
  ms.run();
}

bool men_game() {
  Menusystem ms(_("Game Options"),NULL, 0, fontsprites.data(titledata)->h+30);
  int  speed = dcl_update_speed(MENU_DCLSPEED);

  ms.add_spacer();
  ms.add_option(NULL,men_game_return2game,SDLK_r);
  ms.add_option(NULL,men_options,SDLK_o);
  ms.add_spacer();
  ms.add_option(NULL, men_game_leavegame,SDLK_q);

  ms.set_wraparound(true);
  ms.run();

  dcl_update_speed(speed);
  return(ms.get_state() != 0);
}

static const char *men_yn_option_yes(Menusystem *ms)
{
  if (ms) {
    ms->set_state(1);
    ms->set_exit();
  } 
  return _("Yes");
}

static const char * men_yn_option_no(Menusystem *ms)
{
  if (ms) {
    ms->set_state(0);
    ms->set_exit();
  } 
  return _("No");
}


unsigned char men_yn(char* s, bool defchoice, menuopt_callback_proc pr) {
  Menusystem ms(s, pr, 0, SCREENHEI / 5);
  ms.add_option(NULL,men_yn_option_no,SDLK_n);
  ms.add_option(NULL,men_yn_option_yes,SDLK_y);
  ms.set_state(defchoice ? 1 : 0);
  ms.run();
  return(ms.get_state());
}

void men_info(const char* s, long timeout, int fire) {
    bool ende=false;
  do {
    Menusystem::call_bgproc();
    scr_writetext_center((SCREENHEI / 5), s);
    if (fire)
      scr_writetext_center((SCREENHEI / 5) + 2 * FONTHEI, (fire == 1) ? _("Press fire") : _("Press space"));
    scr_swap();
    dcl_wait();
    Event *e=events.get_input();
    if (timeout > 0) timeout--;
    if (!timeout) ende = true;
    if(e) {
	if ((fire == 2) && (e->get_keysym() == SDLK_SPACE || e->get_keysym() == SDLK_RETURN)) ende = true;
	else if ((fire != 2) && e->get_type()==fire_ev) ende = true;
	delete(e); 
    }
  } while (!ende); 
}


