#include "artworkcache.h"
#include "music.h"

QHash<QString, QImage> ArtworkCache::m_cache = QHash<QString, QImage>();

ArtworkCache::ArtworkCache(QObject *parent) :
    QObject(parent)
{
}

ArtworkCache::~ArtworkCache() {}

QString ArtworkCache::id(const QUrl &url, const QSize &size) const {
    return QString("%1_%2x%3").arg(url.path().section('/', -3, -3)).arg(size.width()).arg(size.height());
}

void ArtworkCache::processNextRequest() {
    if (!m_queue.isEmpty()) {
        ArtworkRequest request = m_queue.dequeue();
        Artwork *artwork = Music::getArtwork(request.url, request.size);
        this->connect(artwork, SIGNAL(ready(Artwork*)), this, SLOT(onArtworkReady(Artwork*)));
    }
}

QImage ArtworkCache::artwork(const QUrl &url, const QSize &size) {
    QString id = this->id(url, size);

    if (m_cache.contains(id)) {
        return m_cache.value(id);
    }

    if (m_queue.isEmpty()) {
        Artwork *artwork = Music::getArtwork(url, size);
        this->connect(artwork, SIGNAL(ready(Artwork*)), this, SLOT(onArtworkReady(Artwork*)));
    }
    else {
        ArtworkRequest request;
        request.url = url;
        request.size = size;
        m_queue.enqueue(request);
    }

    return QImage();
}

void ArtworkCache::onArtworkReady(Artwork *artwork) {
    switch (artwork->error()) {
    case ReplyError::NoError:
        this->insert(artwork->url(), artwork->image());
        emit artworkReady();
        break;
    default:
        break;
    }

    artwork->deleteLater();

    this->processNextRequest();
}

void ArtworkCache::insert(const QUrl &url, const QImage &image) {
    m_cache.insert(this->id(url, image.size()), image);
}

void ArtworkCache::remove(const QUrl &url, const QSize &size) {
    m_cache.remove(this->id(url, size));
}

void ArtworkCache::remove(const QImage &image) {
    m_cache.remove(m_cache.key(image));
}

void ArtworkCache::clear() {
    m_cache.clear();
}
