#include "playbackqueuemodel.h"
#include "definitions.h"
#include "artworkcache.h"

PlaybackQueueModel::PlaybackQueueModel(QObject *parent) :
    QAbstractListModel(parent),
    m_cache(new ArtworkCache(this))
{
    m_roleNames[SongRoles::IdRole] = "id";
    m_roleNames[SongRoles::TitleRole] = "title";
    m_roleNames[SongRoles::ArtistRole] = "artist";
    m_roleNames[SongRoles::ArtistIdRole] = "artistId";
    m_roleNames[SongRoles::AlbumTitleRole] = "albumTitle";
    m_roleNames[SongRoles::AlbumArtistRole] = "albumArtist";
    m_roleNames[SongRoles::AlbumIdRole] = "albumId";
    m_roleNames[SongRoles::GenreRole] = "genre";
    m_roleNames[SongRoles::FilePathRole] = "filePath";
    m_roleNames[SongRoles::FileSuffixRole] = "fileSuffix";
    m_roleNames[SongRoles::MimeTypeRole] = "mimeType";
    m_roleNames[SongRoles::UrlRole] = "url";
    m_roleNames[SongRoles::ArtworkUrlRole] = "artworkUrl";
    m_roleNames[SongRoles::ArtworkRole] = "artwork";
    m_roleNames[SongRoles::StreamUrlRole] = "streamUrl";
    m_roleNames[SongRoles::YearRole] = "year";
    m_roleNames[SongRoles::DiscNumberRole] = "discNumber";
    m_roleNames[SongRoles::BitRateRole] = "bitRate";
    m_roleNames[SongRoles::DurationRole] = "duration";
    m_roleNames[SongRoles::SizeRole] = "size";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(onArtworkReady()));
}

PlaybackQueueModel::~PlaybackQueueModel() {
    qDeleteAll(m_list);
    m_list.clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> PlaybackQueueModel::roleNames() const {
    return m_roleNames;
}
#endif

int PlaybackQueueModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant PlaybackQueueModel::data(const QModelIndex &index, int role) const {
    if (Song *song = this->get(index)) {
        switch (role) {
        case SongRoles::IdRole:
            return song->id();
        case SongRoles::TitleRole:
            return song->title();
        case SongRoles::ArtistRole:
            return song->artist();
        case SongRoles::ArtistIdRole:
            return song->artistId();
        case SongRoles::AlbumTitleRole:
            return song->albumTitle();
        case SongRoles::AlbumArtistRole:
            return song->albumArtist();
        case SongRoles::AlbumIdRole:
            return song->albumId();
        case SongRoles::GenreRole:
            return song->genre();
        case SongRoles::FilePathRole:
            return song->filePath();
        case SongRoles::FileSuffixRole:
            return song->fileSuffix();
        case SongRoles::MimeTypeRole:
            return song->mimeType();
        case SongRoles::UrlRole:
            return song->url();
        case SongRoles::ArtworkUrlRole:
            return song->artworkUrl();
        case SongRoles::ArtworkRole:
            return m_cache->artwork(song->artworkUrl(), THUMBNAIL_SIZE);
        case SongRoles::StreamUrlRole:
            return song->streamUrl();
        case SongRoles::YearRole:
            return song->year();
        case SongRoles::DiscNumberRole:
            return song->discNumber();
        case SongRoles::DurationRole:
            return song->duration();
        case SongRoles::SizeRole:
            return song->size();
        default:
            break;
        }
    }

    return QVariant();
}

QVariant PlaybackQueueModel::data(int row, int role) const {
    return this->data(this->index(row), role);
}

bool PlaybackQueueModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    Q_UNUSED(index)
    Q_UNUSED(value)
    Q_UNUSED(role)

    return false;
}

bool PlaybackQueueModel::setData(int row, const QVariant &value, int role) {
    return this->setData(this->index(row), value, role);
}

QMap<int, QVariant> PlaybackQueueModel::itemData(const QModelIndex &index) const {
    QMap<int, QVariant> map;

    for (int role = SongRoles::IdRole; role <= SongRoles::SizeRole; role++) {
        map[role] = this->data(index, role);
    }

    return map;
}

QMap<int, QVariant> PlaybackQueueModel::itemData(int row) const {
    return this->itemData(this->index(row));
}

Song* PlaybackQueueModel::get(const QModelIndex &index) const {
    if ((index.row() >= 0) && (index.row() < m_list.size())) {
        return m_list.at(index.row());
    }

    return 0;
}

Song* PlaybackQueueModel::get(int row) const {
    return this->get(this->index(row));
}

void PlaybackQueueModel::clear() {
    this->removeSongs(0, m_list.size());
}

void PlaybackQueueModel::appendSong(Song *song) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size());
    m_list.append(new Song(song));
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void PlaybackQueueModel::appendSongs(QList<Song *> songs) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size() + songs.size() - 1);

    while (!songs.isEmpty()) {
        m_list.append(new Song(songs.takeFirst()));
    }

    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void PlaybackQueueModel::insertSong(int i, Song *song) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i);
        m_list.insert(i, new Song(song));
        this->endInsertRows();
    }
    else {
        this->appendSong(song);
    }

    emit countChanged(this->rowCount());
}

void PlaybackQueueModel::insertSongs(int i, QList<Song *> songs) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i + songs.size() - 1);

        while (!songs.isEmpty()) {
            m_list.insert(i, new Song(songs.takeFirst()));
            i++;
        }

        this->endInsertRows();
    }
    else {
        this->appendSongs(songs);
    }

    emit countChanged(this->rowCount());
}

void PlaybackQueueModel::removeSong(int i) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i);
        m_list.takeAt(i)->deleteLater();
        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void PlaybackQueueModel::removeSongs(int i, int count) {
    if ((i >= 0) && ((i + count) <= m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i + count - 1);

        for (int ii = i; ii < (i + count); ii++) {
            m_list.takeAt(i)->deleteLater();
        }

        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void PlaybackQueueModel::onArtworkReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
