#include "artistmodel.h"
#include "music.h"
#include "notifications.h"
#include "definitions.h"
#include "artworkcache.h"
#include <QStringList>

ArtistModel::ArtistModel(QObject *parent) :
    QAbstractListModel(parent),
    m_cache(new ArtworkCache(this)),
    m_loading(false)
{
    m_roleNames[ArtistRoles::IdRole] = "id";
    m_roleNames[ArtistRoles::NameRole] = "name";
    m_roleNames[ArtistRoles::UrlRole] = "url";
    m_roleNames[ArtistRoles::ArtworkUrlRole] = "artworkUrl";
    m_roleNames[ArtistRoles::ArtworkRole] = "artwork";
    m_roleNames[ArtistRoles::SongCountRole] = "songCount";
    m_roleNames[ArtistRoles::AlbumIdsRole] = "albumIds";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(onArtworkReady()));
}

ArtistModel::~ArtistModel() {
    qDeleteAll(m_list);
    m_list.clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> ArtistModel::roleNames() const {
    return m_roleNames;
}
#endif

int ArtistModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant ArtistModel::data(const QModelIndex &index, int role) const {
    if (Artist *artist = this->get(index)) {
        switch (role) {
        case ArtistRoles::IdRole:
            return artist->id();
        case ArtistRoles::NameRole:
            return artist->name();
        case ArtistRoles::UrlRole:
            return artist->url();
        case ArtistRoles::ArtworkUrlRole:
            return artist->artworkUrl();
        case ArtistRoles::ArtworkRole:
            return m_cache->artwork(artist->artworkUrl(), THUMBNAIL_SIZE);
        case ArtistRoles::SongCountRole:
            return artist->songCount();
        case ArtistRoles::AlbumIdsRole:
            return artist->albumIds();
        default:
            break;
        }
    }

    return QVariant();
}

QVariant ArtistModel::data(int row, int role) const {
    return this->data(this->index(row), role);
}

bool ArtistModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    Q_UNUSED(index)
    Q_UNUSED(value)
    Q_UNUSED(role)

    return false;
}

bool ArtistModel::setData(int row, const QVariant &value, int role) {
    return this->setData(this->index(row), value, role);
}

QMap<int, QVariant> ArtistModel::itemData(const QModelIndex &index) const {
    QMap<int, QVariant> map;

    for (int role = ArtistRoles::IdRole; role <= ArtistRoles::AlbumIdsRole; role++) {
        map[role] = this->data(index, role);
    }

    return map;
}

QMap<int, QVariant> ArtistModel::itemData(int row) const {
    return this->itemData(this->index(row));
}

Artist* ArtistModel::get(const QModelIndex &index) const {
    if ((index.row() >= 0) && (index.row() < m_list.size())) {
        return m_list.at(index.row());
    }

    return 0;
}

Artist* ArtistModel::get(int row) const {
    return this->get(this->index(row));
}

bool ArtistModel::loading() const {
    return m_loading;
}

void ArtistModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QList<Artist*> ArtistModel::artists() const {
    return m_list;
}

void ArtistModel::appendArtist(Artist *artist) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size());
    m_list.append(artist);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void ArtistModel::appendArtists(QList<Artist *> artists) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size() + artists.size() - 1);
    m_list.append(artists);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void ArtistModel::insertArtist(int i, Artist *artist) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i);
        m_list.insert(i, artist);
        this->endInsertRows();
    }
    else {
        this->appendArtist(artist);
    }

    emit countChanged(this->rowCount());
}

void ArtistModel::insertArtists(int i, QList<Artist *> artists) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i + artists.size() - 1);

        foreach (Artist *artist, artists) {
            m_list.insert(i, artist);
            i++;
        }

        this->endInsertRows();
    }
    else {
        this->appendArtists(artists);
    }

    emit countChanged(this->rowCount());
}

void ArtistModel::removeArtist(int i) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i);
        m_list.takeAt(i)->deleteLater();
        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void ArtistModel::removeArtists(int i, int count) {
    if ((i >= 0) && ((i + count) <= m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i + count - 1);

        for (int ii = i; ii < (i + count); ii++) {
            m_list.takeAt(i)->deleteLater();
        }

        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void ArtistModel::reload() {
    this->clear();
    this->getArtists();
}

void ArtistModel::clear() {
    this->removeArtists(0, m_list.size());
}

void ArtistModel::getArtists() {
    this->setLoading(true);

    ArtistList *list = Music::getArtists();
    this->connect(list, SIGNAL(ready(ArtistList*)), this, SLOT(addArtists(ArtistList*)));
}

void ArtistModel::addArtists(ArtistList *list) {
    switch (list->error()) {
    case ReplyError::NoError:
        if (list->count() > 0) {
            this->appendArtists(list->artists());
        }

        break;
    default:
        Notifications::showError(list->errorString());
        break;
    }

    this->setLoading(false);

    list->deleteLater();
}

void ArtistModel::onArtworkReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
