#include "filemodel.h"
#include "json.h"
#include "baseurls.h"
#include "utils.h"

using namespace QtJson;

FileModel::FileModel(QObject *parent) :
    QAbstractItemModel(parent),
    m_rootItem(new FileItem),
    m_loading(false)
{
}

FileModel::~FileModel() {
    delete m_rootItem;
}

void FileModel::setRootNode(const QString &node) {
    m_rootItem->setResourcePath(node);
}

void FileModel::setLoading(bool loading) {
    m_loading = loading;
    emit loadingChanged(loading);
}

int FileModel::rowCount(const QModelIndex &parent) const {
    FileItem *parentItem;

    if (parent.column() > 0) {
        return 0;
    }
    if (!parent.isValid()) {
        parentItem = m_rootItem;
    }
    else {
        parentItem = get(parent);
    }

    return parentItem->childFiles().size();
}

int FileModel::columnCount(const QModelIndex &parent) const {
    Q_UNUSED(parent);

    return 1;
}

QVariant FileModel::data(const QModelIndex &index, int role) const {
    if (role == CheckedRole) {
        return QVariant(m_checkedIndexes.contains(index));
    }

    FileItem *item = get(index);

    if (!item) {
        return QVariant();
    }

    switch (role) {
    case FileNameRole:
        return item->fileName();
    case SizeRole:
        return item->size();
    case HashRole:
        return item->hash();
    case CreatedRole:
        return item->created();
    case ModifiedRole:
        return item->modified();
    case SuffixRole:
        return item->suffix();
    case MimeTypeRole:
        return item->mimeType();
    case FolderRole:
        return item->isFolder();
    case ChildrenRole:
        return item->hasChildren();
    case PathRole:
        return item->path();
    case ResourcePathRole:
        return item->resourcePath();
    case ContentPathRole:
        return item->contentPath();
    case VolumePathRole:
        return item->volumePath();
    case ParentPathRole:
        return item->parentPath();
    case KeyRole:
        return item->key();
    case UrlRole:
        return item->url();
    case LiveRole:
        return item->isLive();
    case PublicRole:
        return item->isPublic();
    default:
        return QVariant();
    }
}

bool FileModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    FileItem *file = get(index);

    switch (role) {
    case FileNameRole:
        file->setFileName(value.toString());
        break;
    case SizeRole:
        file->setSize(value.toLongLong());
        break;
    case HashRole:
        file->setHash(value.toString());
        break;
    case CreatedRole:
        file->setCreated(Utils::localDateTimeFromString(value.toString()));
        break;
    case ModifiedRole:
        file->setModified(Utils::localDateTimeFromString(value.toString()));
        break;
    case FolderRole:
        file->setIsFolder(value.toBool());
        break;
    case ChildrenRole:
        file->setHasChildren(value.toBool());
        break;
    case PathRole:
        file->setPath(value.toString());
        break;
    case ResourcePathRole:
        file->setResourcePath(value.toString());
        break;
    case ContentPathRole:
        file->setContentPath(value.toString());
        break;
    case VolumePathRole:
        file->setVolumePath(value.toString());
        break;
    case ParentPathRole:
        file->setParentPath(value.toString());
        break;
    case KeyRole:
        file->setKey(value.toString());
        break;
    case UrlRole:
        file->setUrl(value.toString());
        break;
    case LiveRole:
        file->setLive(value.toBool());
        break;
    case PublicRole:
        file->setPublic(value.toBool());
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);

    return true;
}

QModelIndex FileModel::index(int row, int column, const QModelIndex &parent) const {
    if ((parent.isValid()) && (parent.column() != 0)) {
        return QModelIndex();
    }

    FileItem *parentItem = get(parent);
    FileItem *childItem = parentItem->childFile(row);

    if (childItem) {
        return createIndex(row, column, childItem);
    }
    else {
        return QModelIndex();
    }
}

QModelIndex FileModel::parent(const QModelIndex &child) const {
    if (!child.isValid()) {
        return QModelIndex();
    }

    FileItem *childItem = get(child);
    FileItem *parentItem = childItem->parentFile();

    if (parentItem == m_rootItem) {
        return QModelIndex();
    }

    return createIndex(parentItem->childNumber(), 0, parentItem);
}

FileItem* FileModel::get(const QModelIndex &index) const {
    return index.isValid() ? static_cast<FileItem*>(index.internalPointer()) : m_rootItem;
}

void FileModel::toggleChecked(const QModelIndex &index) {
    if (m_checkedIndexes.contains(index)) {
        m_checkedIndexes.removeOne(index);
    }
    else {
        m_checkedIndexes.append(index);
    }

    emit indexesCheckedChanged(indexesChecked());
    emit dataChanged(index, index);
}

void FileModel::checkNone() {
    m_checkedIndexes.clear();
    emit indexesCheckedChanged(false);
    emit dataChanged(activeIndex(), index(rowCount(activeIndex()), 0, activeIndex()));
}

void FileModel::insertFile(int position, FileItem *file, const QModelIndex &parent) {
    beginInsertRows(parent, position, position);

    if (!parent.isValid()) {
        m_rootItem->insertChildFile(position, file);
    }
    else {
        FileItem *parentItem = get(parent);
        parentItem->insertChildFile(position, file);
    }

    endInsertRows();
}

void FileModel::appendFile(FileItem *file, const QModelIndex &parent) {
    beginInsertRows(parent, rowCount(parent), rowCount(parent));

    if (!parent.isValid()) {
        m_rootItem->appendChildFile(file);
    }
    else {
        FileItem *parentItem = get(parent);
        parentItem->appendChildFile(file);
    }

    endInsertRows();
}

void FileModel::removeFile(int position, const QModelIndex &parent) {
    beginRemoveRows(parent, position, position);
    delete get(parent)->takeChildFile(position);
    endRemoveRows();
}

void FileModel::removeFiles(int position, int count, const QModelIndex &parent) {
    beginRemoveRows(parent, position, position + count - 1);

    for (int i = 0; i < count; i++) {
        delete get(parent)->takeChildFile(position);
    }

    endRemoveRows();
}

void FileModel::getFiles(const QModelIndex &parent) {
    setActiveIndex(parent);
    setLoading(true);

    if (parent.isValid()) {
        emit getFiles(get(parent)->resourcePath());
    }
    else {
        emit getFiles(m_rootItem->resourcePath());
    }    
}

void FileModel::addFiles(const QVariantMap &files) {
    if (!activeIndex().isValid()) {
        m_rootItem->setResourcePath(files.value("resource_path").toString());
        m_rootItem->setContentPath(files.value("content_path").toString());
    }

    QVariantList fileList;
    int folderCount = 0;

    if (files.value("has_children").toBool()) {
        fileList = files.value("children").toList();
    }

    while (!fileList.isEmpty()) {
        QVariantMap item = fileList.takeFirst().toMap();
        FileItem *file = new FileItem(get(activeIndex()));
        file->setPath(item.value("path").toString());
        file->setFileName(file->path().section('/', -1));
        file->setSize(item.value("size").toLongLong());
        file->setIsFolder(item.value("kind").toString() == "directory");
        file->setHasChildren(item.value("has_children").toBool());
        file->setHash(item.value("hash").toString());
        file->setCreated(Utils::localDateTimeFromString(item.value("created").toString()));
        file->setModified(Utils::localDateTimeFromString(item.value("modified").toString()));
        file->setResourcePath(item.value("resource_path").toString());
        file->setContentPath(item.value("content_path").toString());
        file->setVolumePath(item.value("volume_path").toString());
        file->setParentPath(item.value("parent_path").toString());
        file->setKey(item.value("key").toString());
        file->setUrl(item.value("public_url").toString());
        file->setLive(item.value("is_live").toBool());
        file->setPublic(item.value("is_public").toBool());

        if (file->isFolder()) {
            insertFile(folderCount, file, activeIndex());
            folderCount++;
        }
        else {
            appendFile(file, activeIndex());
        }
    }

    setLoading(false);
}

void FileModel::createFolder(const QModelIndex &parent, const QString &name) {
    emit busy(tr("Creating new folder"));
    m_destinationIndex = parent;
    emit createFolder(data(parent, ResourcePathRole).toString() + '/' + name);
}

void FileModel::onFolderCreated(const QVariantMap &folder) {
    emit busyProgressChanged(1);
    emit alert(tr("New folder created"));

    FileItem *parentFolder = get(m_destinationIndex);
    FileItem *newFolder = new FileItem(folder, parentFolder);

    int i = 0;
    bool found = false;
    FileItem *file;

    while ((!found) && (i < parentFolder->childFiles().size())) {
        file = parentFolder->childFile(i);

        found = (file->isFolder()) && (file->fileName().compare(newFolder->fileName(), Qt::CaseInsensitive) > 0);

        if (!found) {
            i++;
        }
    }

    if ((found) || (parentFolder->childFiles().isEmpty())) {
        beginInsertRows(m_destinationIndex, i, i);
        parentFolder->insertChildFile(i, newFolder);
        endInsertRows();
    }    
}

void FileModel::onFileCreated(const QVariantMap &file) {
    FileItem *newFile;
    FileItem *parentFolder;
    QModelIndexList matches = match(index(0, 0, QModelIndex()), ResourcePathRole, file.value("parent_path").toString(), 1, Qt::MatchExactly | Qt::MatchRecursive);
    QModelIndex actionIndex;

    if (!matches.isEmpty()) {
        actionIndex = matches.takeFirst();
    }

    parentFolder = get(actionIndex);
    newFile = new FileItem(file, parentFolder);

    int i = 0;
    bool found = false;
    FileItem *otherFile;

    while ((!found) && (i < parentFolder->childFiles().size())) {
        otherFile = parentFolder->childFile(i);

        found = (!otherFile->isFolder()) && (otherFile->fileName().compare(newFile->fileName(), Qt::CaseInsensitive) > 0);

        if (!found) {
            i++;
        }
    }

    if ((found) || (parentFolder->childFiles().isEmpty())) {
        beginInsertRows(actionIndex, i, i);
        parentFolder->insertChildFile(i, newFile);
        endInsertRows();
    }
}

void FileModel::deleteItems(const QModelIndexList &indexes) {
    emit busy(tr("Deleting item(s)"), indexes.size());
    m_actionsProcessed = 0;
    m_actionIndexes = indexes;

    if (m_actionIndexes.isEmpty()) {
        return;
    }

    deleteItem(m_actionIndexes.first());
}

void FileModel::deleteItem(const QModelIndex &index) {
    emit deleteItem(index.data(ResourcePathRole).toString());
}

void FileModel::onItemDeleted() {
    if (m_actionIndexes.isEmpty()) {
        emit alert(tr("Item(s) deleted"));
        return;
    }

    QModelIndex actionIndex = m_actionIndexes.takeFirst();
    QModelIndex parent = actionIndex.parent();
    beginRemoveRows(parent, actionIndex.row(), actionIndex.row());
    delete get(parent)->takeChildFile(actionIndex.row());
    endRemoveRows();

    emit busyProgressChanged(m_actionsProcessed++);

    if (!m_actionIndexes.isEmpty()) {
        deleteItem(m_actionIndexes.first());
    }
    else {
        emit alert(tr("Item(s) deleted"));
    }
}

void FileModel::moveItems(const QModelIndexList &indexes, const QModelIndex &destination) {
    emit busy(tr("Moving item(s)"), indexes.size());
    m_actionsProcessed = 0;
    m_actionIndexes = indexes;

    if (m_actionIndexes.isEmpty()) {
        return;
    }

    moveItem(m_actionIndexes.first(), destination);
}

void FileModel::moveItem(const QModelIndex &index, const QModelIndex &destination) {
    if (m_actionIndexes.isEmpty()) {
        m_actionIndexes.append(index);
    }

    m_destinationIndex = destination;
    emit moveItem(index.data(ResourcePathRole).toString(), destination.data(PathRole).toString() + '/' + index.data(FileNameRole).toString());
}

void FileModel::onItemMoved(const QVariantMap &newData) {
    if (m_actionIndexes.isEmpty()) {
        emit alert(tr("Item(s) moved"));
        return;
    }

    QModelIndex actionIndex = m_actionIndexes.takeFirst();
    FileItem *file = get(actionIndex);
    file->setPath(newData.value("path").toString());
    file->setFileName(file->path().section('/', -1));
    file->setSize(newData.value("size").toLongLong());
    file->setIsFolder(newData.value("kind").toString() == "directory");
    file->setHasChildren(newData.value("has_children").toBool());
    file->setHash(newData.value("hash").toString());
    file->setCreated(Utils::localDateTimeFromString(newData.value("created").toString()));
    file->setModified(Utils::localDateTimeFromString(newData.value("modified").toString()));
    file->setResourcePath(newData.value("resource_path").toString());
    file->setContentPath(newData.value("content_path").toString());
    file->setVolumePath(newData.value("volume_path").toString());
    file->setParentPath(newData.value("parent_path").toString());
    file->setKey(newData.value("key").toString());
    file->setUrl(newData.value("public_url").toString());
    file->setLive(newData.value("is_live").toBool());
    file->setPublic(newData.value("is_public").toBool());


    FileItem *parentFolder = file->parentFile();
    int row = parentFolder->childFiles().indexOf(file);

    beginRemoveRows(actionIndex.parent(), row, row);
    parentFolder->removeChildFile(row);
    endRemoveRows();

    if (parentFolder->childFiles().isEmpty()) {
        parentFolder->setHasChildren(false);
    }

    parentFolder->setModified(QDateTime::currentDateTime().toString("dd/MM/yyyy | hh:mm"));

    emit dataChanged(actionIndex.parent(), actionIndex.parent());

    row = 0;
    bool found = false;
    FileItem *compareFile;
    parentFolder = get(m_destinationIndex);

    while ((!found) && (row < parentFolder->childFiles().size())) {
        compareFile = parentFolder->childFile(row);

        found = (compareFile->isFolder() == file->isFolder()) && (compareFile->fileName().compare(file->fileName(), Qt::CaseInsensitive) > 0);

        if (!found) {
            row++;
        }
    }

    if ((found) || (parentFolder->childFiles().isEmpty())) {
        beginInsertRows(m_destinationIndex, row, row);
        parentFolder->insertChildFile(row, file);
        endInsertRows();
    }

    parentFolder->setHasChildren(true);
    parentFolder->setModified(QDateTime::currentDateTime().toString("dd/MM/yyyy | hh:mm:ss"));

    emit dataChanged(actionIndex.parent(), actionIndex.parent());

    emit busyProgressChanged(m_actionsProcessed++);

    if (!m_actionIndexes.isEmpty()) {
        moveItem(m_actionIndexes.first(), m_destinationIndex);
    }
    else {
        emit alert(tr("Item(s) moved"));
    }
}

void FileModel::renameItem(const QModelIndex &index, const QString &name) {
    emit busy(tr("Renaming item"));
    m_actionIndexes.append(index);
    emit renameItem(index.data(ResourcePathRole).toString(), index.data(PathRole).toString().section('/', 0, -2) + '/' + name);
}

void FileModel::onItemRenamed(const QVariantMap &newData) {
    emit busyProgressChanged(1);
    emit (tr("Item renamed"));

    if (!m_actionIndexes.isEmpty()) {
        QModelIndex actionIndex = m_actionIndexes.takeFirst();
        FileItem *file = get(actionIndex);
        file->setPath(newData.value("path").toString());
        file->setFileName(file->path().section('/', -1));
        file->setSize(newData.value("size").toLongLong());
        file->setIsFolder(newData.value("kind").toString() == "directory");
        file->setHasChildren(newData.value("has_children").toBool());
        file->setHash(newData.value("hash").toString());
        file->setCreated(Utils::localDateTimeFromString(newData.value("created").toString()));
        file->setModified(Utils::localDateTimeFromString(newData.value("modified").toString()));
        file->setResourcePath(newData.value("resource_path").toString());
        file->setContentPath(newData.value("content_path").toString());
        file->setVolumePath(newData.value("volume_path").toString());
        file->setParentPath(newData.value("parent_path").toString());
        file->setKey(newData.value("key").toString());
        file->setUrl(newData.value("public_url").toString());
        file->setLive(newData.value("is_live").toBool());
        file->setPublic(newData.value("is_public").toBool());

        FileItem *parentFolder = file->parentFile();
        parentFolder->childFiles().removeOne(file);

        int i = 0;
        bool found = false;

        while ((!found) && (i < parentFolder->childFiles().size())) {
            found = (parentFolder->childFile(i)->fileName().compare(file->fileName(), Qt::CaseInsensitive) > 0);

            if (!found) {
                i++;
            }
        }

        if ((found) || (parentFolder->childFiles().isEmpty())) {
            parentFolder->insertChildFile(i, file);
        }

        emit dataChanged(actionIndex.parent(), actionIndex.parent());
    }
}

void FileModel::setItemPublished(const QModelIndex &index, bool publish) {
    emit busy(publish ? tr("Publishing item") : tr("Unpublishing item"));
    m_actionIndexes.append(index);
    emit setItemPublished(index.data(ResourcePathRole).toString(), publish);
}

void FileModel::onItemPublishedChanged(const QVariantMap &newData) {
    emit busyProgressChanged(1);

    if (!m_actionIndexes.isEmpty()) {
        QModelIndex actionIndex = m_actionIndexes.takeFirst();
        FileItem *file = get(actionIndex);
        file->setPath(newData.value("path").toString());
        file->setFileName(file->path().section('/', -1));
        file->setSize(newData.value("size").toLongLong());
        file->setIsFolder(newData.value("kind").toString() == "directory");
        file->setHasChildren(newData.value("has_children").toBool());
        file->setHash(newData.value("hash").toString());
        file->setCreated(Utils::localDateTimeFromString(newData.value("created").toString()));
        file->setModified(Utils::localDateTimeFromString(newData.value("modified").toString()));
        file->setResourcePath(newData.value("resource_path").toString());
        file->setContentPath(newData.value("content_path").toString());
        file->setVolumePath(newData.value("volume_path").toString());
        file->setParentPath(newData.value("parent_path").toString());
        file->setKey(newData.value("key").toString());
        file->setUrl(newData.value("public_url").toString());
        file->setLive(newData.value("is_live").toBool());
        file->setPublic(newData.value("is_public").toBool());

        emit dataChanged(actionIndex, actionIndex);
        emit alert(file->isPublic() ? tr("Item published") : tr("Item unpublished"));
    }
}

void FileModel::reload(const QModelIndex &parent) {
    removeFiles(0, rowCount(parent), parent);
    getFiles(parent);
}

void FileModel::clear() {
    removeFiles(0, rowCount(QModelIndex()), QModelIndex());
}

void FileModel::onFilesError() {
    setLoading(false);
}
