#include "filelistdelegate.h"
#include "filemodel.h"
#include "utils.h"
#include <QStaticText>

FileListDelegate::FileListDelegate(QObject *parent) :
    QStyledItemDelegate(parent)
{
}

void FileListDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const {
    painter->save();

    bool checked = index.data(FileModel::CheckedRole).toBool();

    if (((option.state) & (QStyle::State_Selected)) || (checked)) {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundPressed.png"));
    }
    else {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundNormal.png"));
    }

    QRect iconRect = option.rect;
    iconRect.setSize(QSize(48, 48));
    iconRect.moveTopLeft(QPoint(iconRect.left() + 5, iconRect.top() + 16));

    FileItem::MimeType mimeType = static_cast<FileItem::MimeType>(index.data(FileModel::MimeTypeRole).toInt());

    switch (mimeType) {
    case FileItem::Folder:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/general_folder.png"));
        break;
    case FileItem::Document:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/general_notes.png"));
        break;
    case FileItem::PDF:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/general_pdf.png"));
        break;
    case FileItem::Image:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/general_image.png"));
        break;
    case FileItem::Audio:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/general_audio_file.png"));
        break;
    case FileItem::Video:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/general_video_file.png"));
        break;
    case FileItem::Package:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/filemanager_debian_file.png"));
        break;
    case FileItem::Playlist:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/filemanager_playlist.png"));
        break;
    case FileItem::Flash:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/filemanager_flash_file.png"));
        break;
    default:
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/filemanager_unknown_file.png"));
        break;
    }

    if (checked) {
        iconRect.moveRight(option.rect.right() - 5);
        painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/widgets_tickmark_grid.png"));
    }

    QRect textRect = option.rect;
    textRect.setHeight(35);
    textRect.setTopLeft(QPoint(textRect.left() + 58, textRect.top() + 5));
    textRect.setRight(checked ? textRect.right() - 58 : textRect.right() - 5);

    QStaticText staticText(index.data(FileModel::FileNameRole).toString());
    QTextOption textOption(Qt::AlignTop);
    textOption.setWrapMode(QTextOption::NoWrap);
    staticText.setTextWidth(textRect.width());
    staticText.setTextOption(textOption);

    painter->drawStaticText(textRect.topLeft(), staticText);

    iconRect.setSize(QSize(32, 32));
    iconRect.moveBottomLeft(QPoint(textRect.left(), option.rect.bottom()));

    if (mimeType == FileItem::Folder) {
        textRect.moveBottom(option.rect.bottom() - 5);

    }
    else {
        bool isPublic = index.data(FileModel::PublicRole).toBool();

        if (isPublic) {
            painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/statusarea_sharing.png").scaled(32, 32, Qt::IgnoreAspectRatio, Qt::SmoothTransformation));
        }
        else {
            painter->drawImage(iconRect, QImage("/usr/share/icons/hicolor/48x48/hildon/statusarea_sharing_blocked.png").scaled(32, 32, Qt::IgnoreAspectRatio, Qt::SmoothTransformation));
        }

        textRect.moveBottomLeft(QPoint(iconRect.right() + 5, option.rect.bottom() - 5));
    }

    QFont font;
    font.setPixelSize(20);
    painter->setFont(font);
    painter->setPen(QColor("#4d4d4d"));

    QString fileSize = Utils::fileSizeFromBytes(index.data(FileModel::SizeRole).toLongLong());
    QString lastModified = index.data(FileModel::ModifiedRole).toString();

    painter->drawText(textRect, Qt::AlignBottom, mimeType == FileItem::Folder ? lastModified : QString("%1 - %2").arg(lastModified).arg(fileSize));
    painter->restore();
}

QSize FileListDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const {
    Q_UNUSED(index);

    return QSize(option.rect.width(), 80);
}
