//*****************************************************************************
//*
//*
//*     NameDb.cpp
//*
//*
//*****************************************************************************
//
//  Copyright  2003    Anton Zechner
//
//  AzSmb is distributed under the GNU GENERAL PUBLIC LICENSE (GPL)
//  Sourcecode which use AzSmb must be published. Commercial users
//  must published their code too, or make an licence agreement with me.
//
//
//  AzSmb wird unter GNU GENERAL PUBLIC LICENSE (GPL) vertreiben.
//  Sourcecode welcher AzSmb verwendet muss verffentlicht werden.
//  Kommerzielle Nutzer mssen ihren Code ebenfalls verffentlichen, oder
//  eine Nutzungsvereinbarung mit mir treffen.
//
//  az_software@inode.at
//
//
//  This file contains a Name Database for a NetBios Server
//
#include    <memory.h>
#include    <string.h>
#include    <stdio.h>

#include    "System/BitUtility.h"
#include    "NetBiosDebug.h"
#include    "NameDb.h"


#define     MEM_ENTRY_ADD   64
#define     MEM_DOMAIN_ADD  8
#define     MEM_HEAP_SIZE   0x4000
#define     DB_CMP_SIZE     (DB_NAME_SIZE+sizeof(unsigned))


        int             iDbError        =  0;

static  char            cDbFileName  [MAX_PATH]="";
static  char            cDbFileDomain[MAX_PATH]="";
static  int             bIsChanged      =  0;
static  int             bIsOpen         =  0;
static  int             bIsLock         =  0;
static  int             bIsFile         =  0;

static  SysSemaphore    hSemaphore      =  0;
static  SysFile         hDbDomain       =  0;
static  SysFile         hDbName         =  0;

static  unsigned        uTempName,uTempDomain;

static  NameDbEntry    *pEntries        =  0;       /* pointer to name entries */
static  unsigned       *pMaxEntries     =  0;       /* positon of the higest entry */
static  unsigned        uMemEntries     =  0;       /* entries in memory */
static  unsigned        uUseEntries     =  0;       /* entries used */

static  char           *pDomains        =  0;       /* pointer to domain entries */
static  unsigned       *pMaxDomains     =  0;       /* positon of the higest domain entry */
static  unsigned        uMemDomains     =  0;       /* domain entries in memory */


        char           *pDbLock         =  0;       /* lock bits for entries */
static  char           *pDbUsed         =  0;       /* used bits for entries */
static  int            *pDbDomainCount  =  0;       /* used count for domain entries */
static  int            *pSortPos        =  0;       /* sorted indexes */

static  SysHeap         hDbHeap         =  0;       /* handle do heap */


#define     DbMalloc(s) SysHeapAlloc(hDbHeap,s)
#define     DbFree(p)   SysHeapFree(hDbHeap,p)

//*****************************************************************************
//*
//*     DbCmp
//*
//*****************************************************************************
static int DbCmp(int iPos1,int iPos2,int iUseIp)
{
int     *pName1,*pName2;
int      i,j,k,count;


    pName1=(int*)(pEntries+iPos1);
    pName2=(int*)(pEntries+iPos2);

    count=DB_CMP_SIZE/sizeof(int);
    if(iUseIp)count+=sizeof(IpAddr)/sizeof(int);

    for(i=0;i<count;i++)
        {
        j=pName1[i];
        k=pName2[i];
        if(j==k)continue;
        return (j<k)? -1:1;
        }


return 0;
}

//*****************************************************************************
//*
//*     DbSort
//*
//*****************************************************************************
static void DbSort()
{
int     u,o,m,c;
int     u_mem[30],o_mem[30];
int     ebene,start,ende;
int     pos;


    start=0;
    ende=uUseEntries-1;

    if(start>=ende)return;

    ebene = 0;


    for(;;)
        {
        m = (start + ende)>>1;              /* calc middle */
        u =  start;
        o =  ende + 1;

        pos=pSortPos[m];
            pSortPos[m]=pSortPos[u];
            pSortPos[u]=pos;

        for(;;)
            {
            do  {
                u++;
                if(u>ende)break;
                c=DbCmp(u,start,1);
                }
            while(c<=0);

            do  {
                o--;
                if(o<=start)break;
                c=DbCmp(o,start,1);
                }
            while(c>=0);

            if (o < u)break;

            pos=pSortPos[u];
                pSortPos[u]=pSortPos[o];
                pSortPos[o]=pos;
            }

        pos=pSortPos[start];
            pSortPos[start]=pSortPos[o];
            pSortPos[o]=pos;

        if ( o - 1 - start >= ende - u )
            {
            if (start + 1 < o)
                {
                u_mem[ebene] = start;
                o_mem[ebene] = o - 1;
                ++ebene;
                }
            if (u < ende)
                {
                start = u;
                continue;
                }
            }
        else{
            if (u < ende)
                {
                u_mem[ebene] = u;
                o_mem[ebene] = ende;
                ++ebene;
                }
            if (start + 1 < o)
                {
                ende = o - 1;
                continue;
                }
            }
                                        /* down */
        --ebene;

        if (ebene >= 0)                 /* last */
            {
            start = u_mem[ebene];
            ende  = o_mem[ebene];
            continue;
            }

        return;
        }



}

//*****************************************************************************
//*
//*     DbExpandDomains
//*
//*****************************************************************************
static int  DbExpandDomains(void)
{
return FALSE;
}

//*****************************************************************************
//*
//*     DbExpandEntries
//*
//*****************************************************************************
static int  DbExpandEntries(void)
{
return FALSE;
}


//*****************************************************************************
//*
//*     DbIsOpen
//*
//*****************************************************************************
//  return TRUE is the databsae is open
int  DbIsOpen()
{
    return bIsOpen;
}

//*****************************************************************************
//*
//*     DbOpen
//*
//*****************************************************************************
//  Open the database
//  bFile   : is TRUE if the databaes files are used
//  Return TRUE if the database will be opened
int  DbOpen(const char *pDbName,const char *pDbDomain,int  bFile)
{
unsigned        uSize,i;
NameDbEntry    *pName;


    if(bIsOpen)
        {
        iDbError=DB_ERR_ISOPEN;
        return FALSE;
        }

    if(!pDbName  )pDbName="";
    if(!pDbDomain)pDbDomain="";
    if(!hDbHeap  )hDbHeap=SysHeapCreate(MEM_HEAP_SIZE);

    if(bFile)
        {
            hDbName=SysOpen(pDbName,SYS_READ|SYS_WRITE|SYS_CREATE);
        if(!hDbName)
            {
            iDbError=DB_ERR_OPEN;
            goto fail;
            }

            hDbDomain=SysOpen(pDbDomain,SYS_READ|SYS_WRITE|SYS_CREATE);
        if(!hDbDomain)
            {
            iDbError=DB_ERR_OPEN;
            goto fail;
            }

        bIsFile=1;
        }
    else{
        hDbDomain   = 0;
        hDbName     = 0;
        bIsFile     = 0;
        }

//******************** read name database *************************************

    if(bIsFile)
        {
            uSize=SysGetFileSize(hDbName);
        if( uSize==0)
            {
            if(SysWrite(hDbName,&uSize,sizeof(unsigned))<=0)
                {
                iDbError=DB_ERR_WRITE;
                goto fail;
                }
            uSize=sizeof(unsigned);
            uTempName=0;
            }
        else if(SysRead(hDbName,&uTempName,sizeof(unsigned))!=sizeof(unsigned))
            {
            iDbError=DB_ERR_READ;
            goto fail;
            }

           uSize-=sizeof(unsigned);
        if(uSize!=uTempName*sizeof(NameDbEntry))
            {
            iDbError=DB_ERR_FILE;
            goto fail;
            }

        pMaxEntries=&uTempName;
        uMemEntries=uTempName+MEM_ENTRY_ADD;
        pEntries=(NameDbEntry*)DbMalloc(uMemEntries*sizeof(NameDbEntry));

        if(!pEntries)
            {
            iDbError=DB_ERR_MEM;
            goto fail;
            }

        if(SysRead(hDbName,pEntries,uSize)!=(int)uSize)
            {
            iDbError=DB_ERR_READ;
            goto fail;
            }
        }
    else{
        uTempName=0;
        pMaxEntries=&uTempName;
        uMemEntries=MEM_ENTRY_ADD;
        pEntries=(NameDbEntry*)DbMalloc(uMemEntries*sizeof(NameDbEntry));
        }

//******************** read domain database ***********************************

    if(bIsFile)
        {
            uSize=SysGetFileSize(hDbDomain);
        if( uSize<=0)
            {
            if(SysWrite(hDbDomain,&uSize,sizeof(unsigned))<=0)
                {
                iDbError=DB_ERR_WRITE;
                goto fail;
                }
            uSize=sizeof(unsigned);
            uTempDomain=0;
            }
        else if(SysRead(hDbDomain,&uTempDomain,sizeof(unsigned))!=sizeof(unsigned))
            {
            iDbError=DB_ERR_READ;
            goto fail;
            }

           uSize-=sizeof(unsigned);
        if(uSize!=uTempDomain*DB_DOMAIN_SIZE)
            {
            iDbError=DB_ERR_FILE;
            goto fail;
            }

        pMaxDomains =&uTempDomain;
        uMemDomains= uTempDomain+MEM_DOMAIN_ADD;
        pDomains=(char*)DbMalloc(uMemDomains*DB_DOMAIN_SIZE);

        if(!pDomains)
            {
            iDbError=DB_ERR_MEM;
            goto fail;
            }

        if(SysRead(hDbDomain,pDomains,uSize)!=(int)uSize)
            {
            iDbError=DB_ERR_READ;
            goto fail;
            }
        }
    else{
        uTempDomain=0;
        pMaxDomains =&uTempDomain;
        uMemDomains= uTempDomain+MEM_DOMAIN_ADD;
        pDomains=(char*)DbMalloc(uMemDomains*DB_DOMAIN_SIZE);
        }


//******************** Init database ******************************************

    pDbUsed=0;
    i=(uMemEntries+7)>>3;
    pDbUsed=(char*)DbMalloc(i);

    if(!pDbUsed)
        {
        iDbError=DB_ERR_MEM;
        goto fail;
        }

    memset(pDbUsed,0,i);

    pSortPos=(int*)DbMalloc(uMemEntries*sizeof(pSortPos[0]));
    if(!pSortPos)
        {
        iDbError=DB_ERR_MEM;
        goto fail;
        }

    i=uMemDomains*sizeof(int);
    pDbDomainCount=(int*)DbMalloc(i);

    if(!pDbDomainCount)
        {
        iDbError=DB_ERR_MEM;
        goto fail;
        }

    memset(pDbDomainCount,0,i);

    pName=pEntries;
    uUseEntries=0;

    for(i=0;i<*pMaxEntries;i++,pName++)
        {
        if(!pName->cName[0])continue;

        pSortPos[uUseEntries]=i;
        uUseEntries++;

        bit_set(pDbUsed,i);
        pDbDomainCount[pName->uDomain]++;
        }


    strncpy(cDbFileName  ,pDbName  ,sizeof(cDbFileName  ));
    strncpy(cDbFileDomain,pDbDomain,sizeof(cDbFileDomain));

    DbSort();
    bIsOpen=1;
    bIsLock=0;
    bIsChanged=0;

    hSemaphore=SysSemaphoreCreate();


    return TRUE;


fail:

    if(hDbDomain     ){SysClose(hDbDomain);   hDbDomain     =0;}
    if(hDbName       ){SysClose(hDbName  );   hDbName       =0;}
    if(pEntries      ){DbFree(pEntries);      pEntries      =0;}
    if(pDomains      ){DbFree(pDomains);      pDomains      =0;}
    if(pSortPos      ){DbFree(pSortPos);      pSortPos      =0;}
    if(pDbUsed       ){DbFree(pDbUsed);       pDbUsed       =0;}
    if(pDbDomainCount){DbFree(pDbDomainCount);pDbDomainCount=0;}

    uMemEntries=0;
    uUseEntries=0;
    uMemDomains=0;


return FALSE;
}


//*****************************************************************************
//*
//*     DbClose
//*
//*****************************************************************************
//  Close the database
int  DbClose()
{

    if(!bIsOpen)
        {
        iDbError=DB_ERR_ISOPEN;
        return FALSE;
        }


    DbFlush();


    SysSemaphoreLock(hSemaphore);

    if(hDbDomain     ){SysClose(hDbDomain);   hDbDomain     =0;}
    if(hDbName       ){SysClose(hDbName  );   hDbName       =0;}
    if(pEntries      ){DbFree(pEntries);      pEntries      =0;}
    if(pDomains      ){DbFree(pDomains);      pDomains      =0;}
    if(pSortPos      ){DbFree(pSortPos);      pSortPos      =0;}
    if(pDbUsed       ){DbFree(pDbUsed);       pDbUsed       =0;}
    if(pDbDomainCount){DbFree(pDbDomainCount);pDbDomainCount=0;}

    uMemEntries=0;
    uUseEntries=0;
    uMemDomains=0;

    if(hDbHeap)
        {
        SysHeapDestroy(hDbHeap);
        hDbHeap=0;
        }

    SysSemaphoreUnlock(hSemaphore);

    bIsChanged=0;
    bIsOpen=0;



return TRUE;
}



//*****************************************************************************
//*
//*     DbFlush
//*
//*****************************************************************************
//  Saves the database
int  DbFlush()
{
unsigned    uSize;



    if(!bIsOpen)
        {
        iDbError=DB_ERR_ISOPEN;
        return FALSE;
        }

    if(!bIsFile)
        {
        iDbError=DB_ERR_ISFILE;
        return FALSE;
        }

    if(!bIsChanged)return TRUE;

//******************** write names ********************************************

    if(SysSeek(hDbName,0,SEEK_SET)<0)
        {
        iDbError=DB_ERR_WRITE;
        return FALSE;
        }

    if(SysWrite(hDbName,pMaxEntries,sizeof(unsigned))!=sizeof(unsigned))
        {
        iDbError=DB_ERR_WRITE;
        return FALSE;
        }

    uSize=*pMaxEntries*sizeof(NameDbEntry);

    if(SysWrite(hDbName,pEntries,uSize)!=(int)uSize)
        {
        iDbError=DB_ERR_WRITE;
        return FALSE;
        }

    SysFlush(hDbName);

//******************** write domains ******************************************

    if(SysSeek(hDbDomain,0,SEEK_SET)<0)
        {
        iDbError=DB_ERR_WRITE;
        return FALSE;
        }

    if(SysWrite(hDbDomain,pMaxDomains,sizeof(unsigned))!=sizeof(unsigned))
        {
        iDbError=DB_ERR_WRITE;
        return FALSE;
        }

    uSize=*pMaxEntries*DB_DOMAIN_SIZE;

    if(SysWrite(hDbDomain,pDomains,uSize)!=(int)uSize)
        {
        iDbError=DB_ERR_WRITE;
        return FALSE;
        }

    SysFlush(hDbDomain);



return TRUE;
}



//*****************************************************************************
//*
//*     DbFindDomain
//*
//*****************************************************************************
//  Find a domain name
static unsigned DbFindDomain(const char *pDomain)
{
int      i,iCount,iEmpty;
char    *pDom;



    if( pDomain==0)return 0xFFFF;
    if(*pDomain==0)return 0xFFFF;

    iCount=*pMaxDomains;
    iEmpty=iCount;
    pDom =pDomains;

    for(i=0;i<iCount;i++)                           // find domainname
        {
        if(!pDom[0] && iEmpty==iCount)iEmpty=i;
        if(!strcmp(pDom,pDomain))break;
        pDom+=DB_DOMAIN_SIZE;
        }

    if(i>=iCount)return iEmpty;


return i;
}

//*****************************************************************************
//*
//*     DbFindInsert
//*
//*****************************************************************************
//  Find the insert position
static int DbFindInsert(const NameDbEntry *pEntry)
{
int      iStart,iHalf,iLast,iCmp,iPos;
int      i,j,k,iCount;
int     *pPtr1,*pPtr2;


    if(uUseEntries==0)return 0;


    iStart  = 0;
    iLast   = uUseEntries-1;
    pPtr1   = (int*)pEntry;
    iCount  = (DB_CMP_SIZE+sizeof(IpAddr))/sizeof(int);

    for(;;)
        {
        iHalf = (iLast-iStart+1)>>1;
        iPos  = iStart+iHalf;
        pPtr2 = (int*)(pEntries+pSortPos[iPos]);

        for(i=0,iCmp=0;i<iCount;i++)                // compare entry
            {
            j=pPtr1[i];
            k=pPtr2[i];
            if(j==k)continue;
            iCmp=(j<k)? -1:1;
            break;
            }

        if(iHalf && iCmp)                           // retury
            {
            if(iCmp<0){iLast =iStart+iHalf-1;if(iStart>iLast)iLast =iStart;continue;}
                      iStart=iStart+iHalf+1;if(iStart>iLast)iStart=iLast ;continue;
            }
        if(!iHalf)                                  // end of search
            {
            return (iCmp>=0)? iStart+1:iStart;
            }

        iStart=iPos;                                // multible entries
        }


return -1;
}


//*****************************************************************************
//*
//*     DbFind
//*
//*****************************************************************************
//  Find the insert position
static int DbFind(const NameDbEntry *pEntry,const char *pDomain,int iUseIp)
{
int      start,half,last,cmp,pos;
int      dpos,dval,dset=0,group,ret=-1;
int      i,j,k,count;
int     *p1,*p2;



    if(uUseEntries==0)return -1;

    start=0;
    last=uUseEntries-1;
    p1=(int*)pEntry;
    count=DB_CMP_SIZE/sizeof(int);
    if(iUseIp)count+=sizeof(IpAddr)/sizeof(int);
    dpos =DB_NAME_SIZE/sizeof(int);
    group=pEntry->wGroup;
    dval =group&0xFFFF;

    for(;;)
        {
        half = (last-start+1)>>1;
        pos  =  start+half;
        p2   = (int*)(pEntries+pSortPos[pos]);

        for(i=0,cmp=0;i<count;i++)              // compare entry
            {
            j=p1[i];
            k=p2[i];

            if(i==dpos)                         // get domain pos
                {
                if(!dset)
                    {
                    dval=DbFindDomain(pDomain);
                    dset=1;
                    }
                j=dval;
                }

            if(j==k)continue;
            cmp=(j<k)? -1:1;
            break;
            }

        if(half && cmp)                         // retury
            {
            if(cmp<0){last =pos-1;if(start>last)last =start;continue;}
                      start=pos+1;if(start>last)start=last ;continue;
            }

        if(!cmp)                                // found
            {
            if(!group || !half)return pos;
            ret=pos;
            last=pos-1;                         // find first group entry
            continue;
            }

        return ret;                             // end of search
        }


return -1;
}

//*****************************************************************************
//*
//*     DbIndexEntry
//*
//*****************************************************************************
//  Returns the entry at the position uIndex
NameDbEntry *DbIndexEntry(unsigned  uIndex)
{

    if(uIndex>=uUseEntries)return 0;
    return pEntries+pSortPos[uIndex];
}


//*****************************************************************************
//*
//*     DbIndexMax
//*
//*****************************************************************************
unsigned DbIndexMax()
{
    return uUseEntries;
}


//*****************************************************************************
//*
//*     DbDomainName
//*
//*****************************************************************************
//  returns the domainname of an index value
const char *DbDomainName(unsigned  uIndex)
{
    if(uIndex>=*pMaxDomains)return "";
    return pDomains+uIndex*DB_DOMAIN_SIZE;
}

//*****************************************************************************
//*
//*     DbAddEntry
//*
//*****************************************************************************
//  Adds an entry into the database
//  returns TRUE if the entry was added
int  DbAddEntry(const NameDbEntry *pEntry,const char *pDomain)
{
int             iPos,iSort,iLen;
NameDbEntry    *pNewEntry;
unsigned        uDom;



    if(!bIsOpen)
        {
        iDbError=DB_ERR_INIT;
        return FALSE;
        }


    SysSemaphoreLock(hSemaphore);

                                                // check entry
       iSort=DbFind(pEntry,pDomain,pEntry->wGroup);
    if(iSort>=0)
        {
        SysSemaphoreUnlock(hSemaphore);
        iDbError=DB_ERR_EXIST;
        return FALSE;
        }

       iPos=bit_find_0_next(pDbUsed,uMemEntries,0); // find memory pos
    if(iPos>=(int)uMemEntries)
        {
        if(!DbExpandEntries())
            {
            SysSemaphoreUnlock(hSemaphore);
            iDbError=DB_ERR_MEM;
            return FALSE;
            }
        }

       uDom=DbFindDomain(pDomain);
    if(uDom!=0xFFFF)                            // add domain name
        {
        if(uDom>=uMemDomains)
            {
            if(!DbExpandDomains())
                {
                SysSemaphoreUnlock(hSemaphore);
                iDbError=DB_ERR_MEM;
                return FALSE;
                }
            }

        if(pDbDomainCount[uDom]==0)             // create domain name
            {
            iLen=strlen(pDomain);
            if(iLen>DB_DOMAIN_SIZE-1)
                {
                SysSemaphoreUnlock(hSemaphore);
                iDbError=DB_ERR_DOMAIN;
                return FALSE;
                }
            memcpy(pDomains+uDom*DB_DOMAIN_SIZE,pDomain,iLen);
            memset(pDomains+uDom*DB_DOMAIN_SIZE+iLen,0,DB_DOMAIN_SIZE-iLen);
            if(uDom>=*pMaxDomains)*pMaxDomains=uDom+1;
            }

        pDbDomainCount[uDom]++;
        }


    pNewEntry=pEntries+iPos;                    // copy entry
    memcpy(pNewEntry,pEntry,sizeof(NameDbEntry));
    pNewEntry->uDomain=uDom;
    bit_set(pDbUsed,iPos);
    if(iPos>=(int)*pMaxEntries)*pMaxEntries=iPos+1;

    iSort=DbFindInsert(pNewEntry);
    memmove(pSortPos+iSort+1,pSortPos+iSort,(uUseEntries-iSort)*sizeof(pSortPos[0]));
    pSortPos[iSort]=iPos;
    uUseEntries++;


    bIsChanged=1;
    SysSemaphoreUnlock(hSemaphore);


return TRUE;
}


//*****************************************************************************
//*
//*     DbDelEntry
//*
//*****************************************************************************
//  Adds an entry into the database
//  returns TRUE if the entry was deleted
int  DbDelEntry(const NameDbEntry *pEntry,const char *pDomain)
{
int             iPos,iSort;
NameDbEntry    *pDelEntry;
unsigned        uDom;


    if(!bIsOpen)
        {
        iDbError=DB_ERR_INIT;
        return FALSE;
        }



    SysSemaphoreLock(hSemaphore);

                                                // find entry
       iSort=DbFind(pEntry,pDomain,pEntry->wGroup);
    if(iSort<0)
        {
        SysSemaphoreUnlock(hSemaphore);
        iDbError=DB_ERR_NOTEXIST;
        return FALSE;
        }

    iPos=pSortPos[iSort];

    pDelEntry=pEntries+iPos;
    uDom=pDelEntry->uDomain;                   // delete domain name
    if(uDom!=0xFFFF)
        {
        pDbDomainCount[uDom]--;

        if(pDbDomainCount[uDom]==0)
            {
            memset(pDomains+uDom*DB_DOMAIN_SIZE,0,DB_DOMAIN_SIZE);
            }
        }

    memset(pDelEntry,0,sizeof(NameDbEntry));    // delete entry
    bit_clear(pDbUsed,iPos);
    memmove(pSortPos+iSort,pSortPos+iSort+1,(uUseEntries-iSort-1)*sizeof(pSortPos[0]));
    pSortPos[iSort]=iPos;
    uUseEntries--;


    bIsChanged=1;
    SysSemaphoreUnlock(hSemaphore);


return TRUE;
}


//*****************************************************************************
//*
//*     DbFindEntry
//*
//*****************************************************************************
//  search for an entry and lock it if an entry was found
//  do unlock the database call DbUnlock()
//  returns a pointer to the entry, or zero if no one was found
NameDbEntry *DbFindEntry(const NameDbEntry *pEntry,const char *pDomain,int  bUseIp)
{
int             iPos,iSort;
NameDbEntry    *pFindEntry;



    if(!bIsOpen)
        {
        iDbError=DB_ERR_INIT;
        return NULL;
        }

    SysSemaphoreLock(hSemaphore);

                                                // find entry
       iSort=DbFind(pEntry,pDomain,bUseIp);
    if(iSort<0)
        {
        SysSemaphoreUnlock(hSemaphore);
        iDbError=DB_ERR_NOTEXIST;
        return NULL;
        }

    iPos=pSortPos[iSort];

    pFindEntry=pEntries+iPos;                   // delete entry
    bIsLock=1;


return pFindEntry;
}



//*****************************************************************************
//*
//*     DbUnlock
//*
//*****************************************************************************
//  unlocks the database
void DbUnlock()
{
    if(!bIsLock)return;
    bIsLock=0;
    SysSemaphoreUnlock(hSemaphore);
}

//*****************************************************************************
//*
//*     DbLock
//*
//*****************************************************************************
//  locks the database
void DbLock()
{
    if(bIsLock)return;
    SysSemaphoreLock(hSemaphore);
    bIsLock=1;
}

