/*
  TimerJinni - A general timing application
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Countup.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/TimeSpanChooser.h>

#include <Lum/ButtonRow.h>
#include <Lum/Button.h>
#include <Lum/Panel.h>
#include <Lum/Table.h>
#include <Lum/Toggle.h>

#include <Lum/OS/Manager/Repository.h>

Countup::Countup()
 : startAction(new Lum::Model::Action()),
   paused(new Lum::Model::Boolean(false)),
   lapAction(new Lum::Model::Action()),
   stopAction(new Lum::Model::Action()),
   timer(new Lum::Model::Action()),
   seconds(new Lum::Model::SizeT(0)),
   laps(new Lum::Model::StringTable()),
   state(stateUninitialized),
   clock(new StopWatch())
{
  Observe(seconds);

  Observe(startAction);
  Observe(paused);
  Observe(lapAction);
  Observe(stopAction);
  Observe(timer);
}

Countup::~Countup()
{
  // no code
}

void Countup::SetValue()
{
  if (clock!=NULL) {
    clock->SetText(Lum::Base::TimeSpanToWString(seconds->Get()));
  }
}

void Countup::AllowScreenBlanking(bool allow)
{
  Lum::OS::Manager::DisplayManager *displayManager=Lum::OS::Manager::repository->GetDisplayManager();

  if (displayManager!=NULL) {
    displayManager->AllowScreenBlanking(allow);
  }
}

void Countup::SetState(State state)
{
  if (this->state==state) {
    return;
  }

  this->state=state;

  // Visual representation

  switch (state) {
  case stateUninitialized:
    assert(false);
    break;
  case stateRunning:
    startAction->Enable();
    paused->Enable();
    lapAction->Enable();
    stopAction->Enable();
    AllowScreenBlanking(false);
    break;
  case statePaused:
    startAction->Enable();
    paused->Enable();
    lapAction->Disable();
    stopAction->Enable();
    AllowScreenBlanking(false);
    break;
  case stateStoped:
    startAction->Enable();
    paused->Disable();
    lapAction->Disable();
    stopAction->Disable();
    AllowScreenBlanking(true);
    break;
  }

  // Timer

  switch (state) {
  case stateUninitialized:
    assert(false);
    break;
  case stateRunning:
    paused->Set(false);
    if (time.IsPausing()) {
      time.Resume();
    }
    else {
      laps->Clear();
      time.Start();
    }
    seconds->Set(time.GetTime());
    Lum::OS::display->AddTimer(0,100000,timer);
    break;
  case statePaused:
    seconds->Set(time.GetTime());
    time.Pause();
    Lum::OS::display->RemoveTimer(timer);
    break;
  case stateStoped:
    seconds->Set(time.GetTime());
    time.Reset();
    paused->Set(false);
    Lum::OS::display->RemoveTimer(timer);
    break;
  }
}

void Countup::CalcSize()
{
  Lum::Object           *countdown;
  Lum::Model::HeaderRef headerModel;

  clock->SetFlex(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Time",Lum::Base::Size::stdCharWidth,6);

  lapTable=new Lum::Table();
  lapTable->SetFlex(false,true);
  lapTable->SetModel(laps);
  lapTable->SetHeaderModel(headerModel);
  lapTable->SetShowHeader(false);
  lapTable->GetTableView()->SetAutoHSize(true);

  countdown=Lum::VPanel::Create(true,true)
            ->Add(Lum::HPanel::Create(true,true)
                  ->Add(lapTable)
                  ->AddSpace()
                  ->Add(clock))
            ->AddSpace()
            ->Add(Lum::ButtonRow::Create(true,false)
                  ->Add(Lum::Button::Create(_(L"BUTTON_START",L"_Start"),startAction,true,true))
                  ->Add(Lum::Toggle::Create(_(L"BUTTON_PAUSE",L"Pause"),paused,true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_LAP",L"_Lap"),lapAction,true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_STOP",L"Sto_p"),stopAction,true,true)));

  container=countdown;

  SetState(stateStoped);

  Component::CalcSize();
}

void Countup::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==startAction &&
      startAction->IsFinished()) {
    SetState(stateStoped);
    SetState(stateRunning);
  }
  else if (model==paused) {
    if (state==stateRunning &&
        paused->Get()) {
      SetState(statePaused);
    }
    else if (state==statePaused &&
             !paused->Get()) {
      SetState(stateRunning);
    }
  }
  else if (model==lapAction &&
           lapAction->IsFinished()) {
    Lum::Base::SystemTime now;

    laps->Append(Lum::Base::TimeSpanToWString(time.GetTime()));

    // The update to Adjustment is currently asynchronous in Table,
    // it is possible that it is not valid (or that ShowLastPage()
    // works on old values...
    if (lapTable->GetTableView()->GetVAdjustment()->IsValid()) {
      lapTable->GetTableView()->GetVAdjustment()->ShowLastPage();
    }
  }
  else if (model==stopAction &&
           stopAction->IsFinished()) {
    SetState(stateStoped);
  }
  else if (model==timer && timer->IsFinished()) {
    if (state==stateRunning) {
      seconds->Set(time.GetTime());
      Lum::OS::display->AddTimer(0,100000,timer);
    }
  }
  else if (model==seconds) {
    SetValue();
  }

  Lum::Component::Resync(model,msg);
}
