/*
** Copyright (c) 2009  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "mainwindow.h"
#include "numberitem.h"

QColor CMainWindow::c_Colors[Color_Last];
QPixmap* CMainWindow::c_Pixmaps[Pixmap_Last] = {0};

#ifdef Q_WS_HILDON
//-----------------------------------------------------------------------------
/**
** Callback for display state changes.
**
** \param state The state of the backlight.
** \param data Pointer to the CMainWindow.
*/
void onDisplayStateChanged(osso_display_state_t state, gpointer data)
{
    CMainWindow* pMainWindow = (CMainWindow*)data;
    if (pMainWindow)
    {
        pMainWindow->stopUpdates(state == OSSO_DISPLAY_OFF);
    }
}
#endif

Q_DECLARE_METATYPE(QList<int>);

//-----------------------------------------------------------------------------
/**
** Constructor
**
** \param pParent The parent for the main window (usually NULL)
*/
CMainWindow::CMainWindow(QWidget* pParent)
    : QMainWindow(pParent)
{
    m_pScene = NULL;
    m_pView = NULL;

    m_pTimeGroup = NULL;

    m_pButtonStart = NULL;
    m_pButtonReset = NULL;
    m_pHelpText = NULL;

    m_ShowClockAction = NULL;
    m_KeepBacklightAction = NULL;

    m_bCountDown = false;
    m_bShowTime = false;

    qRegisterMetaType< QList<int> >("QList<int>");

    QSettings settings;
    m_strAlarm = settings.value("Alarm", QApplication::applicationDirPath() + "/resources/buzz.wav").toString();

    if (!settings.contains("Presets"))
    {
        m_Presets << 5 * 60 << 30 * 60 << 60 * 60 << 2 * 60 * 60;
    }
    else
    {
        m_Presets = settings.value("Presets").value< QList<int> >();
    }

    bool bOk = false;
    bOk = connect(&m_Timer, SIGNAL(timeout()), this, SLOT(onTimeout()));
    Q_ASSERT(bOk);

#ifdef Q_WS_HILDON
    m_pOssoContext = osso_initialize("tickstill", "1.0", FALSE, NULL);
    if (m_pOssoContext == NULL)
    {
        qDebug() << "Failed to initialize LibOSSO";
    }
    else
    {
        osso_return_t result = osso_hw_set_display_event_cb(m_pOssoContext, onDisplayStateChanged, this);
    }
#endif
}

//-----------------------------------------------------------------------------
/**
** Destructor
*/
CMainWindow::~CMainWindow()
{
    delete m_pTimeGroup;

    if (m_pScene)
    {
        delete m_pScene;
        m_pScene = NULL;
    }

    if (m_pView)
    {
        delete m_pView;
        m_pView = NULL;
    }

    for (int i = 0; i < Pixmap_Last; i++)
    {
        delete c_Pixmaps[i];
        c_Pixmaps[i] = NULL;
    }

#ifdef Q_WS_HILDON
    if (m_pOssoContext != NULL)
    {
        osso_deinitialize(m_pOssoContext);
        m_pOssoContext = NULL;
    }
#endif
}

//-----------------------------------------------------------------------------
/**
** Initializes the graphics.
**
** \param bInvert If true the graphics and colors are inverted.
*/
void CMainWindow::initGraphics(bool bInvert)
{
    for (int i = 0; i < Pixmap_Last; i++)
    {
        delete c_Pixmaps[i];
        c_Pixmaps[i] = NULL;
    }

    c_Colors[Color_Window_Background] = QColor(225, 235, 221);
    c_Colors[Color_Help_Text] = QColor(125, 135, 121);

    QImage imgNumbers(":/numbers.png");

    if (bInvert)
    {
        QColor c =  c_Colors[Color_Window_Background];
        c_Colors[Color_Window_Background] = QColor(255 - c.red(), 255 - c.green(), 255 - c.blue(), c.alpha());
        c =  c_Colors[Color_Help_Text];
        c_Colors[Color_Help_Text] = QColor(255 - c.red(), 255 - c.green(), 255 - c.blue(), c.alpha());

        imgNumbers.invertPixels(QImage::InvertRgb);
    }

    // Normal numbers
    c_Pixmaps[Pixmap_Numbers] = new QPixmap(QPixmap::fromImage(imgNumbers));

    // Dimmed numbers
    QImage imgDimmedNumbers(imgNumbers.size(), imgNumbers.format());
    imgDimmedNumbers.fill(0);
    QPainter painter2(&imgDimmedNumbers);
    painter2.setOpacity(0.5);
    painter2.drawImage(QPoint(0, 0), imgNumbers);
    c_Pixmaps[Pixmap_Dimmed_Numbers] = new QPixmap(QPixmap::fromImage(imgDimmedNumbers));

    // Flipped numbers
    QImage imgFlippedNumbers(imgNumbers.width(), imgNumbers.height() / 2, imgNumbers.format());
    imgFlippedNumbers.fill(0);

    // Create the flipped numbers with alpha shading
    imgNumbers = imgNumbers.mirrored(false, true);      // Flip the original numbers
    QPainter painter(&imgFlippedNumbers);
    for (int i = 0; i < 11; i++)
    {
        int y = imgFlippedNumbers.height() - NUMBER_HEIGHT / 2 - i * NUMBER_HEIGHT / 2;
        painter.drawImage(0, y, imgNumbers, 0, NUMBER_HEIGHT * i, NUMBER_WIDTH, NUMBER_HEIGHT / 2);
    }

    QLinearGradient gradient(0, 0, 0, NUMBER_HEIGHT / 2);
    gradient.setColorAt(0, QColor(0, 0, 0, bInvert ? 128 : 96));
    gradient.setColorAt(1, QColor(0, 0, 0, 0));
    gradient.setSpread(QGradient::RepeatSpread);

    QBrush brushFade(gradient);
    painter.setCompositionMode(QPainter::CompositionMode_DestinationIn);
    painter.fillRect(QRect(0, 0, NUMBER_WIDTH, imgFlippedNumbers.height()), brushFade);

    c_Pixmaps[Pixmap_Flipped_Numbers] = new QPixmap(QPixmap::fromImage(imgFlippedNumbers));

    QImage imgLeft(":/button-left.png");
    QImage imgReset(":/button-reset.png");
    QImage imgRight(":/button-right.png");
    QImage imgStart(":/button-start.png");
    QImage imgPause(":/button-pause.png");

    if (bInvert)
    {
        imgLeft.invertPixels(QImage::InvertRgb);
        imgReset.invertPixels(QImage::InvertRgb);
        imgRight.invertPixels(QImage::InvertRgb);
        imgStart.invertPixels(QImage::InvertRgb);
        imgPause.invertPixels(QImage::InvertRgb);
    }

    m_pxPause = QPixmap::fromImage(imgPause);
    m_pxStart = QPixmap::fromImage(imgStart);

    m_pButtonReset->setGraphics(QPixmap::fromImage(imgLeft), QPixmap::fromImage(imgReset), QPointF(-10, 10));
    m_pButtonStart->setGraphics(QPixmap::fromImage(imgRight), m_pxStart, QPointF(10, 10));
    m_pHelpText->setColor(c_Colors[Color_Help_Text]);

    m_pScene->setBackgroundBrush(c_Colors[Color_Window_Background]);
}

//-----------------------------------------------------------------------------
/**
** Initializes the scene and adds all items to it.
*/
void CMainWindow::intialize()
{
    bool bOk = false;

    int w = width();
    int h = height();

    m_pScene = new QGraphicsScene(this);

    m_pTimeGroup = new CTimeGroup(this);
    m_pTimeGroup->initialize(m_pScene);

    bOk = connect(m_pTimeGroup, SIGNAL(presetChanged()), this, SLOT(onPresetsChanged()));
    Q_ASSERT(bOk);

    // Add the buttons to the scene too
    m_pButtonReset = new CCustomButton(this);
    m_pButtonStart = new CCustomButton(this);

    bOk = connect(m_pButtonReset, SIGNAL(clicked()), this, SLOT(onReset()));
    Q_ASSERT(bOk);
    bOk = connect(m_pButtonStart, SIGNAL(clicked()), this, SLOT(onStart()));
    Q_ASSERT(bOk);

    m_pScene->addItem(m_pButtonReset);
    m_pScene->addItem(m_pButtonStart);

    m_pHelpText = new CTextItem();
    m_pHelpText->setVisible(false);
    m_pHelpText->setText(tr("Tap and hold to add or remove presets."));
    m_pScene->addItem(m_pHelpText);

#ifdef WITH_DEBUGTEXT
    m_pScene->addItem(CDebugText::instance());
#endif

    initGraphics(false);

    updateValue(0);
    enableEditMode(true);

    // Create the layout for the window and add the view to it
    m_pView = new CView(this);

#ifdef WITH_OPENGL
    m_pView->setViewport(new QGLWidget(QGLFormat(QGL::SampleBuffers)));
    m_pView->setViewportUpdateMode(QGraphicsView::FullViewportUpdate);
#endif

    m_pView->setScene(m_pScene);
    m_pView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_pView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_pView->setFrameStyle(QFrame::NoFrame);
    setCentralWidget(m_pView);
    
    m_pView->setSceneRect(-w / 2, -h / 2, w, h);

    setWindowTitle("Tickstill");
    setObjectName("Tickstill");

    // Create menu items
    QAction* aboutAction = new QAction(tr("&About"), this);
    QAction* chooseAlarmAction = new QAction(tr("&Choose alarm..."), this);
    m_ShowClockAction = new QAction(tr("&Show clock"), this);
    m_KeepBacklightAction = new QAction(tr("&Keep backlight on"), this);
    m_KeepBacklightAction->setCheckable(true);
    m_NightModeAction = new QAction(tr("&Night mode"), this);
    m_NightModeAction->setCheckable(true);
    m_PresetsAction = new QAction(tr("&Presets"), this);

    // Add item into menu
#if defined(Q_WS_HILDON)
    menuBar()->addAction(aboutAction);
    menuBar()->addAction(chooseAlarmAction);
    menuBar()->addAction(m_ShowClockAction);
    menuBar()->addAction(m_KeepBacklightAction);
    menuBar()->addAction(m_PresetsAction);
    menuBar()->addAction(m_NightModeAction);
#else
    // else File menu
    QMenu* menu = new QMenu(tr("&File"), this);
    menu->addAction(aboutAction);
    menu->addAction(chooseAlarmAction);
    menu->addAction(m_ShowClockAction);
    menu->addAction(m_KeepBacklightAction);
    menu->addAction(m_NightModeAction);
    menu->addAction(m_PresetsAction);

    QAction* exitAction = new QAction(tr("E&xit"), this);
    menu->addAction(exitAction);
    bOk = connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));
    Q_ASSERT(bOk);

    menuBar()->addMenu(menu);
#endif

    bOk = connect(aboutAction, SIGNAL(triggered()), this, SLOT(onAbout()));
    Q_ASSERT(bOk);
    bOk = connect(chooseAlarmAction, SIGNAL(triggered()), this, SLOT(onChooseAlarm()));
    Q_ASSERT(bOk);
    bOk = connect(m_ShowClockAction, SIGNAL(triggered()), this, SLOT(onShowClock()));
    Q_ASSERT(bOk);
    bOk = connect(m_NightModeAction, SIGNAL(triggered()), this, SLOT(onNightMode()));
    Q_ASSERT(bOk);
    bOk = connect(m_PresetsAction, SIGNAL(triggered()), this, SLOT(onPresets()));
    Q_ASSERT(bOk);
}

//-----------------------------------------------------------------------------
/**
** Start/stops the UI updates. This is called when the screen blanks to preserve
** battery.
**
** \param bStop Set to true to stop UI updates.
*/
void CMainWindow::stopUpdates(bool bStop)
{
    if (m_pTimeGroup)
    {
        m_pTimeGroup->stopUpdates(bStop);
    }
}

//-----------------------------------------------------------------------------
/**
** Updates the hour, minute and second values to the rotators.
**
** \param value The new value for the time group.
*/
void CMainWindow::updateValue(int value)
{
    if (m_pTimeGroup)
    {
        m_pTimeGroup->setValue(value);
    }
}

//-----------------------------------------------------------------------------
/**
** Enables/disables the edit mode for the rotators.
**
** \param bEnable Set to true to go to enable mode. False to normal mode.
*/
void CMainWindow::enableEditMode(bool bEnable)
{
    if (m_pTimeGroup)
    {
        m_pTimeGroup->enableEditMode(bEnable);
        m_pTimeGroup->setReflection(!bEnable);
        m_pTimeGroup->stopUpdates(false);
    }
}

//-----------------------------------------------------------------------------
/**
** Turns on the device's backlight. Works only in the device (obviously)
*/
void CMainWindow::turnBacklightOn()
{
#ifdef Q_WS_HILDON
    if (m_pOssoContext)
    {
        osso_return_t result;
        result = osso_display_state_on(m_pOssoContext);
        if (result != OSSO_OK)
        {
            qDebug() << QString("osso_display_state_on failed.");
        }
    }
#endif
}

///////////////////////////////////////////////////////////////////////////////
/// OVERRIDES
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Handles the window resizing. Repositions all items in the scene.
**
** \param pEvent The resize event.
*/
void CMainWindow::resizeEvent(QResizeEvent* pEvent)
{
    int w = width();
    int h = height();

    m_pScene->setBackgroundBrush(CMainWindow::getColor(Color_Window_Background));

    m_pButtonReset->setPos(-w / 2, h / 2 - m_pButtonReset->boundingRect().size().height());
    m_pButtonStart->setPos(w / 2 - m_pButtonStart->boundingRect().size().width(), h / 2 - m_pButtonStart->boundingRect().size().height());
    m_pButtonReset->setZValue(1);
    m_pButtonStart->setZValue(1);

#ifdef WITH_DEBUGTEXT
    CDebugText::instance()->setPos(-w / 2, -h / 2);
    CDebugText::instance()->setZValue(1);
#endif

    QSizeF s = m_pHelpText->boundingRect().size();
    m_pHelpText->setPos(-s.width() / 2, h / 2 - s.height() - 20);

    m_pView->setSceneRect(-w / 2, -h / 2, w, h);

    QWidget::resizeEvent(pEvent);
}

///////////////////////////////////////////////////////////////////////////////
/// SLOTS
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the reset button is pressed. Also called when
** the countdown timer goes to zero. Resets all values.
*/
void CMainWindow::onReset()
{
    m_PresetsAction->setEnabled(true);
    m_pTimeGroup->setFixedValues(QList<int>(), -1);
    m_pTimeGroup->stopMovement();
    m_pHelpText->setVisible(false);

    m_Timer.stop();
    if (m_pTimeGroup)
    {
        m_pTimeGroup->stopUpdates(false);
        m_pTimeGroup->setValue(0);
    }
    m_pButtonStart->setIcon(m_pxStart);
    enableEditMode(true);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the start/pause button is clicked.
*/
void CMainWindow::onStart()
{
    bool bPreset = m_pHelpText->isVisible();

    if (!m_bShowTime)
    {
        m_PresetsAction->setEnabled(true);
    }
    m_pTimeGroup->stopMovement();
    m_pTimeGroup->setFixedValues(QList<int>(), -1);
    m_pHelpText->setVisible(false);

    bool bFirstStart = m_pTimeGroup->isEditMode() || bPreset;     // If we're in the edit mode this is Start and not resume

    enableEditMode(false);

    if (m_pButtonStart)
    {
        if (m_Timer.isActive())
        {
            // Timer is running -> Pause
            m_pButtonStart->setIcon(m_pxStart);
            m_Timer.stop();
        }
        else
        {
            // Timer is not running -> Resume
            if (bFirstStart)
            {
                m_bCountDown = (m_pTimeGroup->value() != 0);       // Count direction depends on the start value (0:00 counts up)
            }
            m_pButtonStart->setIcon(m_pxPause);
            m_Timer.start(1000);
        }
    }
}

//-----------------------------------------------------------------------------
/**
** A slot which is called once per second to update the timer. Also checks
** the countdown end and lits the backlight on certain intervals.
*/
void CMainWindow::onTimeout()
{
    int currentValue = m_pTimeGroup->value();

    // Prevent light from dimming on every 30 sec if the user so chooses
    if (m_KeepBacklightAction && m_KeepBacklightAction->isChecked() && ((currentValue % 30) == 0))
    {
        turnBacklightOn();
    }

    if (m_bShowTime)
    {
        QTime t = QTime::currentTime();
        currentValue = t.second() + t.minute() * 60 + t.hour() * 60 * 60;
    }
    else
    {
        DEBUGTEXT("Value", QString("%1").arg(currentValue));

        if (m_bCountDown)
        {
            currentValue--;
        }
        else
        {
            currentValue++;
        }

        // Turn on the backlight on even hours, 30 and 10 minutes to the countdown.
        if (m_bCountDown)
        {
            if (currentValue % (60 * 60) == 0 || currentValue == 30 * 60 || currentValue == 10 * 60)
            {
                turnBacklightOn();
            }
        }

        // Play the alarm at the end
        if (currentValue <= 0)
        {
            if (m_strAlarm.endsWith(".wav", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".mp3", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".aac", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".flag", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".mp4", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".mp4a", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".ogg", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".wma", Qt::CaseInsensitive) ||
                m_strAlarm.endsWith(".au", Qt::CaseInsensitive))
            {
                // It's a known audio file so try to play it
#ifdef WITH_PHONON
                Phonon::MediaObject* music = Phonon::createPlayer(Phonon::MusicCategory, Phonon::MediaSource(m_strAlarm));
                bool bOk = connect(music, SIGNAL(finished()), music, SLOT(deleteLater()));
                Q_ASSERT(bOk);

                music->play();
                if (!music->isValid())
                {
                    qDebug() << music->errorString();
                }
#else
                QSound* music = new QSound(m_strAlarm, this);
                music->play();
#endif
                bOk = connect(m_pButtonStart, SIGNAL(clicked()), music, SLOT(stop()));
                Q_ASSERT(bOk);
                bOk = connect(m_pButtonReset, SIGNAL(clicked()), music, SLOT(stop()));
                Q_ASSERT(bOk);
            }
            else
            {
                // It's not a known audio file so try to run it
                QProcess::startDetached(m_strAlarm);
            }
            onReset();
        }
    }
    updateValue(currentValue);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Opens the about dialog.
*/
void CMainWindow::onAbout()
{
    QString strText = tr("Tickstill by Kimmo Pekkola");
    strText += "\n";
    strText += tr("Double tap to toggle full screen");
    strText += "\n";
    strText += tr("Buzz audio sound by Mike Koenig (soundbible.com)");
    strText += "\n\n";
    strText += tr("Current alarm: ") + m_strAlarm;
    QMessageBox::information(this, "Tickstill", strText);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Opens a file dialog
** for the user to choose the alarm sound/file.
*/
void CMainWindow::onChooseAlarm()
{
    QString strFileName = QFileDialog::getOpenFileName(this, tr("Choose audio file or executable"));
    if (!strFileName.isEmpty())
    {
        m_strAlarm = strFileName;

        QSettings settings;
        settings.setValue("Alarm", m_strAlarm);
   }
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Toggles the current time
** and timer.
*/
void CMainWindow::onShowClock()
{
    if (m_bShowTime)
    {
        m_PresetsAction->setEnabled(true);

        m_ShowClockAction->setText(tr("&Show clock"));
        m_pButtonStart->show();
        m_pButtonReset->show();
        m_bShowTime = false;
        onReset();
    }
    else
    {
        m_PresetsAction->setDisabled(true);

        m_ShowClockAction->setText(tr("&Show timer"));
        m_pButtonStart->hide();
        m_pButtonReset->hide();
        m_bShowTime = true;
        m_Timer.stop();
        onStart();
        onTimeout();
    }
}


//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Toggles the night mode.
*/
void CMainWindow::onNightMode()
{
    initGraphics(m_NightModeAction->isChecked());
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Shows the presets.
*/
void CMainWindow::onPresets()
{
    enableEditMode(false);
    m_pTimeGroup->setReflection(false);

    m_pButtonStart->setIcon(m_pxStart);
    m_Timer.stop();

    m_PresetsAction->setDisabled(true);

    qSort(m_Presets.begin(), m_Presets.end());

    m_pTimeGroup->setFixedValues(m_Presets, m_pTimeGroup->value());
    m_pHelpText->setVisible(true);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when status of a preset has changed. Writes the presets
** to settings.
*/
void CMainWindow::onPresetsChanged()
{
    m_Presets = m_pTimeGroup->fixedValues();

    QSettings settings;
    settings.setValue("Presets", QVariant::fromValue(m_Presets));
}

// EOF
