// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//          Music playback thread class

//#include "stdafx.hpp"
//#include "MainThread.hpp"
//#include "Preferences.hpp"

#include <ting/Thread.hpp>
#include <ting/debug.hpp>
#include <ting/Ref.hpp>
#include <ting/Timer.hpp>
#include <ting/utils.hpp>



#include "PlaybackThread.hpp"


using namespace ting;



PlaybackThread::PlaybackThread() :
		playerHasSwitched(true)
{}



//retuns true if data is not corrupted
//static
bool PlaybackThread::ParseMusicData(Array<MusicAction>& ma, const byte* dataStart, const byte* const dataEnd){
	ASSERT(ma.Size() > 0)
	ASSERT(dataStart < dataEnd)

	MusicAction* curMa = ma.Begin();
	for(; dataStart < dataEnd && curMa < ma.End(); ++curMa){
		if(dataEnd - dataStart < 1 + 4){
			//error, data corrupted
			break;
		}

		curMa->id = *dataStart;
		++dataStart;
		curMa->timestamp = ting::FromNetworkFormat32(dataStart);
		dataStart += 4;

		if(curMa->id == NOTE_ON){ //note on action ID
			if(dataEnd - dataStart < 1 + 4){
				//error, data corrupted
				break;
			}

			curMa->volume = *dataStart;
			++dataStart;
			u32 tmp = ting::FromNetworkFormat32(dataStart);
			curMa->freq = *reinterpret_cast<float*>(&tmp);
			dataStart += 4;
		}
	}

	if(curMa != ma.End() || dataStart != dataEnd)
		return false;

	return true;
}



void PlaybackThread::Run(){
	ting::byte curInstrumentCode;
//	Array<MusicAction> curMusicData;
	
	MusicAction *curMusicAction;

	u32 timestampBias;
	u32 localTimeOfStreamStart;

	while(!this->quitFlag){
		while(Ptr<Message> m = this->queue.PeekMsg()){
			m->Handle();
		}

		//curMusicData.Size() = 0 indicates that it is time to start new music data chunk
		if(this->curMusicData.Size() == 0){
			//if queue is empty then hang on GetMsg() waiting for some event, then check again
			if(this->musicDataQueue.size() == 0){
				//turn off playing note if player has switched
				if(this->playerHasSwitched){
					if(this->ch.IsValid())
						this->ch->Stop();
				}

				this->queue.GetMsg()->Handle();
				continue;
			}

			byte *startOfMusicData;
			byte *endOfMusicData;
			ASSERT(this->musicDataQueue.size() > 0)
			ASSERT(this->musicDataQueue.front().SizeInBytes() >= 4)
			startOfMusicData = this->musicDataQueue.front().Begin();
			endOfMusicData = this->musicDataQueue.front().End();
			
			ASSERT(startOfMusicData[0] == 0x03)

			//init music instrument if necessary
			if(this->instr.IsNotValid() || curInstrumentCode != startOfMusicData[1]){
				curInstrumentCode = startOfMusicData[1];
				switch(curInstrumentCode){
					default:
					case 0:
						this->instr = aumiks::SineWaveInstrument::New();
						break;
					case 1:
						this->instr = aumiks::PluckedStringInstrument::New();
						break;
					case 2:
						this->instr = aumiks::TromboneInstrument::New();
						break;
					case 3:
						this->instr = aumiks::ClarinetInstrument::New();
						break;
					case 4:
						this->instr = aumiks::ViolinInstrument::New();
						break;
					case 5:
						this->instr = aumiks::PianoInstrument::New();
						break;
					case 6:
						this->instr = aumiks::SawtoothInstrument::New();
						break;
					case 7:
						this->instr = aumiks::ThereminInstrument::New();
						break;
				}
			}

			u32 numActions = ting::FromNetworkFormat16(&startOfMusicData[2]);
//			TRACE(<< "PlaybackThread::Run(): numActions = " << numActions << std::endl)

			ASSERT(numActions > 0)
			this->curMusicData.Init(numActions);
			if(!PlaybackThread::ParseMusicData(
					this->curMusicData,
					&startOfMusicData[4],
					endOfMusicData
				))
			{
				this->curMusicData.Reset();
				this->musicDataQueue.pop_front();//pop music data
				ASSERT(false)
			}

			ASSERT(this->curMusicData.Size() > 0)

			curMusicAction = this->curMusicData.Begin();

			//init timestamp bias taking timestamp of first action in chunk
			this->musicDataQueue.pop_front();//pop music data

			//if player has switched, init timestamp bias
			if(this->playerHasSwitched){
				timestampBias = curMusicAction->timestamp;
				localTimeOfStreamStart = ting::GetTicks();
				this->playerHasSwitched = false;
//				TRACE(<< "PlaybackThread::Run(): timestamp bias inited = " << timestampBias << std::endl)
			}
		}//~if(curMusicData.Size() == 0)


		//check if music action triggers
		u32 ticks = ting::GetTicks();
		if(ticks - localTimeOfStreamStart >= curMusicAction->timestamp - timestampBias){
			ASSERT(curMusicAction < this->curMusicData.End())
			switch(curMusicAction->id){
				case NOTE_ON: //note on
					if(this->ch.IsNotValid()){
						this->ch = this->instr->CreateChannel();
					}
					this->ch->SetFrequency(curMusicAction->freq);
					this->ch->SetVolume(curMusicAction->volume);
					this->ch->Play();
					break;
				case NOTE_OFF:
					if(this->ch.IsValid()){
						this->ch->Stop();
					}
					this->ch.Reset();
					break;
				default:
					ASSERT(false)
					break;
			}

			//move to next music action
			++curMusicAction;
			if(curMusicAction >= this->curMusicData.End()){
				this->curMusicData.Reset();
			}
		}else{
			//did not trigger, wait
			u32 ms = curMusicAction->timestamp - timestampBias -
					(ticks - localTimeOfStreamStart);
			if(ms < 5){
				Thread::Sleep(0);
			}else if(ms > 500){
				Thread::Sleep(500);
			}else{
				Thread::Sleep(ms - 5);
			}
		}
	}//~while
	TRACE(<< "PlaybackThread::Run(): exiting" << std::endl)
}



void PlaybackThread::HandlePlayerSwitch_ts(){
	this->PushMessage(Ptr<Message>(
			new PlayerSwitchMessage(this)
		));
}



void PlaybackThread::PushNewPlaybackData_ts(ting::Array<ting::byte> data){
	if(data.SizeInBytes() < 4){
		//ignore chunks without any music actions.
		//Normally, there should be no such chunks at all.
		ASSERT(false)
		return;
	}

	this->PushMessage(Ptr<Message>(
			new PushMusicDataMessage(this, data)
		));
}



void PushMusicDataMessage::Handle(){
	this->t->musicDataQueue.push_back(this->data);
}



void PlaybackThread::ResetPlayback_ts(){
	this->PushMessage(Ptr<Message>(
			new ResetPlaybackMessage(this)
		));
}



//override
void ResetPlaybackMessage::Handle(){
	this->t->curMusicData.Reset();
	this->t->musicDataQueue.clear();

	if(this->t->ch.IsValid())
		this->t->ch->Stop();
}
