/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
 * aumiks 0.1
 */

#include <cmath>

#include <ting/math.hpp>
#include <ting/types.hpp>
#include <ting/Buffer.hpp>

#include "PluckedString.hpp"
#include "../aumiks.hpp"
#include "synth.hpp"
#include "FilterOneZero.hpp"



using namespace ting;
using namespace aumiks;



bool SineWaveInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool stopPlaying = false;
	bool quit;
	ting::uint i = 0;//current sample
	do{//while(!quit)
		quit = true;
		switch(this->envPhase){
			case ATTACK:
				for(; i < mixBuf.Size();){
					real attack = this->curTime / this->attackTime;
					this->releaseFromLevel = attack;
					s32 smp = s32(real(0x7fff) * attack * ting::Sin(this->curPhase));
					this->curPhase += this->phaseStep;
					this->curTime += real(1000.0f / 44100.0f);

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					if(this->curTime >= this->attackTime){
						this->curTime -= this->attackTime;
						quit = false;
						this->envPhase = DECAY;
//						TRACE(<< "switch to DECAY" << std::endl)
						break;//~for
					}
				}
				break;
			case DECAY:
				for(; i < mixBuf.Size();){
					real decay = real(1) - (real(1) - this->sustainLevel) * (this->curTime / this->decayTime);
					this->releaseFromLevel = decay;
					s32 smp = s32(real(0x7fff) * decay * ting::Sin(this->curPhase));
					this->curPhase += this->phaseStep;
					this->curTime += real(1000.0f / 44100.0f);

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					if(this->curTime >= this->decayTime){
//						this->curTime = 0;
						quit = false;
						this->envPhase = SUSTAIN;
//						TRACE(<< "switch to SUSTAIN" << std::endl)
						break;//~for
					}
				}
				break;
			case SUSTAIN:
				if(this->stopFlag){
					this->stopFlag = false;
					this->curTime = 0;
					this->envPhase = RELEASE;
					quit = false;
			//		TRACE(<< "switch to RELEASE" << std::endl)
				}else{
					this->releaseFromLevel = this->sustainLevel;
					for(; i < mixBuf.Size();){
						s32 smp = s32(real(0x7fff) * this->sustainLevel * ting::Sin(this->curPhase));
						this->curPhase += this->phaseStep;
	//					this->curTime += 1.0f / float(44100);
				//		TRACE(<<"t =" << this->t)
				//		TRACE(<<"smp =" << smp)

						//interpolate volume
						ting::u8 vol = ting::u8(startVolume + (endVolume - startVolume) * i / mixBuf.Size());

						//apply volume
						smp = (smp * vol) / 0xff;

						mixBuf[i] += smp;
						++i;
						mixBuf[i] += smp;
						++i;
					}//~for
				}

				{
					real numPeriods = this->curPhase / D2Pi<real>();
					this->curPhase -= real(int(numPeriods)) * D2Pi<real>();
				}

				break;//~switch
			case RELEASE:
				for(; i < mixBuf.Size();){
					real release = this->releaseFromLevel * (real(1) - this->curTime / this->releaseTime);
					s32 smp = s32(real(0x7fff) * release * ting::Sin(this->curPhase));
					this->curPhase += this->phaseStep;
					this->curTime += real(1000.0f / 44100.0f);

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					if(this->curTime >= this->releaseTime){
						this->curTime = 0;
						stopPlaying = true;
						this->envPhase = ATTACK;
//						TRACE(<< "switch to STOP playing" << std::endl)
						break;//~for
					}
				}
				break;
			default:
				ASSERT(false)
				break;
		}//~switch
	}while(!quit);
	
	return stopPlaying;
}



Ref<Instrument::Channel> SineWaveInstrument::CreateChannel(){
	Ref<SineWaveInstrument::Channel> ch(new SineWaveInstrument::Channel());
	return ch;
}



void SineWaveInstrument::Channel::SetFrequency(float freq){
	this->phaseStep = D2Pi<real>() * real(freq) / real(44100);
	this->attackTime = 60;//1000.0f * 30.0f / freq;
	this->decayTime = 50;//1000.0f * 50.0f / freq;
	this->sustainLevel = 0.8f;
	this->releaseTime = 50;//1000.0f * 30.0f / freq;
}



//override
bool PluckedStringInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool ret = false;

	switch(this->state){
		case PLAYING:
			if(this->stopFlag){
				this->state = RELEASING;
				this->curTime = 0.2;
				this->pluckedString.Mute();
			}
			break;
		case RELEASING:
			this->curTime -= float(mixBuf.Size()) / (2 * 44100.0f);
			if(curTime <= 0){
				this->state = STOPPED;
				ret = true;
			}
			break;
		default:
			break;
	}

	for(ting::uint i=0; i < mixBuf.Size();){
		float s = 2 * this->pluckedString.Tick();
		s32 smp = s32(0x7fff * s);

		//with distortion
//		s32 smp = s32( 0x7fff * (s > 0 ? tanh(4 * s) : s ) );

		//interpolate volume
		ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

		//apply volume
		smp = (smp * vol) / 0xff;


		mixBuf[i] += smp;
		++i;
		mixBuf[i] += smp;
		++i;
	}//~for

	return ret;
}



PianoInstrument::PianoInstrument(){
	float xx[] = {
		0.3728299,  - 0.2122525,  - 0.0674189,  - 0.2268258,  - 0.0089568,     0.0220279,    0.1111207,  - 0.0125976,  - 0.0085174,    0.0192382,   0.0352035,  - 0.0163387,    0.0064105,    0.0052186,    0.0045427
	};

	float yy[] = {
		0.0076323,    0.0842224,    0.2661944,    0.1654580,  - 0.0401976,   - 0.0395773,  - 0.0426222,  - 0.0056013,  - 0.0208229,    0.0668553, - 0.0034432,    0.0243109,    0.0041368,    0.0196354,    0.0073706
	};

	ting::StaticBuffer<float, sizeof(xx)/sizeof(xx[0])> x, y;

	memcpy(x.Buf(), xx, x.SizeInBytes());
	memcpy(y.Buf(), yy, y.SizeInBytes());

	for(ting::uint i = 0; i < y.Size(); ++i){
		x[i] /= 2;
		y[i] /= 2;
	}

	this->pianoPulseSample = Sample::GenerateFromFourier(
			x,
			y,
			ting::uint((44100.0f / 25.0f) * 1.0f),
			25
		);
}



//override
bool PianoInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool ret = false;

	switch(this->state){
		case PLAYING:
			if(this->stopFlag){
				this->state = RELEASING;
				this->curTime = 0.2;
				this->pluckedString.Mute();
			}
			break;
		case RELEASING:
			this->curTime -= float(mixBuf.Size()) / (2 * 44100.0f);
			if(curTime <= 0){
				this->state = STOPPED;
				ret = true;
			}
			break;
		default:
			break;
	}

	for(ting::uint i=0; i < mixBuf.Size();){
		float s = this->pluckedString.Tick(this->pulseGen.Tick());
		s32 smp = s32(0x7fff * s);

		//interpolate volume
		ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

		//apply volume
		smp = (smp * vol) / 0xff;


		mixBuf[i] += smp;
		++i;
		mixBuf[i] += smp;
		++i;
	}//~for

	return ret;
}



//override
bool BowedInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	bool ret = false;

	if(this->stopFlag){
		ret = true;
	}

	for(ting::uint i=0; i < mixBuf.Size();){
		float s = this->bowed.Tick();
		s32 smp = s32(0x7fff * s);

		//interpolate volume
		ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

		//apply volume
		smp = (smp * vol) / 0xff;


		mixBuf[i] += smp;
		++i;
		mixBuf[i] += smp;
		++i;
	}//~for

	return ret;
}



//override
bool SampledInstrument::Channel::MixToMixBuf(Array<s32>& mixBuf){
	if(this->lastVolume < 0){
		this->lastVolume = this->volume;
	}
	int startVolume = this->lastVolume;
	int endVolume = this->volume;
	this->lastVolume = this->volume;

	ASSERT(this->instr)
	const Array<real> &sample = this->instr->sample->data;
	ASSERT(sample.Size() > 0)

	real sustainLevel = this->instr->adsrParams.sustainLevel;

	bool stopPlaying = false;
	bool quit;
	ting::uint i = 0;//current sample index
	
	do{//while(!quit)
		quit = true;
		
		if(this->stopFlag && this->envPhase != RELEASE){
			//NOTE: stopFlag can be rised more than once, so, do the actions only if phase is not set to RELEASE yet.
			this->curTime = 0;
			this->envPhase = RELEASE;
		}

		switch(this->envPhase){
			case ATTACK:
				for(; i < mixBuf.Size();){
					real attack = this->curTime / this->attackTime;
					this->releaseFromLevel = attack;

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * attack * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					this->curTime += real(1000.0f / 44100.0f);
					if(this->curTime >= this->attackTime){
						this->curTime -= this->attackTime;
						quit = false;
						this->envPhase = DECAY;
//						TRACE(<< "switch to DECAY" << std::endl)
						break;//~for
					}
				}
				break;
				
			case DECAY:
				for(; i < mixBuf.Size();){
					real decay = real(1) - (real(1) - sustainLevel) * (this->curTime / this->decayTime);
					this->releaseFromLevel = decay;

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * decay * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}
					
					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					this->curTime += real(1000.0f / 44100.0f);
					if(this->curTime >= this->decayTime){
//						this->curTime = 0;
						quit = false;
						this->envPhase = SUSTAIN;
//						TRACE(<< "switch to SUSTAIN" << std::endl)
						break;//~for
					}
				}
				break;
				
			case SUSTAIN:
				this->releaseFromLevel = sustainLevel;
				for(; i < mixBuf.Size();){

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * sustainLevel * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}

//						TRACE(<<"t =" << this->t)
//						TRACE(<<"s =" << sample[ting::uint(this->curPhase)] << std::endl)
//						TRACE(<<"smp =" << smp << std::endl)

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + (endVolume - startVolume) * i / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;
				}//~for
				break;//~switch
				
			case RELEASE:
				for(; i < mixBuf.Size();){
					real release = this->releaseFromLevel * (real(1) - this->curTime / this->releaseTime);

					ASSERT(ting::uint(this->curPhase) < sample.Size())
					s32 smp = s32(real(0x7fff) * release * sample[ting::uint(this->curPhase)]);
					this->curPhase += this->phaseStep;
					if(ting::uint(this->curPhase) >= sample.Size()){
						this->curPhase -= real(sample.Size());
					}

					//interpolate volume
					ting::u8 vol = ting::u8(startVolume + ((endVolume - startVolume) * i) / mixBuf.Size());

					//apply volume
					smp = (smp * vol) / 0xff;

					mixBuf[i] += smp;
					++i;
					mixBuf[i] += smp;
					++i;

					this->curTime += real(1000.0f / 44100.0f);
					if(this->curTime >= this->releaseTime){
						this->curTime = 0;
						stopPlaying = true;
						this->envPhase = ATTACK;
//						TRACE(<< "switch to STOP playing" << std::endl)
						break;//~for
					}
				}
				break;
				
			default:
				ASSERT(false)
				break;
		}//~switch
	}while(!quit);

	return stopPlaying;
}



//override
Ref<Instrument::Channel> SampledInstrument::CreateChannel(){
	Ref<SampledInstrument::Channel> ch(new SampledInstrument::Channel(
			Ref<SampledInstrument>(this)
		));
	return ch;
};



//static
Ref<aumiks::Sample> TromboneInstrument::GenerateSample(){
	float cosines[] = {
		0.002f, -0.256f, -0.177f, -0.268f, -0.056f, 0.010f, 0.013f, 0.001f
	};

	float sines[] = {
		-0.181f, -0.128f, 0.122f, 0.021f, 0.034f, -0.024f, 0.000f, -0.004f
	};

	ting::StaticBuffer<float, sizeof(cosines) / sizeof(cosines[0])> c, s;
	ASSERT(c.SizeInBytes() == sizeof(cosines))
	ASSERT(s.SizeInBytes() == sizeof(sines))
	memcpy(c.Begin(), cosines, c.SizeInBytes());
	memcpy(s.Begin(), sines, s.SizeInBytes());

	//sampling rate 44100Hz, sound base freq is 25Hz
	return Sample::GenerateFromFourier(c, s, 44100 / 25, 25);
}



SampledInstrument::ADSRParams TromboneInstrument::GetADSRParams(){
	ADSRParams adsr = {
		15, //attack
		30, //decay
		0.8, //sustainLevel
		5 //release
	};
	return adsr;
}



//static
Ref<aumiks::Sample> ClarinetInstrument::GenerateSample(){
	Ref<Sample> ret(new Sample());

	//sampling rate 44100Hz, sound base freq is 25Hz
	ret->freq = real(25);
	ret->data.Init(44100 / ting::uint(ret->freq));//hold one period

	Array<real> &d = ret->data;

	typedef float T_Real;

	float f = D2Pi<float>() * float(ret->freq);

	for(ting::uint i = 0; i < ret->data.Size(); ++i){
		ASSERT(i < d.Size())
		float t = float(i) * float(1.0f / 44100.0f);
		d[i] = real(
				T_Real(-0.010f) * Cos(f * t)             + T_Real(0.077f)  * Sin(f * t) +
				T_Real(-0.130f) * Cos(T_Real(3) * f * t) + T_Real(-0.005f) * Sin(T_Real(3) * f * t) +
				T_Real(-0.210f) * Cos(T_Real(5) * f * t) + T_Real(0.134f)  * Sin(T_Real(5) * f * t) +
				T_Real(-0.040f) * Cos(T_Real(6) * f * t) + T_Real(0.030f)  * Sin(T_Real(6) * f * t) +
				T_Real(-0.050f) * Cos(T_Real(7) * f * t) + T_Real(-0.030f) * Sin(T_Real(7) * f * t) +
				T_Real(-0.030f) * Cos(T_Real(9) * f * t) + T_Real(0.016f)  * Sin(T_Real(9) * f * t)
			);
	}

	return ret;
}



SampledInstrument::ADSRParams ClarinetInstrument::GetADSRParams(){
	ADSRParams adsr = {
		10, //attack
		10, //decay
		0.8, //sustainLevel
		20 //release
	};
	return adsr;
}



//static
Ref<aumiks::Sample> ViolinInstrument::GenerateSample(){
	typedef float T_Real;

	T_Real yData[] = {
		0,
		0.4,
		0.28,
		0.36,
		0.2,
		0.76,
		0.04,
		- 0.2,
		- 0.2,
		0.52,
		0.16,
		1,
		- 0.44,
		0.92,
		- 0.68,
		- 0.24,
		- 0.48,
		- 0.48,
		- 0.04,
		- 0.88,
		- 0.88,
		- 0.72,
		- 0.88,
		- 0.76,
		0
	};

	T_Real xData[] = {
		0,
		4,
		8,
		10,
		13,
		17,
		28,
		34,
		40,
		51,
		56,
		64,
		77,
		90,
		97,
		105,
		109,
		112,
		124,
		134,
		140,
		143,
		146,
		149,
		152
	};

	ting::StaticBuffer<float, sizeof(xData) / sizeof(xData[0])> x, y;
	ASSERT(x.SizeInBytes() == sizeof(x))
	ASSERT(y.SizeInBytes() == sizeof(y))
	memcpy(x.Begin(), xData, x.SizeInBytes());
	memcpy(y.Begin(), yData, y.SizeInBytes());

	//sampling rate 44100Hz, need to hold one period
	//sound base freq is 25Hz
	Ref<Sample> ret = Sample::GenerateFromLineString(
			x,
			y,
			44100 / ting::uint(25),
			25
		);

	return ret;
}



SampledInstrument::ADSRParams ViolinInstrument::GetADSRParams(){
	ADSRParams adsr = {
		20, //attack
		100, //decay
		0.8, //sustainLevel
		20 //release
	};
	return adsr;
}



//static
Ref<aumiks::Sample> SawtoothInstrument::GenerateSample(){
	typedef float T_Real;

	T_Real yData[] = {
		0, 1, -1, 0
	};

	T_Real xData[] = {
		0, 50, 51, 100
	};

	ting::StaticBuffer<float, sizeof(xData) / sizeof(xData[0])> x, y;
	ASSERT(x.SizeInBytes() == sizeof(x))
	ASSERT(y.SizeInBytes() == sizeof(y))
	memcpy(x.Begin(), xData, x.SizeInBytes());
	memcpy(y.Begin(), yData, y.SizeInBytes());

	//sampling rate 44100Hz, need to hold one period
	//sound base freq is 25Hz
	Ref<Sample> ret = Sample::GenerateFromLineString(
			x,
			y,
			44100 / ting::uint(25),
			25
		);

	FilterOneZero f(0.5, 0.5);

	for(ting::uint i = 0; i < ret->data.Size(); ++i){
		ret->data[i] = f.Tick(ret->data[i]);
	}

	return ret;
}



SampledInstrument::ADSRParams SawtoothInstrument::GetADSRParams(){
	ADSRParams adsr = {
		20, //attack
		100, //decay
		0.8, //sustainLevel
		50 //release
	};
	return adsr;
}



//static
Ref<aumiks::Sample> ThereminInstrument::GenerateSample(){
	float cosines[] = {
		- 0.0443446,  - 0.0107732
	};

	float sines[] = {
		- 0.9653393 , - 0.1647417
	};

	ting::StaticBuffer<float, sizeof(cosines) / sizeof(cosines[0])> c, s;
	ASSERT(c.SizeInBytes() == sizeof(cosines))
	ASSERT(s.SizeInBytes() == sizeof(sines))
	memcpy(c.Begin(), cosines, c.SizeInBytes());
	memcpy(s.Begin(), sines, s.SizeInBytes());

	//sampling rate 44100Hz, sound base freq is 25Hz
	return Sample::GenerateFromFourier(c, s, 44100 / 25, 25);
}



SampledInstrument::ADSRParams ThereminInstrument::GetADSRParams(){
	ADSRParams adsr = {
		20, //attack
		0, //decay
		1, //sustainLevel
		30 //release
	};
	return adsr;
}

