//      stopish.c
//
//      Copyright 2010 Michael Cronenworth <mike@cchtml.com>
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
//      MA 02110-1301, USA.

#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#define MCE_SIGNAL_MATCH "type='signal'," \
        "interface='" MCE_SIGNAL_IF   "'," \
        "member='" MCE_DEVICE_ORIENTATION_SIG "'"

#include "stopish.h"

// Application data struct
typedef struct _AppData AppData;
struct _AppData {
    GtkWindow *main_window;
    GtkWindow *countdown_window;
    osso_context_t *osso_context;
    DBusConnection *system_bus;
};


static AppData appdata;
static int stopishMode = STOPISH_MODE_START;
static int stopishOrientation = STOPISH_LANDSCAPE;
static int stopishType = STOPISH_TYPE_STOPWATCH;


//Prototypes
gint dbus_callback( const gchar *interface, const gchar *method,
	                GArray *arguments, gpointer data, osso_rpc_t *retval );
static void main_menu( GtkWindow *window );
static void change_type_cb( GtkButton* button, gpointer data );
static void close_cb( GtkButton* button, gpointer data );
static void accelerometer_enable( void );
static void accelerometer_disable( void );
static DBusHandlerResult mce_filter_func( DBusConnection * connection,
                                          DBusMessage * message,
                                          void *data );


int main( int argc, char *argv[] )
{
    osso_return_t ret;
    HildonProgram *program;

    appdata.osso_context = osso_initialize( "com.nokia.stopish",
                                            PACKAGE_VERSION, TRUE, NULL );
    if ( appdata.osso_context == NULL ) {
        fprintf( stderr, "osso_initialize failed.\n" );
        exit( 1 );
    }

    // initialize Hildonized GTK libraries
    hildon_gtk_init( &argc, &argv );
    program = hildon_program_get_instance(  );

    // create main window
    appdata.main_window = stopish_stopwatch_new(  );

    // attach signals to main window
    g_signal_connect( G_OBJECT( appdata.main_window ), "destroy",
                      G_CALLBACK( close_cb ), appdata.main_window );
    g_signal_connect( G_OBJECT( appdata.main_window ), "focus-in-event",
                      G_CALLBACK( stopish_focus_in_cb ), NULL );
    g_signal_connect( G_OBJECT( appdata.main_window ), "focus-out-event",
                      G_CALLBACK( stopish_focus_out_cb ), NULL );

    // setup main menu
    main_menu( appdata.main_window );

    hildon_program_add_window( program, HILDON_WINDOW( appdata.main_window ) );

    // Connect to session bus, add a match rule, install filter callback
    appdata.system_bus = osso_get_sys_dbus_connection( appdata.osso_context );
    if ( appdata.system_bus ) {
        dbus_bus_add_match( appdata.system_bus, MCE_SIGNAL_MATCH, NULL );
        dbus_connection_add_filter( appdata.system_bus,
                                    mce_filter_func,
                                    NULL, NULL );
    }
    else
        g_printerr( "ERROR: Cannot connect to system dbus.\n" );

    ret = osso_rpc_set_default_cb_f( appdata.osso_context,
                                     dbus_callback, appdata.main_window );
    if ( ret != OSSO_OK ) {
        fprintf( stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret );
        exit( 1 );
    }

    gtk_main(  );

    return 0;
}


gint dbus_callback( const gchar *interface, const gchar *method,
    	            GArray *arguments, gpointer data, osso_rpc_t *retval )
{
    //printf( "stopish dbus: %s, %s\n", interface, method );

    if ( !strcmp( method, "top_application" ) )
        gtk_window_present( GTK_WINDOW( data ) );

    retval->type = DBUS_TYPE_INVALID;

    return OSSO_OK;
}


void stopish_about_cb( GtkButton* button, gpointer data )
{
    GdkPixbuf *logo;
    GError *error;
    GtkWidget *dialog;
    char *authors[2];

    authors[0] = strdup( "Michael Cronenworth" );
    authors[1] = NULL;

    dialog = gtk_about_dialog_new(  );

    gtk_about_dialog_set_program_name( GTK_ABOUT_DIALOG( dialog ),
                                       "Stopish" );
    gtk_about_dialog_set_version( GTK_ABOUT_DIALOG( dialog ),
                                  STOPISH_VERSION_STR );
    gtk_about_dialog_set_authors( GTK_ABOUT_DIALOG( dialog ),
                                  ( const char ** ) authors );
    logo = gdk_pixbuf_new_from_file( "/usr/share/icons/hicolor/40x40/hildon/stopish.png",
                                     &error );
    gtk_about_dialog_set_logo( GTK_ABOUT_DIALOG( dialog ),
                               logo );

    gtk_dialog_run( GTK_DIALOG( dialog ) );

    gtk_widget_destroy( dialog );
    free( authors[0] );
}


int stopish_get_mode( void )
{
    return stopishMode;
}


void stopish_set_mode( int newMode )
{
    stopishMode = newMode;
}


int stopish_get_type( void )
{
    return stopishType;
}


void stopish_set_type( int newType )
{
    stopishType = newType;
}


int stopish_get_orientation( void )
{
    return stopishOrientation;
}


gboolean stopish_focus_in_cb( GtkWidget *widget, GdkEventFocus *event,
                              gpointer data )
{
    // enable accelerometer hardware for portrait mode support
    accelerometer_enable(  );

    return FALSE;
}


gboolean stopish_focus_out_cb( GtkWidget *widget, GdkEventFocus *event,
                               gpointer data )
{
    // disable accelerometer for battery savings
    accelerometer_disable(  );

    return FALSE;
}


static void main_menu( GtkWindow *window )
{
    HildonAppMenu *menu;
    GtkWidget *button, *radio;

    menu = ( HildonAppMenu * ) hildon_app_menu_new(  );

    button = gtk_button_new_with_label( "Countdown" );
    g_signal_connect_after( G_OBJECT( button ), "clicked",
                            G_CALLBACK( change_type_cb ), NULL );
    hildon_app_menu_append( menu, GTK_BUTTON( button ) );

    button = gtk_button_new_with_label( "About" );
    g_signal_connect_after( G_OBJECT( button ), "clicked",
                            G_CALLBACK( stopish_about_cb ), NULL );
    hildon_app_menu_append( menu, GTK_BUTTON( button ) );

    // Hour preference
    radio = gtk_radio_button_new_with_label( NULL, "Hour" );
    gtk_toggle_button_set_mode( GTK_TOGGLE_BUTTON( radio ), FALSE );
    g_signal_connect_after( G_OBJECT( radio ), "clicked",
                            G_CALLBACK( stopish_stopwatch_perf_timer_hour ), NULL );
    hildon_app_menu_add_filter( menu, GTK_BUTTON( radio ) );

    // Minute preference
    radio = gtk_radio_button_new_with_label_from_widget( GTK_RADIO_BUTTON( radio ), "Minute" );
    gtk_toggle_button_set_mode( GTK_TOGGLE_BUTTON ( radio ), FALSE );
    g_signal_connect_after( G_OBJECT( radio ), "clicked",
                            G_CALLBACK( stopish_stopwatch_perf_timer_minute ), NULL );
    hildon_app_menu_add_filter( menu, GTK_BUTTON( radio ) );

    // default to minute
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( radio ), TRUE );

    gtk_widget_show_all( GTK_WIDGET( menu ) );

    hildon_window_set_app_menu( HILDON_WINDOW( window ), menu );
}


static void change_type_cb( GtkButton* button, gpointer data )
{
    stopish_stopwatch_reset(  );
    stopishType = STOPISH_TYPE_COUNTDOWN;
    stopish_countdown_new(  );
}


static void close_cb( GtkButton* button, gpointer data )
{
    // disable accelerometer for battery savings
    accelerometer_disable(  );

    // destroy main window and exit gtk main loop
    gtk_widget_destroy( GTK_WIDGET( data ) );
    gtk_main_quit(  );
}


static void accelerometer_enable( void )
{
    if ( osso_rpc_run_system( appdata.osso_context, MCE_SERVICE,
                              MCE_REQUEST_PATH, MCE_REQUEST_IF,
                              "req_accelerometer_enable", NULL,
                              DBUS_TYPE_INVALID ) != OSSO_OK ) {
        g_printerr("WARN: Cannot enable accelerometers\n");
    }
}


static void accelerometer_disable( void )
{
    if ( osso_rpc_run_system( appdata.osso_context, MCE_SERVICE,
                              MCE_REQUEST_PATH, MCE_REQUEST_IF,
                              "req_accelerometer_disable", NULL,
                              DBUS_TYPE_INVALID ) != OSSO_OK ) {
        g_printerr("WARN: Cannot disable accelerometers\n");
    }
}


static DBusHandlerResult mce_filter_func( DBusConnection * connection,
                                          DBusMessage * message,
                                          void *data )
{
    DBusMessageIter iter;
    char *rotation = NULL;

    if ( dbus_message_is_signal( message, MCE_SIGNAL_IF,
                                 MCE_DEVICE_ORIENTATION_SIG ) ) {
        // here if we received an orientation dbus signal
        if ( dbus_message_iter_init( message, &iter ) ) {
            dbus_message_iter_get_basic( &iter, &rotation );

            // Rotate main window
            if ( !strcmp( rotation, MCE_ORIENTATION_PORTRAIT ) ) {
                hildon_gtk_window_set_portrait_flags( GTK_WINDOW( appdata.main_window ),
                                                      HILDON_PORTRAIT_MODE_REQUEST );
                if ( stopishType == STOPISH_TYPE_STOPWATCH )
                    stopish_stopwatch_label_timer_portrait(  );
                else
                    stopish_countdown_label_timer_portrait(  );
                stopishOrientation = STOPISH_PORTRAIT;
            }
            else {
                hildon_gtk_window_set_portrait_flags( GTK_WINDOW( appdata.main_window ),
                                                      ~HILDON_PORTRAIT_MODE_REQUEST );
                if ( stopishType == STOPISH_TYPE_STOPWATCH )
                    stopish_stopwatch_label_timer_landscape(  );
                else
                    stopish_countdown_label_timer_landscape(  );
                stopishOrientation = STOPISH_LANDSCAPE;
            }
        }
        else
            g_printerr( "ERROR: dbus_message_iter_init() failed.\n" );
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
