#include "stickynoteswidget.hpp"
#include "ui_stickynoteswidget.h"
#include "dialogs.hpp"
#include "stickynotestextedit.hpp"
#include <QtMaemo5>
#include <QtWebKit/QtWebKit>

#define SETTING_NOTES_TEXT "StickyNotesText"
#define SETTING_NOTES_BG "StickyNotesBackground"
#define SETTING_NOTES_FG "StickyNotesForeground"
#define SETTING_NOTES_FONT "StickyNotesFont"
#define SETTING_NOTES_Toolbar "StickyNotesToolbar"
#define SETTING_SAVE_WAITINTERVAL 3000

StickyNotesWidget *StickyNotesWidget::createAndShowNew()
{
    StickyNotesWidget *widget = new StickyNotesWidget();
    widget->show();
    QeMaemo5DynamicWidgetHelper::globalInstance()->registerWidget(widget);
    return widget;
}

StickyNotesWidget *StickyNotesWidget::createAndShowReadOnly(const QString &text)
{
    StickyNotesWidget *widget = createAndShowNew();
    widget->_isReadOnly = true;
    widget->ui->txtNote->setReadOnly(true);
    widget->ui->txtNote->setFocusPolicy(Qt::NoFocus);
    widget->ui->txtNote->setHtml(text);
    return widget;
}

StickyNotesWidget::StickyNotesWidget(QWidget *parent)
    : QeMaemo5DynamicHomescreenWidget(true, parent),
    ui(new Ui::StickyNotesWidget),
    _isChanged(false),
    _isReadOnly(false),
    _backgroundColor(QApplication::palette().window().color()),
    _timer(new QTimer(this))
{
    ui->setupUi(this);
    ui->txtNote->setInputMethodHints(Qt::ImhNone);
    _backgroundColor.setAlpha(STICKY_NOTES_ALPHA_DEFAULT_VALUE);
    setForegroundColor(QApplication::palette().windowText().color());

    _timer->setInterval(SETTING_SAVE_WAITINTERVAL);
    _timer->setSingleShot(true);
    connect(_timer, SIGNAL(timeout()), this, SLOT(saveText()));

    connect(QApplication::instance(), SIGNAL(aboutToQuit()), this, SLOT(saveText()));
    connect(this, SIGNAL(optionsButtonClicked()), Dialogs::instance(), SLOT(showQuickOptionsDialog()));

    QShortcut *saveShortcut = new QShortcut(QKeySequence("Ctrl+S"), this);
    connect(saveShortcut, SIGNAL(activated()), this, SLOT(saveText()));
    QShortcut *boldShortcut = new QShortcut(QKeySequence("Ctrl+B"), this);
    connect(boldShortcut, SIGNAL(activated()), this, SLOT(on_btnBold_clicked()));
    QShortcut *italicShortcut = new QShortcut(QKeySequence("Ctrl+I"), this);
    connect(italicShortcut, SIGNAL(activated()), this, SLOT(on_btnItalic_clicked()));
    QShortcut *underlineShortcut = new QShortcut(QKeySequence("Ctrl+U"), this);
    connect(underlineShortcut, SIGNAL(activated()), this, SLOT(on_btnUnderline_clicked()));
    QShortcut *strikeThroughShortcut = new QShortcut(QKeySequence(tr("Ctrl+T")), this);
    connect(strikeThroughShortcut, SIGNAL(activated()), this, SLOT(on_btnStrikeThrough_clicked()));

    ui->editorToolBar->hide();
    connect(ui->txtNote, SIGNAL(focusChanged()), this, SLOT(toggleToolbar()));
}

StickyNotesWidget::~StickyNotesWidget()
{
    disconnect(QApplication::instance(), SIGNAL(aboutToQuit()), this, SLOT(saveText()));
    delete ui;
}

void StickyNotesWidget::paintEvent(QPaintEvent *event)
{
    QPainter p(this);
    p.setBrush(_backgroundColor);
    p.setPen(Qt::NoPen);
    QRect myRect(rect().left() + 8, rect().top(), rect().width() - 16, rect().height() - 65);
    p.drawRect(myRect);

    QWidget::paintEvent(event);
}

void StickyNotesWidget::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type())
    {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void StickyNotesWidget::showSettingsDialog()
{
    Dialogs::instance()->showAboutDialog();
}

bool StickyNotesWidget::restoreWidgetState()
{
    QString noteText = loadSetting(SETTING_NOTES_TEXT).toString();
    ui->txtNote->setHtml(noteText);

    QColor color = loadSetting(SETTING_NOTES_BG, _backgroundColor).value<QColor>();
    setBackgroundColor(color);

    color = loadSetting(SETTING_NOTES_FG, _foregroundColor).value<QColor>();
    setForegroundColor(color);

    QFont font = loadSetting(SETTING_NOTES_FONT, QFont()).value<QFont>();
    setFont(font);

    _isChanged = false;

    return true;
}

void StickyNotesWidget::saveText()
{
    if (_isChanged && !_isReadOnly)
    {
        _isChanged = false;
        saveSetting(SETTING_NOTES_TEXT, ui->txtNote->toHtml());
        qDebug() << "saved text for " << appletId();
    }
}

bool StickyNotesWidget::isChanged() const
{
    return _isChanged;
}

bool StickyNotesWidget::isReadOnly() const
{
    return _isReadOnly;
}

void StickyNotesWidget::setBackgroundColor(const QColor &color)
{
    if (color.isValid())
    {
        _backgroundColor = color;
        saveSetting(SETTING_NOTES_BG, _backgroundColor);
        update();
    }
}

void StickyNotesWidget::setForegroundColor(const QColor &color)
{
    if (color.isValid())
    {
        _foregroundColor = color;
        saveSetting(SETTING_NOTES_FG, _foregroundColor);

        QTextCursor cursor = ui->txtNote->textCursor();
        ui->txtNote->selectAll();
        ui->txtNote->setTextColor(color);
        QPalette palette = ui->txtNote->palette();
        palette.setColor(QPalette::WindowText, color);
        palette.setColor(QPalette::Text, color);
        ui->txtNote->setPalette(palette);
        ui->txtNote->setTextCursor(cursor);
    }
}

void StickyNotesWidget::setFont(const QFont &newFont)
{
    QFont oldFont = font();
    QeMaemo5DynamicHomescreenWidget::setFont(newFont);

    //qDebug() << "\n\nbefore:" << ui->txtNote->toHtml();
    QString str = ui->txtNote->toHtml()
                  .replace(QRegExp("font-family:\\s?([^;]+);"), "font-family: '" + newFont.family() + "';")
                  .replace(QRegExp("font-size:\\s?([^;]+);"), "font-size: '" + QString::number(newFont.pointSize()) + "';");
    //qDebug() << "\n\nafter:" << str;

    ui->txtNote->setHtml(str);
    ui->txtNote->setFont(newFont);

    if (newFont.bold() != oldFont.bold() || newFont.italic() != oldFont.italic())
    {
        QTextCursor cursor = ui->txtNote->textCursor();
        ui->txtNote->selectAll();
        ui->txtNote->setFontWeight(newFont.bold() ? QFont::Bold : QFont::Normal);
        ui->txtNote->setFontItalic(newFont.italic());
        ui->txtNote->setTextCursor(cursor);
    }

    saveSetting(SETTING_NOTES_FONT, font());
}

void StickyNotesWidget::setAlpha(int newAlpha)
{
    _backgroundColor.setAlpha(newAlpha);
    setBackgroundColor(_backgroundColor);
}

const QColor &StickyNotesWidget::backgroundColor() const
{
    return _backgroundColor;
}

const  QColor &StickyNotesWidget::foregroundColor() const
{
    return _foregroundColor;
}

void StickyNotesWidget::toggleToolbar()
{
    if (ui->editorToolBar->isVisible())
    {
        ui->txtNote->setFixedHeight(ui->txtNote->height() + ui->editorToolBar->height());
        ui->editorToolBar->hide();
    }
    else
    {
        ui->editorToolBar->show();
        ui->txtNote->setFixedHeight(ui->txtNote->height() - ui->editorToolBar->height());
    }
}

void StickyNotesWidget::setToggleButtonsState()
{
    const QTextCharFormat &format = ui->txtNote->currentCharFormat();

    if (format.fontWeight() > QFont::Normal)
        ui->btnBold->setChecked(true);
    else
        ui->btnBold->setChecked(false);

    if (format.fontItalic())
        ui->btnItalic->setChecked(true);
    else
        ui->btnItalic->setChecked(false);

    if (format.fontUnderline())
        ui->btnUnderline->setChecked(true);
    else
        ui->btnUnderline->setChecked(false);

    if (format.fontStrikeOut())
        ui->btnStrikeThrough->setChecked(true);
    else
        ui->btnStrikeThrough->setChecked(false);
}

void StickyNotesWidget::on_btnAdd_clicked()
{
    StickyNotesWidget *newWidget = new StickyNotesWidget;
    newWidget->setBackgroundColor(_backgroundColor);
    newWidget->setForegroundColor(_foregroundColor);
    newWidget->setFont(font());
    newWidget->show();
    QeMaemo5DynamicWidgetHelper::globalInstance()->registerWidget(newWidget);
}

void StickyNotesWidget::on_btnOptions_clicked()
{
    optionsButtonClicked();
}

void StickyNotesWidget::on_btnExit_clicked()
{
    if (ui->txtNote->toPlainText() == STICKY_NOTES_TEXT_EDIT_INITIAL_TEXT || QMessageBox::Yes == QMessageBox::question(NULL, "Are you sure?", "Do you really want to close this widget?\nThe note inside it will be permamently deleted!", QMessageBox::Yes, QMessageBox::No, QMessageBox::Cancel))
        close();
}

void StickyNotesWidget::on_txtNote_textChanged()
{
    _isChanged = true;
    _timer->start();
}

void StickyNotesWidget::on_btnBold_clicked()
{
    ui->txtNote->setFontWeight(ui->txtNote->fontWeight() > QFont::Normal ? QFont::Normal : QFont::Black);
    ui->txtNote->setFocus();
    setToggleButtonsState();
}

void StickyNotesWidget::on_btnItalic_clicked()
{
    ui->txtNote->setFontItalic(!ui->txtNote->fontItalic());
    ui->txtNote->setFocus();
    setToggleButtonsState();
}

void StickyNotesWidget::on_btnUnderline_clicked()
{
    ui->txtNote->setFontUnderline(!ui->txtNote->fontUnderline());
    ui->txtNote->setFocus();
    setToggleButtonsState();
}

void StickyNotesWidget::on_btnStrikeThrough_clicked()
{
    ui->txtNote->toggleStrikeThrough();
    ui->txtNote->setFocus();
    setToggleButtonsState();
}

void StickyNotesWidget::on_txtNote_cursorPositionChanged()
{
    setToggleButtonsState();
}

void StickyNotesWidget::on_txtNote_selectionChanged()
{
    setToggleButtonsState();
}
