/*
 * This file is a part of hildon-extras
 *
 * Copyright (C) 2010 Timur Kristf <venemo@msn.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define QT_NO_KEYWORDS

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include "qt-he-wrapper.h"
#include "he-about-dialog.h"
#include "he-simple-color-dialog.h"
#include "he-color-dialog.h"
#include "he-font-dialog.h"

#define QSTRING_TO_CHAR_PTR(qstring) (qstring.length() == 0 ? NULL : qstring.toLatin1().data())
#define BOOL_TO_GBOOLEAN(cppbool) (cppbool ? TRUE : FALSE)

static GdkColor *qColorToGdkColor(const QColor& qcolor)
{
    GdkColor *result = NULL;

    if (qcolor.isValid())
    {
        result = new GdkColor;
        result->red = qcolor.red() * 256;
        result->green = qcolor.green() * 256;
        result->blue = qcolor.blue() * 256;
    }

    return result;
}

static QColor gdkColorToQColor(const GdkColor *gdkcolor)
{
    return QColor(gdkcolor->red / 256, gdkcolor->green / 256, gdkcolor->blue / 256);
}

QColor QtHeWrapper::showHeSimpleColorDialog(QWidget *parent, const QColor &defaultColor, bool showAdvancedButton, const QString &dialogTitle)
{
    GdkColor *gdkDefaultColor = qColorToGdkColor(defaultColor);
    GdkColor *gdkResult = he_simple_color_dialog_present_interop(parent == NULL ? 0 : parent->winId(), BOOL_TO_GBOOLEAN(showAdvancedButton), gdkDefaultColor, QSTRING_TO_CHAR_PTR(dialogTitle));

    QColor result;

    if (gdkResult != NULL)
    {
        result = gdkColorToQColor(gdkResult);
        gdk_color_free(gdkResult);
    }

    if (gdkDefaultColor != NULL)
        delete gdkDefaultColor;

    return result;
}

QColor QtHeWrapper::showHeColorDialog(QWidget *parent, const QColor &defaultColor, const QString &dialogTitle)
{
    GdkColor *gdkDefaultColor = qColorToGdkColor(defaultColor);
    GdkColor *gdkResult = he_color_dialog_present_interop(parent == NULL ? 0 : parent->winId(), gdkDefaultColor, QSTRING_TO_CHAR_PTR(dialogTitle));

    QColor result;

    if (gdkResult != NULL)
    {
        result = gdkColorToQColor(gdkResult);
        gdk_color_free(gdkResult);
    }
    if (gdkDefaultColor != NULL)
    {
        delete gdkDefaultColor;
    }

    return result;
}

void QtHeWrapper::showHeAboutDialog(QWidget *parent, const QString &description, const QString &copyright, const QString &websiteUrl, const QString &bugtrackerUrl, const QString &donateUrl, const QString &iconName, const QString &appName, const QString &version)
{
    he_about_dialog_present_interop(parent == NULL ? 0 : parent->winId(),
                            QSTRING_TO_CHAR_PTR(appName),
                            QSTRING_TO_CHAR_PTR(iconName),
                            QSTRING_TO_CHAR_PTR(version),
                            QSTRING_TO_CHAR_PTR(description),
                            QSTRING_TO_CHAR_PTR(copyright),
                            QSTRING_TO_CHAR_PTR(websiteUrl),
                            QSTRING_TO_CHAR_PTR(bugtrackerUrl),
                            QSTRING_TO_CHAR_PTR(donateUrl));
}

QFont QtHeWrapper::showHeFontDialog(QWidget *parent, const QFont &defaultFont, const QString &dialogTitle)
{
    QFont result;

    // The format of the font string is:
    // font_family [bold] [italic] font_pointsize
    // We must give the HeFontDialog this format, and it also returns its string in this format.

    QString fontString = defaultFont.family();
    if (defaultFont.weight() > QFont::Normal)
        fontString += " bold";
    if (defaultFont.italic())
        fontString += " italic";
    fontString += " " + QString::number(defaultFont.pointSize());

    gchar *gResult = he_font_dialog_present_interop(parent == NULL ? 0 : parent->winId(), fontString.toLatin1().data(), QSTRING_TO_CHAR_PTR(dialogTitle));
    fontString = gResult;
    delete gResult;

    QStringList list = fontString.split(' ');
    int size = list.last().toInt();
    if (size > 0)
        result.setPointSize(size);
    list.removeLast();

    bool foundItalic = false;
    bool foundBold = false;

    fontString = "";
    for (int i = list.count() - 1; i >= 0; i--)
    {
        if (!foundItalic && list[i] == "italic")
        {
            result.setItalic(true);
            foundItalic = true;
            list.removeLast();
        }
        else if (!foundBold && list[i] == "bold")
        {
            result.setBold(QFont::Black);
            foundBold = true;
            list.removeLast();
        }
        else
        {
            foundItalic = true;
            foundBold = true;
            fontString = list[i] + " " + fontString;
        }
    }
    fontString = fontString.trimmed();
    result.setFamily(fontString);

    return result;
}
