//-----------------------------------------------------------------------------
// $Id: squeezster.cpp 233 2012-02-06 20:27:07Z pakarlss $
//-----------------------------------------------------------------------------

#include "squeezstercore.h"

SqueezsterCore::SqueezsterCore(int debug,QObject *parent) :
    QObject(parent),    networkSession(0)
{
    setDebugFlag(debug);
    //-------------------------------------------------
    // Do initialization
    //-------------------------------------------------
    //jsonTimer=0;
    cometReply=0;
    cometReplyDelete=0;
    initCometReply=0;
    cometTimeOut=30000;
    cometTimer = new QTimer();
    cometTimer->setInterval(cometTimeOut);
    cometTimer->setSingleShot(true);
    genericTimer.setInterval(JSON_TIMEOUT);
    genericTimer.setSingleShot(true);
    disconnectEnable=false;
    activePlayerSet=false;
    initDone=false;
    cometID="";
    playerCount=0;
    setActivePlayerIndex(-1);
    setCometTimeOutCount(0);
    setCurrentSong(QVariantMap());
    netInfo = new QSystemNetworkInfo(this);

    //-------------------------------------------------------------------------

    qDebug() << "-----------------------------";
    qDebug() << "Squeezster by Patrik Karlsson";
    qDebug() << "Version (Core): " << VERSION_CORE;
    qDebug() << "-----------------------------";

    // Server discovery. Wait 1 s before calling:
    clientUdpSocket = new QUdpSocket(this);
    clientUdpSocket->bind(DISCOVERYPORT_SC, QUdpSocket::ShareAddress);
    connect(clientUdpSocket, SIGNAL(readyRead()), this, SLOT(receiveUdpDatagram()));
    QTimer serverDiscoveryTimer;
    serverDiscoveryTimer.setInterval(60000);
    serverDiscoveryTimer.setSingleShot(false);
    connect(&serverDiscoveryTimer,SIGNAL(timeout()),this,SLOT(serverDiscovery()));
    serverDiscoveryTimer.start();
    QTimer::singleShot(2000,this,SLOT(serverDiscovery())); // Run on startup

    QCoreApplication::setOrganizationName("Pakarlss");
    QCoreApplication::setApplicationName("Squeezster");
    squeezsterSettings = new QSettings("pakarlss", "Squeezster");
    readSettings(); //settings);
    uuid = QUuid::createUuid().toString().toUtf8();

    connect(this,SIGNAL(connectionSettingsChanged()),this,SLOT(writeConnectionSettings()));
    connect(this,SIGNAL(optionsSettingsChanged()),this,SLOT(writeOptionsSettings()));
    connect(this,SIGNAL(activePlayerChanged()),this,SLOT(writeOptionsSettings()));
    connect(this,SIGNAL(activePlayerChanged()),this,SLOT(playerStatusSlot()));

    connect(this,SIGNAL(jsonServerVersion(QString)),this,SLOT(checkServerVersion(QString)));
    connect(this,SIGNAL(serverVersionOK()),this,SLOT(cometInit()));
}

//----------------------------------------------
//----------------------------------------------
void SqueezsterCore::start()
{
    _DEBUG_ << "SqueezsterCore::start()";
    if ((manager.capabilities() & QNetworkConfigurationManager::NetworkSessionRequired) &&
            (manager.capabilities() & QNetworkConfigurationManager::CanStartAndStopInterfaces)) {
        _DEBUG_ << "Starting network session";
        openNetworkSession();
    } else {
        _DEBUG_ << "Cannot get an available network configuration.";
        setMailbox("Cannot get an available network configuration.");
    }
}

void SqueezsterCore::stop(QString server)
{
    cometDisconnect(server);
}

void SqueezsterCore::restart()
{
    stop();
    QTimer::singleShot(500,this,SIGNAL(start()));
}

//----------------------------------------------
//----------------------------------------------
void SqueezsterCore::sessionErrorSlot()
{
    _DEBUG_ << "sessionErrorSlot() Error:" << networkSession->errorString();
    setMailbox(networkSession->errorString());
}
void SqueezsterCore::openNetworkSession() {
    _DEBUG_ << "openNetworkSession()";
    if(networkSession) {
        disconnect(networkSession,SIGNAL(closed()),this,SLOT(openNetworkSession()));
        networkSession->close();
        networkSession->deleteLater();
    }
    networkSession= new QNetworkSession(manager.defaultConfiguration(),this);
    networkSession->open();
    if(networkSession->waitForOpened(10000)) {
        _DEBUG_ << "Networksession successfully opened.";
    } else {
        _DEBUG_ << "NetworkSession TimeOut!\nNetwork is required!\nPlease connect or close the application";
        setMailbox("Network is required!\n\nPlease connect or close the application");
        networkSession->open();
        networkSession->waitForOpened(-1);
    }
    // Initialize networkmanager
    if(jsonNetworkManager)
        delete jsonNetworkManager;
    if(cometNetworkManager)
        delete cometNetworkManager;
    if(httpNetworkManager)
        delete httpNetworkManager;
    jsonNetworkManager = new QNetworkAccessManager(this);
    cometNetworkManager = new QNetworkAccessManager(this);
    httpNetworkManager = new QNetworkAccessManager(this);
    connect(jsonNetworkManager,SIGNAL(authenticationRequired(QNetworkReply*,QAuthenticator*)),this,SLOT(httpAuthentSlot(QNetworkReply*, QAuthenticator*)));
    connect(cometNetworkManager,SIGNAL(authenticationRequired(QNetworkReply*,QAuthenticator*)),this,SLOT(httpAuthentSlot(QNetworkReply*, QAuthenticator*)));
    connect(httpNetworkManager,SIGNAL(authenticationRequired(QNetworkReply*,QAuthenticator*)),this,SLOT(httpAuthentSlot(QNetworkReply*, QAuthenticator*)));
    connect(networkSession,SIGNAL(closed()),this,SLOT(openNetworkSession()));
    QTimer::singleShot(0,this,SLOT(initEnvironment()));
}

//----------------------------------------------
// SqueezsterCore::initEnvironment()
// Init server connection,
// find players & request basic data.
//----------------------------------------------
void SqueezsterCore::initEnvironment()
{
    _DEBUG_ << "\ninitEnvironment() network name\n" << netInfo->networkName(netInfo->currentMode())<<netInfo->currentMode();
    _DEBUG_ << "network accessible? " << jsonNetworkManager->networkAccessible();
#if defined(MEEGO_EDITION_HARMATTAN) || defined(Q_WS_MAEMO_5)
    if(manager.isOnline()) { // && networkSession->isOpen()) {
#endif
        if(connectionSettings().value("ipaddress").toString().isEmpty() && !connectionSettings().value("mysb").toBool()) {
            _DEBUG_ << "Ipaddress empty";
            emit networkError();
        } else {
            disconnect(&manager,SIGNAL(onlineStateChanged(bool)), this, SLOT(initEnvironment()));
            cometTimer->stop();

            jsonNetworkRequest.setUrl(jsonHttpAddress());
            jsonNetworkRequest.setRawHeader("Content-type","application/json");

            envClear();
            disconnectEnable=false;
            if(onMysb()) // && loginSucceeded)
                mysbLogin();
            else {
                QTimer::singleShot(1000,this,SLOT(requestServerVersion()));
            }
            _DEBUG_<< "********************************";
            _DEBUG_<< "**         initDone           **";
            _DEBUG_<< "********************************";
        }
#if defined(MEEGO_EDITION_HARMATTAN) || defined(Q_WS_MAEMO_5)
    } else {
        _DEBUG_ << "Offline.";// Trying again in 3 seconds.";
        connect(&manager,SIGNAL(onlineStateChanged(bool)), this, SLOT(initEnvironment()));
    }
#endif
}

//----------------------------------------------
// SqueezsterCore::mysbLogin
//----------------------------------------------
void SqueezsterCore::mysbLogin()
{
    _DEBUG_ << "mysbLogin()";
    QByteArray authBA = QByteArray("?email="+connectionSettings().value("mysbusername").toString().toUtf8());
    authBA += QByteArray("&password="+connectionSettings().value("mysbpassword").toString().toUtf8());
    QNetworkRequest loginRequest;
    loginRequest.setUrl(QString(MYSB_ADDRESS_LOGIN+authBA));

    QNetworkReply* cometReply=cometNetworkManager->post(loginRequest,QByteArray());
    connect(cometReply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(networkErrorSlot(QNetworkReply::NetworkError)));
    QNetworkReply* jsonReply=jsonNetworkManager->post(loginRequest,QByteArray());
    connect(jsonReply,SIGNAL(finished()),this,SLOT(requestServerVersion()));
    connect(jsonReply,SIGNAL(finished()),this,SLOT(jsonParse()));
    connect(jsonReply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(networkErrorSlot(QNetworkReply::NetworkError)));

    // Revert json address:
    jsonNetworkRequest.setUrl(jsonHttpAddress());
}

//----------------------------------------------
// SqueezsterCore::requestServerVersion()
//----------------------------------------------
void SqueezsterCore::requestServerVersion()
{
    _DEBUG_ << "requestServerVersion()";
    QStringList jsonParam;
    jsonParam << "version" << "?";
    cometTimer->stop();

    if(onMysb()) {
        emit serverVersionOK(); // Ack. server version on MySB
    } else {
        //connect(&genericTimer,SIGNAL(timeout()),this,SLOT(jsonTimeOutSlot()));
        //genericTimer.start();
        jsonReply=sendJson(jsonParam);
    }
}

//----------------------------------------------
// SqueezsterCore::initLists()
//----------------------------------------------
void SqueezsterCore::initLists()
{
    // Preload interesting lists from the server:
    _DEBUG_ << "initLists()";
    playerStatusSlot();
    requestList("radios","",true); // Radios list
    requestList("apps","",true); // Applications list
    //    if(!onMysb())
    //        requestList("albums"); //,MAXITEMSPERREQUEST);
}

//----------------------------------------------
// SqueezsterCore::checkServerVersion()
//----------------------------------------------
void SqueezsterCore::checkServerVersion(const QString& serverVersionStr)
{
    _DEBUG_ << "Server version:" << serverVersionStr;
    QStringList serverVersionStrLis = serverVersionStr.split(".",QString::SkipEmptyParts);
    bool oldServer=true;
    if (serverVersionStrLis.at(0).toInt()<SERVERVERSION_MAJOR) {
        oldServer=true; }
    else if(serverVersionStrLis.at(0).toInt()>SERVERVERSION_MAJOR) {
        oldServer=false; }
    else {
        if (serverVersionStrLis.at(1).toInt()<SERVERVERSION_MIDDLE) {
            oldServer=true; }
        else if (serverVersionStrLis.at(1).toInt()>SERVERVERSION_MIDDLE) {
            oldServer=false; }
        else {
            if (serverVersionStrLis.at(2).toInt()<SERVERVERSION_MINOR) {
                oldServer=true; }
            else {
                oldServer=false; }
        }
    }
    if (oldServer && !onMysb()) {
        QByteArray message = "Squeezebox Server version too old. Please update the server.\nMinimum required version: ";
        message.append(QByteArray::number(int(SERVERVERSION_MAJOR))+".");
        message.append(QByteArray::number(int(SERVERVERSION_MIDDLE))+".");
        message.append(QByteArray::number(int(SERVERVERSION_MINOR)));
        setMailbox(message);
    }
    else {
        emit serverVersionOK();
    }
}
//----------------------------------------------
// SqueezsterCore::httpAuthentSlot
// Update connection if settings have changed.
//----------------------------------------------
void SqueezsterCore::httpAuthentSlot(QNetworkReply* reply, QAuthenticator* authenticator)
{
    if(onMysb()) {
        authenticator->setUser(connectionSettings().value("mysbusername").toString()); // settings.username());
        authenticator->setPassword(connectionSettings().value("mysbpassword").toString()); // settings.password());
    } else {
        authenticator->setUser(connectionSettings().value("username").toString()); // settings.username());
        authenticator->setPassword(connectionSettings().value("password").toString()); // settings.password());
    }
    _DEBUG_ << "httpAuthentSlot(QNetworkReply* networkReply, QAuthenticator* authenticator)"; // << authenticator->user() << authenticator->password();
    authCount_i++;
    if(authCount() == 25) {// Probably wrong credentials
        _DEBUG_ << "Calling setMailbox ";
        setMailbox("Check your username and password");
        emit networkError();
    }
}

//----------------------------------------------
// SqueezsterCore::sendJson()
//----------------------------------------------
QNetworkReply* SqueezsterCore::sendJson(const QStringList& commandList,QByteArray playerID,QByteArray server)
{
    _DEBUG_ << "sendJson()";
    QByteArray paramSer,commandSer;
    QStringList parameters=QStringList(commandList);
    QString string;
    QByteArray stringBA;
    paramSer = "\[";
    while (!parameters.isEmpty()) {
        paramSer.append("\"");
        string=parameters.takeFirst();
        stringBA=string.toUtf8();
        string=QUrl::fromPercentEncoding(stringBA);
        _DEBUG_ << "sendJson() string: " << string;
        paramSer.append(string.toUtf8()); // URI encoded
        paramSer.append("\"");
        if(!parameters.isEmpty())
            paramSer.append(",");
    }
    _DEBUG_ << "sendJson() paramSer: " << paramSer;
    paramSer.append("]");
    commandSer=QByteArray("[\"" +playerID+"\","+paramSer+"]");
    QByteArray jsonObjectSer = "{\"id\":1,\"method\":\"slim.request\",\"params\":"+commandSer;
    jsonObjectSer += "}";

    if(server.isEmpty()) {
        _DEBUG_ << "sendJson() jsonServer" << jsonHttpAddress();
        jsonNetworkRequest.setUrl(jsonHttpAddress());
    } else {
        _DEBUG_ << "sendJson() server";
        jsonNetworkRequest.setUrl(jsonHttpAddress());
    }
    _DEBUG_ << "sendJson()"<< jsonNetworkRequest.url() << jsonObjectSer << server << jsonHttpAddress();

    QNetworkReply* jsonReply = jsonNetworkManager->post(jsonNetworkRequest,jsonObjectSer);
    connect(jsonReply, SIGNAL(finished()), this, SLOT(jsonParse()));
    connect(jsonReply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(networkErrorSlot(QNetworkReply::NetworkError)));
    return(jsonReply);
}

//----------------------------------------------
// SqueezsterCore::jsonParse()
//----------------------------------------------
void SqueezsterCore::jsonParse()
{
    QNetworkReply* jsonReply = static_cast<QNetworkReply*>(QObject::sender());
    QByteArray replyMessage = jsonReply->readAll();
    _DEBUG_ << "SqueezsterCore::jsonParse()" << replyMessage;
    QVariantList request;
    QByteArray playerID="";
    QByteArray command;
    QByteArray param1;
    QVariantMap jsonObject=Json::parse(replyMessage).toMap();
    QVariantList  params = jsonObject.value("params").toList(); // = new(QVariantMap);
    QVariantMap  result = jsonObject.value("result").toMap(); // = new(QVariantMap);
    QByteArray error = jsonObject.value("error").toString().toUtf8();
    QByteArray message = jsonObject.value("textarea").toString().toUtf8();
    int mustRegister = jsonObject.value("must_register").toInt();
    bool items = false;
    //_DEBUG_ << "SqueezsterCore::jsonParse() params:" << params;

    genericTimer.stop(); // Stop timeout timer
    disconnect(&genericTimer,SIGNAL(timeout()),this,SLOT(jsonTimeOutSlot()));

    if(!params.isEmpty()) //Check playerID
        playerID=params.at(0).toString().toUtf8();
    if(params.count()>1 && !params.at(1).toList().isEmpty()) // Check originating command
        request=params.at(1).toList();
    if(request.count()) { // Check originating command
        command=request.takeFirst().toString().toUtf8();
    }
    if(request.count()) { // Check parameter
        param1=request.takeFirst().toString().toUtf8();
        if(param1=="items") {
            items = true;
            param1=request.takeFirst().toString().toUtf8();
        }
    }
    jsonReply->deleteLater();
    _DEBUG_ << "SqueezsterCore::jsonParse() Command: " << command; // << "Message: " << replyMessage;
    _DEBUG_ << "SqueezsterCore::jsonParse() Param1: " << param1;

    if (mustRegister) // Cookie not valid?
        mysbLogin(); // Login to mysb.com
    else if(!error.isEmpty())
    {   _DEBUG_ << "SqueezsterCore::jsonParse() Error: " << error;
        setMailbox(error); }
    else if(!message.isEmpty())
    {   _DEBUG_ << "SqueezsterCore::jsonParse() Message: " << message;
        setMailbox(message);
        emit networkError();
    } else if (!initDone && replyMessage.startsWith("<?xml version=")) { // Login Failed
        _DEBUG_ << "Sending to mailbox.";
        setMailbox("Login failed. Please check your connection settings.");
        emit networkError(); }
    else // Parse message
        mapParse(result,command,param1,playerID,false,items);
}

//----------------------------------------------
// SqueezsterCore::cometInit()
// Init comet communication
//----------------------------------------------
void SqueezsterCore::cometInit() {
    _DEBUG_ << "cometInit()";
    //QJson::Serializer serializer;
    cID=0;
    cometID="";
    uuid.replace("-","");
    uuid.replace("{","");
    uuid.replace("}","");

    cometExtVM["rev"]="1.2.1";
    cometExtVM["uuid"]=uuid;
#if defined(MEEGO_EDITION_HARMATTAN) || defined(Q_WS_MAEMO_5)// || defined(Q_OS_SYMBIAN)
    cometExtVM["mac"]=netInfo->macAddress(QSystemNetworkInfo::EthernetMode);
#endif
    //    cometExtBA=QByteArray("\"ext\":"+serializer.serialize(cometExtVM));
    cometExtBA=QByteArray("\"ext\":"+Json::serialize(cometExtVM));
    cIDStr="\"id\":\""+QByteArray::number(cID++)+"\"";
    QByteArray cometMessage=
            "[{\"channel\":\"/meta/handshake\","+cIDStr+","+cometExtBA+",\"supportedConnectionTypes\":[\"streaming\"],\"version\":\"1.0\"}]";
    _DEBUG_ << "Comet Init. Init message:" << cometMessage;
    _DEBUG_ << "Comet Init. IP Address:" << cometHttpAddress();

    cometNetworkRequest.setAttribute(QNetworkRequest::CacheLoadControlAttribute,QNetworkRequest::AlwaysNetwork);
    cometNetworkRequest.setUrl(cometHttpAddress());
    cometNetworkRequest.setRawHeader("Content-type","application/json");
    cometNetworkRequest.setRawHeader("Connection","Keep-Alive");
    cometNetworkRequest.setAttribute(QNetworkRequest::User,QString("initComet"));
    cometNetworkRequest.setRawHeader("CometID",QByteArray::number(cID));
    initCometReply = cometNetworkManager->post(cometNetworkRequest,cometMessage);
    connect(initCometReply,SIGNAL(finished()),this,SLOT(cometInitSubscribe()));
    connect(initCometReply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(networkErrorSlot(QNetworkReply::NetworkError)));
    connect(cometTimer,SIGNAL(timeout()),this,SLOT(cometTimeOutSlot()));
    cometTimer->start();
}

//----------------------------------------------
// SqueezsterCore::cometInitSubscribe()
// Start subscription
//----------------------------------------------
void SqueezsterCore::cometInitSubscribe() {
    //disconnect(cometTimer,SIGNAL(timeout()),this,0);
    //connect(cometTimer,SIGNAL(timeout()),this,SLOT(cometTimeOutSlot()));
    QNetworkReply* reply = static_cast<QNetworkReply*>(this->sender());
    QByteArray cometMessage = reply->readAll();
    QVariantMap cometMap;
    QVariantList cometObject=Json::parse(cometMessage).toList();
    QByteArray command;

    cometTimer->stop();

    if (initCometReply)
        initCometReply->deleteLater();

    if(!cometObject.isEmpty()) {
        cometMap=cometObject.takeFirst().toMap();
        _DEBUG_ << "cometHandshaken() Successful: " << cometMap.value("successful").toBool() << cometMap;
        if(cometMap.value("successful").toBool()) {
            cometID = cometMap.value("clientId").toString().toUtf8();
            cometTimeOut = cometMap.value("advice").toMap().value("timeout").toInt();
            cometTimer->setInterval(cometTimeOut);
            cometInterval = cometMap.value("advice").toMap().value("interval").toInt();
            _DEBUG_ << "Comet details, ID:" << cometID << "TimeOut:" << cometTimeOut << "Interval:" << cometInterval;
            cIDStr="\"id\":\""+QByteArray::number(cID++)+"\"";
            cometMessage =QByteArray("[{\"channel\":\"/meta/connect\",\"clientId\":\""+cometID+ "\","+cIDStr+","+cometExtBA+",\"connectionType\":\"streaming\"},");
            cIDStr="\"id\":\""+QByteArray::number(cID++)+"\"";

            cometMessage+=QByteArray("{\"channel\":\"/meta/subscribe\",\"clientId\":\"" +cometID+ "\","+cIDStr+","+cometExtBA+",\"subscription\":\"/"+cometID+"/**\"}]");
            //_DEBUG_ << "Comet Mess:" << cometMessage;

            initCometReply = cometNetworkManager->post(cometNetworkRequest,cometMessage);
            connect(initCometReply,SIGNAL(readyRead()),this,SLOT(cometRead()));
            connect(initCometReply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(networkErrorSlot(QNetworkReply::NetworkError)));

            cometTimer->start();

            if(onMysb()) {
                QByteArray loginMessage;
                loginReply=true;
                loginMessage= QByteArray("register_sn 0 100 login_password 0 100 email:"+connectionSettings().value("mysbusername").toString().toUtf8()+" password:"+connectionSettings().value("mysbpassword").toString().toUtf8()+" useContextMenu:1");
                cometRequest(cometNetworkManager,cometID,QByteArray(COMET_REQUEST_CH),loginMessage,"");
                QTimer::singleShot(2000,this,SLOT(initServer()));
            } else {
                initServer();
            }
        } else {
            // Subscription lost, reconnect:
            cometDisconnect(cometHttpAddress());
            QTimer::singleShot(0,this,SLOT(initEnvironment()));
        }
    }
}

//----------------------------------------------
// SqueezsterCore::initServer()
//----------------------------------------------
void SqueezsterCore::initServer()
{
    initDone=true;
    // Start subscribing on server:
    QByteArray command="serverstatus 0 255 subscribe:30 prefs:ignoredarticles,browseagelimit,noGenreFilter,PLUGIN_TRACKSTAT,audiodir playerprefs:playtrackalbum,digitalVolumeControl ";
    sendComet(command,CometSubscription);
    emit cometInitDone();
}

//----------------------------------------------
// SqueezsterCore::cometDisconnect(QString)
//----------------------------------------------
void SqueezsterCore::cometDisconnect(QString server) {
    server=server.isEmpty()?(prevServerHttpAddress_i+"/cometd"):server;
    _DEBUG_ << "cometDisconnect(QString server)" << server << cometID;
    disconnectEnable=true;
    if (!cometID.isEmpty()) {
        QByteArray cometMessage =QByteArray("[{\"channel\":\"/meta/disconnect\",\"clientId\":\""+cometID+ "\","+cIDStr+","+cometExtBA+",\"connectionType\":\"streaming\"},");
        cIDStr="\"id\":\""+QByteArray::number(cID++)+"\"";
        cometNetworkRequest.setUrl(server);
        cometReply = cometNetworkManager->post(cometNetworkRequest,cometMessage);
        cometTimer->stop();
        // Delete previos subscription
        _DEBUG_ << "Closing initCometReply";
        if(initCometReply) {
            initCometReply->close();
            initCometReply->deleteLater();
        }
        if(jsonReply)
            jsonReply->deleteLater();
        cometNetworkRequest.setUrl(cometHttpAddress());
    }
}

//----------------------------------------------
// SqueezsterCore::replyDelete()
//----------------------------------------------
void SqueezsterCore::replyDelete() {
    QNetworkReply* reply = static_cast<QNetworkReply*>(this->sender());
    reply->deleteLater();
}

//----------------------------------------------
// SqueezsterCore::cometRead()
// Read comet message
//----------------------------------------------
void SqueezsterCore::cometRead() {
    _DEBUG_ << "cometRead()";

    QNetworkReply* reply = static_cast<QNetworkReply*>(this->sender());
    QString cometMessage;
    cometRawMessage += QString::fromUtf8(reply->readAll());
    QVariantMap cometMap;
    QVariantList cometObject;
    QByteArray cometChannel;
    QVariantMap cometData;

    // Iterate the received messages:
    if(cometRawMessage.endsWith(']')) {
        _DEBUG_ << "cometRead() Parsing...";// << cometRawMessage;
        cometMessageList=cometRawMessage.split("][{\"");
        while(!cometMessageList.isEmpty()) {
            cometMessage=cometMessageList.takeFirst();
            if(cometMessage.left(1)!="[")
                cometMessage="[{\""+cometMessage;
            if(cometMessage.right(1)!="]")
                cometMessage=cometMessage+"]";
            cometObject=Json::parse(cometMessage).toList();
            while(!cometObject.isEmpty()) {
                cometMap=cometObject.takeFirst().toMap();
                cometChannel=cometMap.value("channel").toString().toUtf8();
                cometData=cometMap.value("data").toMap();
                _DEBUG_ << "cometRead() Channel:" << cometChannel;
                //_DEBUG_ << "cometRead() CometData:" << cometData << "\n\n";
                cometParse(&cometChannel, cometData);
            }
        }
        cometRawMessage="";
    }
}

//----------------------------------------------
// SqueezsterCore::cometReadDelete()
// Comet request reply received.
//----------------------------------------------
void SqueezsterCore::cometReadDelete() {
    QNetworkReply* reply = static_cast<QNetworkReply*>(this->sender());
    //QJson::Parser parser;
    QByteArray replyBA = reply->readAll();
    QVariantMap replyMap;
    _DEBUG_ << "cometReadDelete()" << replyBA;
    reply->deleteLater();
    bool ok;
    QVariantList replyList = Json::parse(replyBA,ok).toList();
    if(!replyList.isEmpty()) {
        replyMap=replyList.first().toMap();
        if(!replyMap.value("successful").toBool()) {
            qWarning() << "SqueezsterCore::cometReadDelete() Request failed!";
            cometDisconnect(cometHttpAddress());
            QTimer::singleShot(0,this,SLOT(initEnvironment()));
            //                    cometInit();
        }
    }
}

//----------------------------------------------
// SqueezsterCore::cometReadDelete()
// Comet request reply received.
//----------------------------------------------
QNetworkReply* SqueezsterCore::cometRequest(QNetworkAccessManager* nam, QByteArray clientID, QByteArray channel, QByteArray command, QByteArray playerID, quint16 *idPtr)
{
    QNetworkRequest cometRequest(cometHttpAddress());
    cometRequest.setRawHeader("Connection","Keep-Alive");
    cometNetworkRequest.setRawHeader("CometID",QByteArray::number(cID));
    QByteArray cometMessage = cometRequestEncode(clientID,channel,command,playerID,idPtr);
    QNetworkReply* reply = nam->post(cometRequest,cometMessage);
    connect(reply,SIGNAL(readyRead()),this,SLOT(cometReadDelete()));
    connect(reply,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(networkErrorSlot(QNetworkReply::NetworkError)));
    return(reply);
}

//----------------------------------------------
// SqueezsterCore::cometRequestEncode()
// Encode a comet request. Returns the encoded QByteArray
//----------------------------------------------
QByteArray SqueezsterCore::cometRequestEncode(QByteArray clientID, QByteArray channel, QByteArray command, QByteArray playerID, quint16 *idPtr)
{
    _DEBUG_ << "cometRequestEncode" << command;
    //    QJson::Serializer serializer;
    QVariantList messageList, requestList;
    QVariantMap message, data;
    QByteArray response = QByteArray("/"+clientID);
    quint16 id;
    QStringList commandList=QString(command).split(" ");

    // Request channel depending on playerID (vs Server)
    if(playerID=="") {
        requestList << "" << commandList;
        response=response+"/server/0/"+commandList.join("/").toUtf8();
    } else
    {
        requestList << playerID << commandList;
        response=response+"/player/"+playerID+"/"+commandList.join("/").toUtf8();
    }
    data["request"]=requestList;
    data["response"]=response;

    message["channel"]=channel;
    message["data"]=data;
    message["ext"]=cometExtVM;
    if(idPtr) {
        id=*idPtr;
        message["id"]=id++;
    }
    messageList << message;
    QByteArray messageBA = Json::serialize(messageList);
    return(QUrl::fromPercentEncoding(messageBA).toUtf8());
}


//----------------------------------------------
// SqueezsterCore::cometTimeOutSlot()
//----------------------------------------------
void SqueezsterCore::cometTimeOutSlot() {
    _DEBUG_ << "cometTimeoutSlot() Check connectionsettings";
    envClear();
    //emit networkError();
    //cometInit();
    setCometTimeOutCount(cometTimeOutCount()+1);
    cometDisconnect(cometHttpAddress());
    QTimer::singleShot(0,this,SLOT(initEnvironment()));
}

//----------------------------------------------
// SqueezsterCore::jsonTimeOutSlot()
//----------------------------------------------
void SqueezsterCore::jsonTimeOutSlot() {
    qWarning() << "jsonTimeoutSlot() Check connectionsettings";
    //if(jsonReply)
    //    jsonReply->deleteLater();
    //envClear();
    //emit networkError();
    //setMailbox("Json request timeout.");
    setJsonTimeOutCount(jsonTimeOutCount()+1);
}

//-----------------------------------------------------------------
// SqueezsterCore::networkErrorSlot(QNetworkReply::NetworkError code)
//-----------------------------------------------------------------
void SqueezsterCore::networkErrorSlot(QNetworkReply::NetworkError code) {
    QNetworkReply* reply = static_cast<QNetworkReply*>(this->sender());

    qWarning() << "networkErrorSlot(QNetworkReply::NetworkError code)" << code << reply->url() << reply->errorString() << "\nCheck connection settings.";
    genericTimer.stop();
    //disconnect(&genericTimer,SIGNAL(timeout()),this,SLOT(jsonTimeOutSlot()));

    if(!connectionSettings().value("ipaddress").toString().isEmpty() ||
            connectionSettings().value("mysb").toBool()){
        if(!disconnectEnable && (code!=2)) // Dont emit error on "Connection closed"
        {
            _DEBUG_ << "Sending to mailbox.";
            emit networkError();
            mailMessage =QString("Network error. Please check your connection settings.\n"+reply->errorString());
            QTimer::singleShot(500,this,SLOT(setMailboxSlot())); // Make sure that QML part has initialized
        }
    } else
        emit networkError(); // Just open connection settings
}

//----------------------------------------------
// SqueezsterCore::cometParse
// Parse comet message
//----------------------------------------------
void SqueezsterCore::cometParse(QByteArray* cometChannel, const QVariantMap& cometData) {
    _DEBUG_ << "cometParse()";
    QList<QByteArray> commandList=QList<QByteArray>();
    bool items=false;
    bool treeChild=false;

    // Decode message:
    if(cometChannel->isEmpty()) {
        _DEBUG_ << "cometParse, no channel. Message:" << cometData;
    } else {
        commandList = cometChannel->split('/');
        if(!commandList.isEmpty())
            commandList.takeFirst(); // Remove first empty entry
        if((!commandList.isEmpty()) && ((commandList.first()=="slim") || (commandList.first()=="meta")))
            return;
        QByteArray messCometID = commandList.isEmpty()?"":commandList.takeFirst();
        QByteArray channel = commandList.isEmpty()?"":commandList.takeFirst();
        QByteArray playerID  = commandList.isEmpty()?"":commandList.takeFirst();
        QByteArray command = commandList.isEmpty()?"":commandList.takeFirst();
        QByteArray param1=commandList.isEmpty()?"":commandList.takeFirst();

        if(messCometID==cometID) {
            // Restart timer
            cometTimer->stop();
            cometTimer->start();
            if(param1=="items") {
                items=true;
                param1=commandList.takeFirst();
            }
            if(cometChannel->contains("item_id:"))
                treeChild=true;
            mapParse(cometData,command,param1,playerID,treeChild,items);
        } else
            _DEBUG_ << "Wrong cometID. Received:"<< messCometID << "Expected:" << cometID;
    }
}

//--------------------------------------------------
// void SqueezsterCore::mapParse(const QVariantMap map, QByteArray command, bool treeChild)
//--------------------------------------------------
void SqueezsterCore::mapParse(const QVariantMap& map, const QByteArray& command,
                              const QByteArray& param1, const QByteArray& playerID, bool treeChild, bool items)
{
    _DEBUG_ << "SqueezsterCore::mapParse()" << command << map;
    QByteArray playerID_i;
    bool activePlayerFound=false;

    if(command=="status") {
        if (param1=="-") {
            _DEBUG_ << "Status received for player" << map.value(P_NAME).toString();
            playersMap[playerID]=map;
            if(playerID==activePlayer()) {
                _DEBUG_ << "Calling UpdateUI()";// << map;
                setPlayerStatus(map);
                updateUI(); //map);
            }
        } else if (param1=="0") { // is a playlist status
            playersMap[playerID]=map;
            if(playerID==activePlayer()) {
                _DEBUG_ << "Playlist received for player" << map.value(P_NAME).toString();
                //if( map.contains(P_LOOP))
                setNowPlayingList(map.value(P_LOOP).toList());
            }
        }
    } else if(command=="serverstatus") {
        _DEBUG_ << "mapParse(): Serverstatus received." << "Playerslist:" << playersList();
        setServerStatus(map);
        bool playerInitialized=activePlayerSet && (validPlayerSet()); // Make sure no old player is subscribed on

        if(!playerListsEqual(playersList(),serverStatus()[S_P_LOOP].toList()))
        {
            playerCount=serverStatus()[S_P_COUNT].toString().toInt();
            _DEBUG_ << "mapParse(): Player count:" << playerCount;
            setOtherPlayersList(serverStatus()[S_O_P_LOOP].toList());
            playersList_i = serverStatus()[S_P_LOOP].toList(); // Do not send a signal yet
            for(int i=0;i<playerCount;i++) {
                playerID_i=playersList().at(i).toMap()[S_P_ID].toString().toUtf8();

                if(isValidPlayer(playerID_i)) { // Avoid SQUEEZENETWORK player FF:FF:FF:FF:FF
                    playersMap[playerID_i]=playersList().at(i).toMap();
                    _DEBUG_ << "mapParse(): Player ID:" << playerID_i << "index:" << i;
                    if(activePlayer()==playerID_i) {
                        setPlayersList(serverStatus()[S_P_LOOP].toList(),i);
                        //setActivePlayerIndex(i);
                        activePlayerFound=true;
                    }
                    if(!onMysb())
                        subscribeOnPlayer(playerID_i);
                }
            }
            // If the saved player isn't available, choose the last from the list
            if(!activePlayerFound){
                setActivePlayer(playerID_i);
                setPlayersList(serverStatus()[S_P_LOOP].toList(),0);
                //setActivePlayerIndex(playerCount-1);
            }
            _DEBUG_ << "Active player index:" << activePlayerIndex();
            playerStatusSlot();
            // Init lists first time after the player is set:
            if(!playerInitialized)
                initLists();
        }
    } else if(!map.value("_version").toString().isEmpty()) {
        _DEBUG_ << "Server version" << map.value("_version").toString();
        emit jsonServerVersion(map.value("_version").toString());
    } else if((command =="apps") && !treeChild) {
        _DEBUG_ << "mapParse(): Apps list received.";
        if( map.contains("count")) {
            // if( map.contains("appss_loop")) {
            QVariantList aList = map.value("appss_loop").toList();
            setAppsList(aList);
        }
    } else if((command =="radios") && !treeChild) {
        _DEBUG_ << "mapParse(): Radios list received.";
        if( map.contains("count")) {
            // if( map.contains("radioss_loop")) {
            QVariantList rList = map.value("radioss_loop").toList();
            setRadiosList(rList);
        }
    } else if(command=="playlists") {
        _DEBUG_ << "mapParse(): playlists list received.";
        if( map.contains("count")) {
            // if( map.contains("playlists_loop")) {
            QVariantList pList = map.value("playlists_loop").toList();
            setListCommand(command);
            setListList(pList);
        }
    } else if(command=="albums") {
        _DEBUG_ << "mapParse(): albums list received.";
        if( map.contains("count")) {
            if(!listSearch().isEmpty()) {
                QVariantList aList = map.value("albums_loop").toList();
                setListSearch("");
                setListCommand(command);
                setListList(aList);
            } else {
                //       if( map.contains("albums_loop")) {
                QVariantList aList = map.value("albums_loop").toList();
                setAlbumsList(aList);
            }
        }
    } else if(command=="players") {
    } else if(command=="playlistcontrol") {
    } else if((command=="favorites") && (param1=="exists")) {
        _DEBUG_ << "mapParse(): favorites exits received." << map.value("exists").toBool();
        setFavoriteIndex(map.value("index").toInt());
        setFavoriteExists(map.value("exists").toBool());
    } else if(command=="artists") {
        _DEBUG_ << "mapParse(): artists received." << command;
        if(map.contains("count")) {
            QVariantList tList = map.value("artists_loop").toList();
            setListCommand(command);
            setListList(tList);
        }
    } else if(command=="titles") {
        _DEBUG_ << "mapParse(): titles received." << command;
        if( map.contains("count")) {
            QVariantList tList = map.value("titles_loop").toList();
            setListCommand(command);
            setListList(tList);
        }
    } else if(command=="musicfolder") {
        _DEBUG_ << "mapParse(): musicfolder received." << command;
        if( map.contains("count")) {
            QVariantList tList = map.value("folder_loop").toList();
            setTreeTitle(map.value("title").toString());
            setTreeCommand(command);
            setTreeList(tList);
        }
    } else if(command=="songinfo") { // Add favorite song
        _DEBUG_ << "mapParse(): songinfo received." << command << favoriteName();
        if(!favoriteName().isEmpty() && map.contains("songinfo_loop")) // Add favorite
        {
            QVariantList list = map.value("songinfo_loop").toList();
            while(!list.isEmpty()) {
                QVariantMap songInfo=list.takeFirst().toMap();
                _DEBUG_ << "songInfo: " << songInfo;
                if(!songInfo.value("url").toString().isEmpty()) {
                    QString commandStr="favorites add url:";
                    commandStr+=songInfo.value("url").toString();
                    commandStr+=" title:"+QUrl::toPercentEncoding(favoriteName());
                    sendJson(commandStr.split(" "),activePlayer());
                    setFavoriteName("");
                }
            }
        }
    } else if((map.value("count").canConvert<int>()) && items) {
        // Should be a decodable tree message
        _DEBUG_ << "mapParse(): treeItems received. Count: " << map.value("count").toInt() << command << map;
        QVariantList tList=map.value("loop_loop").toList();
        setTreeTitle(map.value("title").toString());
        setTreeCommand(command);
        setTreeList(tList);
    }
    else {
        _DEBUG_ << "mapParse(): doing nothing:" << command << "count" << map.value("count").toString();
    }
}

//----------------------------------------------
// SqueezsterCore::playerStatusSlot()
// Need to fetch status manually on mysb.com
//----------------------------------------------
void SqueezsterCore::playerStatusSlot() {
    if(onMysb()) {
        if(validPlayerSet()) {
            sendJson(QString(QString("status - 1")+STATUSTAGS).split(" "),activePlayer()); }
        QTimer::singleShot(5000,this,SLOT(playerStatusSlot()));
    }
}

//----------------------------------------------
// SqueezsterCore::playerListsEqual
//----------------------------------------------
bool SqueezsterCore::playerListsEqual(QVariantList list1, QVariantList list2)
{
    bool equal=true;
    if (list1.count() != list2.count())
        equal = false;
    else {
        for(int i=0;i<list1.count();i++) {
            equal= (equal && (list1.at(i).toMap()[S_P_ID].toString() == list2.at(i).toMap()[S_P_ID].toString()));
        }
    }
    return equal;
}

//----------------------------------------------
// SqueezsterCore::updateUI()
// Update the UI to reflect the player status
//----------------------------------------------
void SqueezsterCore::updateUI() // const QVariantMap &statusMap)
{
    _DEBUG_ << "updateUI() Player" << playerStatus()[P_NAME].toString() << playerStatus()[P_LOOP].toList();
    if(!playerStatus()[P_LOOP].toList().isEmpty())
        setCurrentSong(playerStatus()[P_LOOP].toList().first().toMap());

    _DEBUG_ << "updateUI" << "playlistChanged:" << (playerStatus()[P_TIMESTAMP].toDouble() - playlistTimeStamp())  <<playerChanged();
    if(playlistChanged() || playerChanged()) {
        _DEBUG_ << "updateUI" << "playlistChanged! Updating";
        requestList("status",activePlayer());
        setPlayerChanged(false);
        setPlaylistTimeStamp(playerStatus()[P_TIMESTAMP].toDouble());
    }

    //---------------------------------------------
    // If valid album art, queue http request
    //---------------------------------------------
    _DEBUG_ << "Remote" << currentSong()[SONG_REMOTE] << "CoverID:"<< currentSong()[SONG_COVERID] << "CoverArt" << currentSong()[SONG_COVERART];
    QString newUrl=currentSong()[SONG_ARTWORK_URL].toString();

    if (!currentSong()[SONG_REMOTE].toBool()) { // Local
        newUrl=serverHttpAddress();
        newUrl+="/music/";
        newUrl+=currentSong()[SONG_COVERID].toString();
        newUrl+=LOCALALBUMARTSUFFIX;
    }
    else if(!newUrl.isEmpty()){ // Remote
        if(!newUrl.contains("http:")) {
            newUrl=QString(serverHttpAddress()+"/"+newUrl);
        }
    }

    if(newUrl!=albumartUrl()) {
        _DEBUG_ << "Setting albumart URL" << newUrl;
        setAlbumartUrl(newUrl);
    } else {
        _DEBUG_ << "Keeping albumart URL" << newUrl;
    }
}

//--------------------------------------------------
// bool SqueezsterCore::playlistChanged()
//--------------------------------------------------
bool SqueezsterCore::playlistChanged()
{
    //    bool changed=!((playerStatus_i[P_SONG_COUNT]==prevPlayerStatus[P_SONG_COUNT]) &&
    //                   (playerStatus_i[P_SONG_TITLE]==prevPlayerStatus[P_SONG_TITLE]) &&
    //                   (playerStatus_i[P_SHUFFLE]==prevPlayerStatus[P_SHUFFLE]) &&
    //                   (playerStatus_i[P_IP]==prevPlayerStatus[P_IP]) &&
    //                   (playerStatus_i[P_REMOTE]==prevPlayerStatus[P_REMOTE]) &&
    //                   (playerStatus_i["playlist mode")==prevPlayerStatus["playlist mode"]) &&
    //                   (playerStatus_i["playlist_id"]==prevPlayerStatus["playlist_id"]));

    bool changed=((playerStatus()[P_TIMESTAMP].toDouble() != playlistTimeStamp())|| playerChanged());
    prevPlayerStatus=playerStatus_i;
    return changed;

}

//--------------------------------------------------
// SqueezsterCore::requestList()
// Request an element list based on "command".
// quint16 count: Total number of available elements
// quint16 start: First element in list
// quint16 batch: Number of elements per request
// QString tags: Ask for certain info, "tags"
//---------------------------------------------------
void SqueezsterCore::requestList(const QString& command, QByteArray playerID, bool subscribe, quint32 count, quint32 start, quint32 batch, QByteArray tags) {
    _DEBUG_ << "requestList()" << command;
    quint16 modCount;
    QByteArray newTags;
    QByteArray commandBA =command.toUtf8();

    if (batch==0)
        modCount=count;
    else {
        if(count>batch)
            modCount=batch;
        else
            modCount=count;
    }

    if (tags.isEmpty()) {
        if (command.startsWith("albums"))
            newTags = ALBUMTAGS;
        else
            newTags = STATUSTAGS;
    }
    else
        newTags = tags;

    commandBA += QByteArray(" "+QByteArray::number(start)+" "+QByteArray::number(count-1)+" "+newTags);

    // Do not subscribe on high bandwidth lists
    // Do not subscribe on status. It will ruin the previous subscription
    if(!onMysb()) { // Use comet for local server
        if(command!="status" && command!="albums" && command!="titles" && command!="musicfolder" && !subscribe)
            initCometSubscription(commandBA,5*60,playerID);
        else // Subscribe on the rest. (Still not sure if it works)
            sendComet(commandBA,CometRequest,playerID);
    }
    else if(validPlayerSet()) // Use Json for mysb. Do nothing if no player is defined (gives error)
        sendJson(QString(commandBA).split(" "),activePlayer()); //playerID); // Do not use Comet for mysb.com
}

//--------------------------------------------------
// SqueezsterCore::sendComet()
// Ask the server to return list based on "command"
//--------------------------------------------------
void SqueezsterCore::sendComet(const QByteArray& message, CometType type, QByteArray playerID) {
    _DEBUG_ << "sendComet()" << message;
    if(!initDone) {
        _DEBUG_ << "Not initialized yet";
        return;
    }

    QByteArray channel;
    if(type==CometRequest)
        channel=COMET_REQUEST_CH;
    else // Subscription
        channel=COMET_SUBSCR_CH;
    if(!cometID.isEmpty()) {
        cIDStr="\"id\":\""+QByteArray::number(cID++)+"\"";
        QByteArray cometMessage=cometRequestEncode(cometID,channel,message,playerID,&cID);
        _DEBUG_ << "Comet Mess:" << cometMessage;
        cometNetworkRequest.setRawHeader("CometID",QByteArray::number(cID));
        cometReplyDelete = cometNetworkManager->post(cometNetworkRequest,cometMessage);
        connect(cometReplyDelete,SIGNAL(readyRead()),this,SLOT(cometReadDelete()));
        connect(cometReplyDelete,SIGNAL(error(QNetworkReply::NetworkError)),this,SLOT(networkErrorSlot(QNetworkReply::NetworkError)));
    } else {
        _DEBUG_ << "sendComet() CometID not initialized!";
        //cometInit();
        cometDisconnect(cometHttpAddress());
        QTimer::singleShot(0,this,SLOT(initEnvironment()));
    }
}

//--------------------------------------------------------------------------------
// initCometSubscription(QString command,quint16 period, QByteArray playerID="");
//--------------------------------------------------------------------------------
void SqueezsterCore::initCometSubscription(const QByteArray& command, quint16 period, QByteArray playerID) {
    _DEBUG_ << "initCometSubscription" << command;
    sendComet(QByteArray(command+" subscribe:"+QByteArray::number(period)),CometSubscription,playerID);
}

//--------------------------------------------------
// SqueezsterCore::sendQmlCometRequest()
// Sends a server request from the QML domain
//--------------------------------------------------
void SqueezsterCore::sendQmlCometRequest(const QString& command, QString id, QString search, QString type) {
    _DEBUG_ << "sendQmlCometRequest(QString command, QString id, QString type)" << command << id << search;
    QByteArray commandBA =command.toUtf8();
    if(command=="artists" || command=="titles" || command=="albums" || command=="playlists" ||
            command=="favorites items" || command=="musicfolder") {
        commandBA += +" "+QByteArray::number(0)+" "+QByteArray::number(MAX_ITEMS); // artist,album
        // Add search tag:
        if (!search.isEmpty()) {
            commandBA+=" search:"+search;
        }
        // Add command specific tags:
        if (command=="albums") {
            if (!id.isEmpty())
                commandBA+=" artist_id:"+QUrl::fromPercentEncoding(id.toUtf8());
            commandBA+=ALBUMTAGS;
        } else if (command=="titles") {
            if (!id.isEmpty())
                commandBA+=" sort:tracknum album_id:"+QUrl::fromPercentEncoding(id.toUtf8());
            commandBA+=TITLES_TAGS;
        } else if (command=="favorites items") {
            if(!id.isEmpty())
                commandBA+=" item_id:"+QUrl::fromPercentEncoding(id.toUtf8());
            commandBA+=" want_url:1";
        } else if(command=="musicfolder") {
            if(!id.isEmpty())
                commandBA+=" folder_id:"+QUrl::fromPercentEncoding(id.toUtf8());
            commandBA+=" want_url:1";
            commandBA+=FOLDER_TAGS;
        } else
            commandBA+=GENERAL_TAGS;
    }
    // XML browser commands (tree commands):
    else if(id=="xmlbrowser") {
        commandBA += QByteArray(" items "+QByteArray::number(0)+" "+QByteArray::number(MAX_ITEMS)+" want_url:1") ;
    } else if(id=="xmlbrowser_search") {
        commandBA += QByteArray(" items "+QByteArray::number(0)+" "+QByteArray::number(MAX_ITEMS)+" want_url:1 search:"+search.toUtf8());
    } else if(search.isEmpty()) {
        commandBA += QByteArray(" items "+QByteArray::number(0)+" "+QByteArray::number(MAX_ITEMS)+" want_url:1 item_id:"+id.toUtf8());
    } else {
        commandBA += QByteArray(" items "+QByteArray::number(0)+" "+QByteArray::number(MAX_ITEMS)+" want_url:1 item_id:"+id.toUtf8()+" search:"+search.toUtf8());
    }
    if(command=="favorites items" || command=="playlists")
        commandBA += " subscribe:300";
    // Send as Json to mysb, or Comet to local server:
    if(onMysb())
        sendJson(QString(commandBA).split(" "),activePlayer());
    else
        sendComet(commandBA,CometRequest,activePlayer());
}

//----------------------------------------------
// SqueezsterCore::writeConnectionSettings()
// Writes the user settings to the system
//----------------------------------------------
void SqueezsterCore::writeConnectionSettings()
{
    _DEBUG_ << "SqueezsterCore::writeConnectionSettings(const QVariantMap newSettings) " << connectionSettings();

    //squeezsterSettings->beginGroup("connection");
    squeezsterSettings->setValue("ipaddress", connectionSettings().value("ipaddress").toString());
    squeezsterSettings->setValue("httpport", connectionSettings().value("httpport").toInt());
    squeezsterSettings->setValue("username", connectionSettings().value("username").toString());
    squeezsterSettings->setValue("password", connectionSettings().value("password").toString());
    squeezsterSettings->setValue("mysb", onMysb());
    squeezsterSettings->setValue("mysbusername", connectionSettings().value("mysbusername").toString());
    squeezsterSettings->setValue("mysbpassword", connectionSettings().value("mysbpassword").toString());
    squeezsterSettings->setValue("wolmac", connectionSettings().value("wolmac").toString());
    squeezsterSettings->setValue("wolport", connectionSettings().value("wolport").toInt());
    //squeezsterSettings->endGroup();
}

//----------------------------------------------
// SqueezsterCore::writeOptionsSettings()
// Writes the user settings to the system
//----------------------------------------------
void SqueezsterCore::writeOptionsSettings()
{
    _DEBUG_ << "SqueezsterCore::writeOptionsSettings(const QVariantMap newSettings) " << optionsSettings() << activePlayer();
    squeezsterSettings->setValue("lastusedplayer", activePlayer());
    squeezsterSettings->setValue("screensaverinhibited", optionsSettings().value("screensaverinhibited").toBool());
    squeezsterSettings->setValue("albumicons",optionsSettings().value("largealbumicons").toBool());
    squeezsterSettings->setValue("largealbumicons", optionsSettings().value("largealbumicons").toBool());
    squeezsterSettings->setValue("libraryview", optionsSettings().value("libraryview").toString());
}

//----------------------------------------------------------
// SqueezsterCore::readSettings()
// Reads (from the system) or initializes the user settings
//----------------------------------------------------------
void SqueezsterCore::readSettings() // UserSettings &newSettings)
{
    qDebug() << "SqueezsterCore::readSettings(UserSettings &newSettings)";

    QVariantMap connectionSettings;
    QVariantMap optionsSettings;

    QByteArray newActivePlayer=squeezsterSettings->value("lastusedplayer","").toByteArray();
    qDebug() << "readSettings() Setting player: " << squeezsterSettings->value("lastusedplayer","").toByteArray();
    optionsSettings["screensaverinhibited"]=squeezsterSettings->value("screensaverinhibited",false).toBool();
    optionsSettings["albumicons"]=squeezsterSettings->value("albumicons",false).toBool();
    optionsSettings["largealbumicons"]=squeezsterSettings->value("largealbumicons",false).toBool();
    optionsSettings["libraryview"]=squeezsterSettings->value("libraryview","albums").toString();
    setOptionsSettings(optionsSettings);
    setActivePlayer(newActivePlayer);

    connectionSettings["ipaddress"]=squeezsterSettings->value("ipaddress","").toString();
    connectionSettings["httpport"]=squeezsterSettings->value("httpport",HTTPPORT).toUInt();
    connectionSettings["username"]=squeezsterSettings->value("username",QString("")).toString();
    connectionSettings["password"]=squeezsterSettings->value("password",QString("")).toString();
    connectionSettings["mysb"]=squeezsterSettings->value("mysb",true).toBool();
    connectionSettings["mysbusername"]=squeezsterSettings->value("mysbusername",QString("pakarlss@gmail.com")).toString();
    connectionSettings["mysbpassword"]=squeezsterSettings->value("mysbpassword",QString("2ec4u2")).toString();
    connectionSettings["wolmac"]=squeezsterSettings->value("wolmac","0000000000").toString();
    connectionSettings["wolport"]=squeezsterSettings->value("wolport","7").toInt();

    setConnectionSettings(connectionSettings);
}

//----------------------------------------------
// SqueezsterCore::setOptionsSettings(const QVariantMap& optionsSettings)
//----------------------------------------------
void SqueezsterCore::setOptionsSettings(const QVariantMap& optionsSettings)
{
    _DEBUG_ << "setOptionsSettings: " << optionsSettings;
    optionsSettings_i = optionsSettings;

#if defined(MEEGO_EDITION_HARMATTAN) || defined(Q_WS_MAEMO_5) || defined(Q_OS_SYMBIAN)
    //#if not defined(Q_OS_SYMBIAN)
    // Delete inhibiter and construct new screensaver
    if(screenSaverInhibiter) {
        delete screenSaverInhibiter;
    }

    if (optionsSettings_i.value("screensaverinhibited").toBool())
    {
        _DEBUG_ << "Enabling inhibiter for Screensaver";
        screenSaverInhibiter = new QtMobility::QSystemScreenSaver();
        screenSaverInhibiter->setScreenSaverInhibit();
    }
#endif
    emit optionsSettingsChanged();
}

//----------------------------------------------
// SqueezsterCore::serverDiscovery()
// Broadcast discovery UDP message.
//----------------------------------------------
void SqueezsterCore::serverDiscovery()
{
    _DEBUG_ << "Server Discovery";
    char dataChar[16] = {'e','I','P','A','D','\0','N','A','M','E','\0','J','S','O','N','\0'};
    clientUdpSocket->writeDatagram(dataChar,16,QHostAddress::Broadcast, DISCOVERYPORT_CS); //DISCOVERYPORT_CS);
}

//----------------------------------------------
// SqueezsterCore::receiveUdpDatagram()
// Receive and parse UDP messages from servers
//----------------------------------------------
void SqueezsterCore::receiveUdpDatagram()
{
    _DEBUG_ << "receiveUdpDatagram() Size:" << clientUdpSocket->pendingDatagramSize();

    QByteArray datagram;
    datagram.resize(256); //clientUdpSocket->pendingDatagramSize());
    QHostAddress sender;
    QVariantMap map;
    quint16 senderPort;
    QStringList stringlist;
    QString name;
    int port=9000;

    while(clientUdpSocket->hasPendingDatagrams())
    {
        int size = clientUdpSocket->pendingDatagramSize();
        clientUdpSocket->readDatagram(datagram.data(),size,&sender, &senderPort);
        datagram.data()[size]='\0';
        QVariantList list=serverList();
        if(senderPort==3483) {
            //            for(int i=0;i<size;i++)
            //                _DEBUG_ << "Datagram: " << datagram.data()[i];
            stringlist=QString(datagram).split("ENAME",QString::KeepEmptyParts,Qt::CaseInsensitive);
            if(stringlist.count() > 1 && stringlist.takeFirst().isEmpty()) {
                QStringList stringlist2=stringlist.takeFirst().split("JSON",QString::KeepEmptyParts,Qt::CaseInsensitive);
                if(!stringlist2.isEmpty()) {
                    name=stringlist2.takeFirst(); // Parse name
                    name=name.right(name.length()-1);
                    _DEBUG_ << "Name: " << name << stringlist2;
                    if(!stringlist2.isEmpty()) {
                        QString tmp=stringlist2.takeFirst();
                        tmp=tmp.right(tmp.length()-1);
                        bool ok;
                        if(tmp.toInt(&ok) && ok)
                            port=tmp.toInt(&ok);
                    }
                    map["name"]=name;
                    map["port"]=port;
                    map["ip"]=sender.toString();

                    // Add to list:
                    if(!list.contains(map)) {
                        list.append(map);
                        setServerList(list);
                    }
                }
            }
        }
    }
}

//----------------------------------------------
// Squeezster deconstructor
//----------------------------------------------
SqueezsterCore::~SqueezsterCore()
{
    disconnect(this,0,0,0);
    if(networkSession) {
        disconnect(networkSession,SIGNAL(closed()),this,SLOT(openNetworkSession()));
        networkSession->close();
        networkSession->deleteLater();
    }
    cometDisconnect(cometHttpAddress());
#if defined(MEEGO_EDITION_HARMATTAN) || defined(Q_WS_MAEMO_5) || defined(Q_OS_SYMBIAN)
    if (screenSaverInhibiter)
        delete screenSaverInhibiter;
#endif
    _DEBUG_ << "Exit main()";
}

