#ifndef SQUEEZSTER_H
#define SQUEEZSTER_H

#include <QMainWindow>
#include <QtNetwork>
#include <QMetaType>
#include <QtGui>
#include <algorithm>
#include <QHash>
#include <QList>
#include <QVariant>
#include <QUrl>
#include <QTimer>
#include <QStringList>
#include <QFontMetrics>
#include <QButtonGroup>
#include <QCheckBox>
#include <QUuid>
#include <qplatformdefs.h>
#include <QListWidgetItem>
#include <QTreeWidgetItem>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include "squeezster_icons.h"
#include "../squeezstercore.h"
#include "connectiondialog.h"
#include "optionsdialog.h"
#include "libraryviewdialog.h"
#include "woldialog.h"
#include "maptreeitem.h"
#include "maplistitem.h"
#include "pushbuttonnetworkrequest.h"
#include "listwidgetitemnetworkrequest.h"
#include "treewidgetitemnetworkrequest.h"
#include "metatypes.h"
#include "constants.h"

QTM_USE_NAMESPACE

namespace Ui {
class squeezster;
}

class squeezster : public QMainWindow
{
    Q_OBJECT

public:
    explicit squeezster(int debug, QWidget *parent = 0);
    ~squeezster();

signals:
    void pixmapDone();
    void initDone();
    void updateListUISignal(QVariantList paramsList, QVariantMap* resultMap);
    void updateTreeUISignal(QVariantList paramsList, QVariantMap* resultMap);
    void volumeUp();
    void volumeDown();

private:
    Ui::squeezster *ui;
    SqueezsterCore core;

    QPointer<connectionDialog> connectionUI;

    static const int mapItemCol = 0;
    static const int httpObjectVariant = 1024;

    enum mapListFlags {noUpdate,updateIcons};

    int debugFlag_i;

    QTimer progressTimer;
    QTimer itemSelectTimer;
    QListWidgetItem *itemPressed;
    QTreeWidgetItem *pressedTreeItem;
    quint16 screenMode;
    QVariant treeItemParent;
    QVariant listItemParent;
    QVariant itemSelected;

    bool loadAlbumsBusy;
    bool albumsInitDone;
    bool musicFolderInitDone;
    bool pixmapBusy;
    QStringList busyList;
    QVariantMap httpGetMap;
    QList<QVariant*> pixmapDLList;
    QList<QUrl*> pixmapUrl;

    QIcon bulletIcon,smallBulletIcon,plusIcon,minusIcon,powerIcon,playIcon,pauseIcon,stopIcon;
    QIcon rewIcon,fwdIcon,noteIcon;
    QIcon emptyAlbumArtIcon;
    QIcon largeEmptyAlbumArtIcon;
    QIcon smallEmptyAlbumArtIcon;
    QString albumartsuffix;

protected:
#ifdef Q_WS_MAEMO_5
    void keyPressEvent(QKeyEvent* event);
#endif

private slots:
    void setDebugFlag(int debugFlag) {debugFlag_i = debugFlag;}
    int debugFlag() const {return(debugFlag_i);}

    void coreRestart(const QVariantMap& connectionSettings);
    void requestList(const QString& command, QByteArray playerID);
    void sendCometRequest(const QString& command, const QString& id, const QString& search, const QString& type);
    void sendRequest(QString command);

    void nowPlayingListChangedSlot();
    void albumsListChangedSlot();
    void radiosListChangedSlot();
    void appsListChangedSlot();
    void playersListChangedSlot();
    void serverListChangedSlot();
    void activePlayerIndexChangedSlot();
    void listListChangedSlot();
    void treeListChangedSlot();

    void constructConnDialog();
    void initUI();
    void updateAlbumArtLabel();
    void updateNowPlayingIcons();
    void slotOptions();
    void slotMusicLibraryView();
    void downloadImage(MapTreeItem* item);
    void downloadImage(QPushButton* item, QString url);
    void downloadAlbumImage(MapListItem* item, QString id);
    void downloadAlbumImage(MapListItem* item);
    void pixmapDownloaded();

    void slotTabWidgetCurrentChanged(int currentIndex);
    void initAlbumsList();
    void initPlaylistsList();
    void initTitlesTree();
    void initArtistsTree();
    void showAlbums();
    void showTitles();
    void showArtists();
    void showPlaylists();

    void showMusicLibraryView();
    void setAlbumsListView();
    void treeContextMenuHandler(QPoint point);
    void listContextMenuHandler(QPoint point);

    void displayError(QAbstractSocket::SocketError socketError);
    void displayMessage();
    void setupScreenMode();
    void isBusy(const QString& busyString=QString());
    void updateStatus();
    void updateUI();
    void updateProgress();
    void playlistItemClicked(QListWidgetItem* song);

    void updateTreeUI(const QString &command, const QVariantList& resultList);
    void updateListUI(QString command, const QVariantList& resultList);
    void treeItemClicked(QTreeWidgetItem* treeItem);
    void listItemClicked(QListWidgetItem* listItem);
    void treeItemPlay(QString cmd, MapTreeItem* item);
    void listItemPlay(QString cmd, MapListItem* item);
    void treeItemAdd(QString cmd, MapTreeItem* item);
    void listItemAdd(QString cmd, MapListItem* item);

    QString treeItemUrl(MapTreeItem* item);
    QString favId(QString id);
    QString favParent(QString id);
    void treeItemAddFav(MapTreeItem *item);
    void treeItemDelFav(MapTreeItem *item);
    void treeItemRenameFav(MapTreeItem *item);
    //----------------------------------------------
    // Command slots:
    //----------------------------------------------
    void sendPlay() {_DEBUG_ << "sendPlay()";playerPlay=true;playerPause=false;playerStop=false;core.sendQmlRequest("play");updateUI();}
    void sendPause() {_DEBUG_ << "sendPause()";playerPlay=false;playerPause=true;playerStop=false;core.sendQmlRequest("pause");updateUI();}
    void sendStop() {_DEBUG_ << "sendStop()";playerPlay=false;playerPause=false;playerStop=true;core.sendQmlRequest("stop");updateUI();}
    void sendSkipBack() {_DEBUG_ << "sendBackward()";core.sendQmlRequest("playlist index -1");}
    void sendSkipForward() {_DEBUG_ << "sendSkipForward()";core.sendQmlRequest("playlist index +1");}
    void sendPlaylistIndex(quint16 index) {_DEBUG_ << "sendPlaylistIndex() ";core.sendQmlRequest("playlist index " + QString::number(index));}
    void sendPower() {_DEBUG_ << "sendPower()";playerPower=!playerPower;core.sendQmlRequest("power "+QString(core.playerPower()?"0":"1"));updateUI();}
    void sendPlaylistClear() {_DEBUG_ << "sendPlaylistClear()";core.sendQmlRequest("playlist clear");}
    void setVolume(int index) {quint16 roundedIndex = ((int)index/5)*5; if(core.playerVolume()!= roundedIndex)
        {core.sendQmlRequest("mixer volume "+QString::number(roundedIndex));}}
    void increaseVolume();
    void decreaseVolume();
    void setNewProgress(int index) {quint16 roundedIndex = ((int)index/5)*5;core.sendQmlRequest("time "+QString::number(roundedIndex));}

    void incrementScreenMode() {
        _DEBUG_ << "Screenmode" << screenMode;
        if(++screenMode>2)screenMode=0;setupScreenMode();}
    void slotShuffle(){core.sendQmlRequest("playlist shuffle "+QString(core.playerShuffle()?"0":"1"));}
    void slotRepeat(){core.sendQmlRequest("playlist repeat "+QString(core.playerRepeat()?"0":"2"));}
    void savePlaylist();
    void slotRescan() {core.sendQmlRequest("rescan");}
    void itemUnselect();

#ifdef Q_WS_MAEMO_5
    void grabZoomKeys(bool grab);
#endif

private:
    bool playerPower;
    bool playerPlay;
    bool playerPause;
    bool playerStop;
    int playerShuffle;
    int playerRepeat;
    QVariantMap currentSong;
};

#endif // SQUEEZSTER_H
