#include "mainwindow.h"

#include <QtCore/QCoreApplication>

#if defined(Q_OS_SYMBIAN) && defined(ORIENTATIONLOCK)
#include <eikenv.h>
#include <eikappui.h>
#include <aknenv.h>
#include <aknappui.h>
#endif // Q_OS_SYMBIAN && ORIENTATIONLOCK

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{    
    central = new QWidget(this);
    glay = new QGridLayout(central);

    webView = new QWebView(this);

    glay->addWidget(webView);
    central->setLayout(glay);
    setCentralWidget(central);

    // Init toolbar
    tb = new QToolBar(this);

    m_fullscreen = new QAction(QIcon::fromTheme("general_fullsize"), QString(), this);
    m_fullscreen->setCheckable(true);
    connect(m_fullscreen, SIGNAL(toggled(bool)), this, SLOT(toggleFullScreen(bool)));

    m_spotme = new QAction(QIcon::fromTheme("act_spotme", QIcon(QLatin1String(":/spotme.png"))), QString(), this);
    m_spotme->setCheckable(true);
    QObject::connect(m_spotme, SIGNAL(toggled(bool)), this, SLOT(on_m_spotme_toggled(bool)));

    m_info = new QAction(QIcon::fromTheme("act_info", QIcon(QLatin1String(":/info.png"))), QString(), this);
    m_info->setCheckable(true);
    connect(m_info, SIGNAL(toggled(bool)), this, SLOT(on_m_info_toggled(bool)));

    m_route = new QAction(QIcon::fromTheme("act_route", QIcon(QLatin1String(":/footPath.png"))), QString(), this);
    m_route->setCheckable(true);
    connect(m_route, SIGNAL(toggled(bool)), this, SLOT(on_m_route_clicked(bool)));

    m_configure = new QAction(QIcon::fromTheme("act_config", QIcon(QLatin1String(":/configure.png"))), QString(), this);
    connect(m_configure, SIGNAL(triggered()), this, SLOT(configure_toggled()));

    tb->addAction(m_fullscreen);
    tb->addSeparator();
    tb->addAction(m_spotme);
    tb->addSeparator();
    tb->addAction(m_info);
    tb->addSeparator();
    tb->addAction(m_route);
    tb->addSeparator();
    tb->addAction(m_configure);
    addToolBar(tb);


    // Init general variables and QNetworkConfiguration
    gpsUpdateInterval = 1000;
    hideTbTimeout = 4000;
    routeEnabled = false;
    TotalDistance = 0;
    activeConfigs = mgr.allConfigurations(QNetworkConfiguration::Active);
    qDebug() << "activeConfigs.count" << activeConfigs.count();

    // Load index.html to QWebView    
    webView->load(QUrl("qrc:/index.html"));

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(isInitialised()));
    timer->start(500);

    // Create  filter
    keyEventListener = new KeyEventListener(webView);
    QObject::connect(keyEventListener,SIGNAL(screenTouched(bool)),this,SLOT(screenSlot(bool)));

    // Install it into your application for catching events
    webView->installEventFilter(keyEventListener);


    //********************
    // Setup GPS
    //********************
    source = QGeoPositionInfoSource::createDefaultSource(this);
    if (source)
    {
        source->setUpdateInterval(gpsUpdateInterval); // time in milliseconds
        source->setPreferredPositioningMethods(QGeoPositionInfoSource::AllPositioningMethods);
    }

    QTimer::singleShot(5000, this, SLOT(hideToolBar()));
}


/* Config Dialog */
void MainWindow::configure_toggled()
{
    qDebug() << "configure_toggled";

    dlg = new QDialog(this);
    QGridLayout * lay = new QGridLayout(dlg);

    QLabel *label = new QLabel("GPS update interval (ms)");
    lay->addWidget(label,0,0);

    combo = new QComboBox();
    combo->setEditable(true);
    combo->lineEdit()->setReadOnly(true);
    combo->lineEdit()->setAlignment(Qt::AlignCenter);

    combo->addItem("500");
    combo->addItem("1000");
    combo->addItem("1500");
    combo->addItem("2000");

    combo->setEditText(QString::number(gpsUpdateInterval));
    connect(combo, SIGNAL(activated(int)),this, SLOT(changeUpdateInterval(int)));

    lay->addWidget(combo,0,1);

    dlg->setLayout(lay);
    dlg->show();
}


void MainWindow::changeUpdateInterval(int index)
{
    qDebug() << "currentText " << combo->currentText();
    bool ok;
    gpsUpdateInterval = combo->currentText().toInt(&ok,10);
    source->setUpdateInterval(gpsUpdateInterval);
}


void MainWindow::isInitialised()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QCoreApplication::processEvents();
    QVariant var = webView->page()->mainFrame()->evaluateJavaScript("isInitialised()");

    bool done = var.toBool();

    if(done)
    {
        timer->stop();
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    }
}


void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "closeEvent";
    event->accept();
    disable();
}


void MainWindow::hideToolBar()
{
    if(tb->isVisible())
        tb->setVisible(false);
}


void MainWindow::screenSlot(bool b)
{
    if(!tb->isVisible())
    {
        tb->setVisible(true);
        QTimer::singleShot(hideTbTimeout, this, SLOT(hideToolBar()));
    }
}
/*
void MainWindow::InitLOG()
{
    logfile.setFileName("spotme.txt");
    //out.setString(&logfile);
}

void MainWindow::LOG(QString txt)
{
    if(dir.exists("/media/mmc1/"))
        dir.setCurrent("/media/mmc1/");
    else
        dir.setCurrent("/home/user/");
}
*/
void MainWindow::positionUpdated(const QGeoPositionInfo &info)
{
    if (info.isValid())
    {
        qDebug() << ++updateCount << info;
        QGeoCoordinate co = info.coordinate();

        //try fixing location/signal
        if( isNewPositionOK( co ) )
        {
            qDebug() << "newCooordinateCount:" << ++newCooordinateCount;
            consecutiveNonOK = 0;

            sLastLat = QString::number(dLastLat);
            sLastLng = QString::number(dLastLng);

            //source->stopUpdates();
            updateGoogleMap();
            //source->startUpdates();
        }
        else
        {
            qDebug() << "isNewPositionOK" << false;
            /*
            if(++consecutiveNonOK > 10)
            {
                qDebug() << "consecutiveNonOK" << consecutiveNonOK;
                QMessageBox::information(this, tr("GPS Error"),
                    tr("Invalid/Weak GPS signal timeout. SpotMe will switch off."),
                    QMessageBox::Ok);
                on_m_spotme_toggled(false);
            }
            */
        }
    }
    else {
//        QMessageBox::information(this, tr("GPS Error"),
//                tr("Invalid GPS details. Possible cause weak GPS signal."),
//                QMessageBox::Ok);
    }
}


/*==================================================================
  If certain conditions are not met the new positon will be ignored.
  Because N900 gives spurious values if signal is weak.
  ==================================================================*/
bool MainWindow::isNewPositionOK(QGeoCoordinate newCoord)
{
    bool b = false;

    //if(dLastLat == 0 && dLastLng == 0)
    //    return true;
    // Give time for fixing

    double lat = newCoord.latitude();
    double lng = newCoord.longitude();

    // Check if it is new position (moving)
    if( (lat == dLastLat) && (lng == dLastLng) && (newCooordinateCount > 0) )
        return false;

    //qDebug() << "dLastLat" << dLastLat << "dLastLng" << dLastLng;
    QGeoCoordinate lastCood(dLastLat,dLastLng);
    qreal distance = newCoord.distanceTo(lastCood); //in meters
    qDebug() << "distance" << distance;

    dLastLat = lat;
    dLastLng = lng;

    /*------------------------------------
     Cond1: Distance
        If we assume max speed is 100 Km/h
        then no way to do more than 27.7 m/s
     -------------------------------------*/
    if( (distance > 27) and (newCooordinateCount > 0) )
    {
        if(updateCount > 10)
            return false;
    }
    else
    {
        TotalDistance += distance;
        qDebug() << "TotalDistance" << TotalDistance;
    }

    return true;
}


void MainWindow::updateRoute()
{
    webView->page()->mainFrame()->evaluateJavaScript("showRoute(\""+sLastLat+","+sLastLng+"\")");
}


/* route */
void MainWindow::on_m_route_clicked(bool checked)
{
    if(checked)
    {
        webView->page()->mainFrame()->evaluateJavaScript("delclrPolyCoords()");
        routeEnabled = true; TotalDistance = 0;
    }
    else
        routeEnabled = false;
}


/* info */
void MainWindow::on_m_info_toggled(bool checked)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QCoreApplication::processEvents();
    //qDebug()<< "LastLat" << LastLat;
    //qDebug()<< "LastLng" << LastLng;
    //qDebug()<< "activeConfigs.count()" << activeConfigs.count();
    //qDebug()<< "mgr.isOnline()" << mgr.isOnline();

    if(checked)
    {
        if( !sLastLat.isEmpty() && !sLastLng.isEmpty() && (activeConfigs.count() > 0) && mgr.isOnline() )
            webView->page()->mainFrame()->evaluateJavaScript("getInfo(\""+sLastLat+","+sLastLng+"\")");
        else
            m_info->setChecked(false);
    }
    else
        webView->page()->mainFrame()->evaluateJavaScript("closeInfoWindow()");

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}


void MainWindow::updateGoogleMap()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QCoreApplication::processEvents();

    qDebug() << "updateGoogleMap";
    webView->page()->mainFrame()->evaluateJavaScript("place_marker("+sLastLat+","+sLastLng+")");

    if(routeEnabled)
        updateRoute();

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}


/* start spotme button */
void MainWindow::on_m_spotme_toggled(bool checked)
{
    if (activeConfigs.count() > 0)
        Q_ASSERT(mgr.isOnline());   //Prints a warning message containing the source code file name and line number if test is false.
    else
    {
        Q_ASSERT(!mgr.isOnline());
        if(!mgr.isOnline())
        {
            QMessageBox::information(this, tr("Error"),
                    tr("Can not detect internet connection.\nMake sure you connect to the internet before proceeding."),
                    QMessageBox::Ok);
        }
    }

    if(checked)
    {
        enable();
        qDebug() << "GPS enabled";
    } else {
        on_m_info_toggled(false);
        m_info->setChecked(false);
        m_route->setCheckable(false);
        disable();
        qDebug() << "GPS disabled";
    }
}


void MainWindow::toggleFullScreen(bool b)
{
    setWindowState(b ? windowState() | Qt::WindowFullScreen
                     : windowState() & ~Qt::WindowFullScreen);

    if(b)
    {
        //tb->setVisible(false);
        QTimer::singleShot(hideTbTimeout, this, SLOT(hideToolBar()));
    }
    else
        tb->setVisible(true);
}


QString MainWindow::getCoordinates(QGeoPositionInfo info)
{
    QGeoCoordinate co = info.coordinate();
    QString lat = QString::number(co.latitude());
    QString lng = QString::number(co.longitude());
    return (lat + ","+ lng);
}


void MainWindow::disable()
{
    if (source)
    {
        source->stopUpdates();
        disconnect(source, SIGNAL(positionUpdated(QGeoPositionInfo)),
                   this, SLOT(positionUpdated(QGeoPositionInfo)));
    }
}


void MainWindow::enable()
{
    if (source)
    {
        connect(source, SIGNAL(positionUpdated(QGeoPositionInfo)),
                this, SLOT(positionUpdated(QGeoPositionInfo)));
        source->startUpdates();

        // Reset variables
        dLastLat = 0; dLastLng = 0;
        sLastLat.clear(); sLastLng.clear();
        updateCount = 0;
        newCooordinateCount = 0;
        consecutiveNonOK = 0;
    }
}


MainWindow::~MainWindow()
{
}


void MainWindow::setOrientation(ScreenOrientation orientation)
{
#ifdef Q_OS_SYMBIAN
    if (orientation != ScreenOrientationAuto) {
#if defined(ORIENTATIONLOCK)
        const CAknAppUiBase::TAppUiOrientation uiOrientation =
                (orientation == ScreenOrientationLockPortrait) ? CAknAppUi::EAppUiOrientationPortrait
                    : CAknAppUi::EAppUiOrientationLandscape;
        CAknAppUi* appUi = dynamic_cast<CAknAppUi*> (CEikonEnv::Static()->AppUi());
        TRAPD(error,
            if (appUi)
                appUi->SetOrientationL(uiOrientation);
        );
        Q_UNUSED(error)
#else // ORIENTATIONLOCK
        qWarning("'ORIENTATIONLOCK' needs to be defined on Symbian when locking the orientation.");
#endif // ORIENTATIONLOCK
    }
#elif defined(Q_WS_MAEMO_5)
    Qt::WidgetAttribute attribute;
    switch (orientation) {
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_Maemo5PortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_Maemo5LandscapeOrientation;
        break;
    case ScreenOrientationAuto:
    default:
        attribute = Qt::WA_Maemo5AutoOrientation;
        break;
    }
    setAttribute(attribute, true);
#else // Q_OS_SYMBIAN
    Q_UNUSED(orientation);
#endif // Q_OS_SYMBIAN
}

void MainWindow::showExpanded()
{
#ifdef Q_OS_SYMBIAN
    showFullScreen();
#elif defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
    showMaximized();
#else
    show();
#endif
}
