/**
 * Copyright 2010 Arjan Scherpenisse <arjan@scherpenisse.net>
 * See LICENSE for details.
 */

using GLib;
using Gtk;
using Hildon;
using Soup;
using Thread;
using Pango;

class Speedy : Hildon.Program
{
    const string downloadUrl = "http://news.bbc.co.uk/nol/shared/spl/hi/uk/08/broad_band_britain/swf/download.jpg";

    private Hildon.Window window;
    private Hildon.Button button;

    private Gtk.Label  speedLabel;
    private Gtk.Label  helpLabel;
    private Gtk.ProgressBar bar;

    private Timer timer;

    Soup.Session session;
    bool started = false;
    int64 contentLength = 0;
    int64 bytesReceived = 0;


    construct
    {
        window = new Hildon.Window ();
        add_window (window);
        window.destroy += Gtk.main_quit;

        Environment.set_application_name ("Speedy");

        var box = new Gtk.VBox(false, 1);
        box.set_border_width(10);

        speedLabel = new Label("0.00 KB/s");
        var d = new Pango.FontDescription();
        d.set_size(48000);
        speedLabel.modify_font(d);
        box.pack_start(speedLabel, true, true, 0);

        bar = new ProgressBar();
        box.pack_start(bar, false, true, 10);

        helpLabel = new Label("Click the button to start measuring.");
        box.pack_start(helpLabel, false, false, 10);

        button = new Hildon.Button(SizeType.THUMB_HEIGHT, ButtonArrangement.VERTICAL);
        button.set_title("Start");
        button.clicked.connect((source) => toggleDownload());

        box.pack_start(button, false, false, 10);

        timer = new Timer();

        window.add (box);
    }


    void toggleDownload()
    {

        if (!started)
        {
            started = true;
            button.set_title("Stop");
            helpLabel.set_label("Downloading, hold on...");
            doDownload();
        }
        else
        {
            started = false;
            button.set_title("Start");
            helpLabel.set_label("Click the start button to measure again.");

            if (session != null) {
                session.abort();
            }
            timer.stop();
            session = null;
        }
    }


    void doDownload()
    {
        session = new Soup.SessionAsync ();
        var message = new Soup.Message ("GET", downloadUrl);

        message.got_headers.connect((msg) => {
                timer.start();
                contentLength = msg.response_headers.get_content_length();
            });
        message.got_chunk.connect(gotChunk);

        contentLength = 0;
        bytesReceived = 0;
        bar.set_fraction(0.0);

        /* send a sync request */
        session.send_message (message);
    }


    void gotChunk(Soup.Message msg, Soup.Buffer buf)
    {
        bytesReceived += buf.length;
        bar.set_fraction(bytesReceived/(float)contentLength);

        speedLabel.set_label("%.2f KB/s".printf( (bytesReceived/1024)/timer.elapsed()));

        if (bytesReceived >= contentLength) {
            toggleDownload();
        }
    }


    void run()
    {
        window.show_all();
        Gtk.main ();
    }


    static int main(string[] args)
    {
        Gtk.init (ref args);
        if (!Thread.supported ()) {
            error ("Cannot run without thread support.");
        }


        (new Speedy()).run();
        return 0;
    }
}
