# -*- coding: utf-8 -*-
##############################################################
# smscon_service - remote control daemon for Nokia N900 phone#
# $Id: smscon_service.py 78 2011-12-23 23:12:29Z yablacky $
##############################################################

# standard modules
import os
import sys
import time
import re
import random
import signal
import dbus
from dbus.mainloop.glib import DBusGMainLoop

# for GPS control
import location
import gobject
from operator import itemgetter 

# for email sending
import smtplib
import socket
import urllib
from email.MIMEText import MIMEText
from email.MIMEImage import MIMEImage
from email.MIMEMultipart import MIMEMultipart

import smsconlib as smscon
from smsconlib import InstPath, LogPath, PhotoName, MapName, AudioFile, DaemonFile 
from smsconlib import IF, IsHostReachable, IsInternetReachable, IsSamePhoneNumber, IsFileReadable, DeleteFile, shellArg, pidof, SendSMSviaGSMmodem

LOGGER = smscon.LOGGER = smscon.StartLogging('DAEMON', withConsole = False)

#############################################
#    constants

PingCountMax                = 3                                 # max number of pings for connection status check
LastKnownGoodConfigFile     = '.%s_lkg' % smscon.NAME

#############################################
#    variables

GpsCoordinateList           = []
GpsIsActive                 = False                             # true if GPS device is enabled
GpsIsTest                   = False                             # true if GPS device is enabled from test mode command
GpsDataOK                   = False                             # true if GPS device receives valid coordinates
GpsStartTime                = 0

ConnectionType              = None 
ConnectionIAP               = None
ConnectionName              = None
ConnectionBehaviour         = None

IsKeyboardSliderCheckActive = False
IsBatteryStatusCheckActive  = False
IsCheckHostActive           = False

LastBatteryStatusMessage    = None

CheckHostLastCommand        = '-1'      # "restart"
CheckHostStatusCount        = 0         # positive: successes, negative: errors
CheckHostStatusTimestamp    = time.gmtime()

smscnf = None               # the UserConfiguration actually used by the daemon.

#############################################

def GetTimeString(*fromTime):
    """
    @return string Current time in custom format.
    """
    return time.strftime(smscnf.LOG_TIMEFORMAT, *fromTime)

#############################################

def RevealThat(Subject, Text, SendMethod = None, Attachment = None, Attachhint = None):
    """
    Send message to all enabled targets. In case SMS is enabled, the
    message is send to the originating SMS SENDERNUMBER as well as to
    the SMS MASTERNUMBER if set and is a different number.
    The device hostname and a timestamp is added to the message automatically.
    @param string Subject Short description of event.
    @param string|list Text Message body.
    @param string SendMethod Defaults to smscnf.MESSAGESEND
    @param string|none Attachment Name of file to attach (to email only)
    @param string|none Attachhint Hint about the attachment file's MIME type. 
    @return void
    """
    if  SendMethod == None:
        SendMethod = smscnf.MESSAGESEND
        
    if type(Text).__name__ == 'list': Text = ''.join(Text)
    Hostname = os.uname()[1]
    Revealed = []
    if SendMethod in ['both', 'sms']:
        Message = 'N900[%s]\n' '%s\n' '-%s-\n' % (Hostname, GetTimeString(), Subject) + Text

        # Split Message into list of lines to give SMSsend hint where to split 
        TextLines = []
        while Message != '':
            Line, Sep, Message = Message.partition('\n');
            TextLines.append(Line + Sep);

        # Send to the last known SMS sender
        if SMSsend(smscnf.SENDERNUMBER, TextLines, 'send SMS message (subject=%s) to {RECIPIENTNUMBER}' % Subject):
            Revealed.append(smscnf.SENDERNUMBER)

        # Send to the master if that is different 
        if not IsSamePhoneNumber(smscnf.MASTERNUMBER, smscnf.SENDERNUMBER):
            if Revealed: 
                TextLines.append('\nBC:%s' % smscnf.SENDERNUMBER)   # "blind copy" information
            if SMSsend(smscnf.MASTERNUMBER, TextLines, 'send SMS message (subject=%s) to {RECIPIENTNUMBER}' % Subject):
                Revealed.append("MASTERNUMBER") # do not expose the encrypted number this way!

    if SendMethod in ['both', 'email']:
        if Revealed:
            Text += '\nSMS copies have been sent to: (' + ', '.join(Revealed) + ')' 

        if EMAILsend(EmailTo     = smscnf.EMAILADDRESS,
                  Subject        = 'NOKIA N900[%s]: %s' % (Hostname, Subject),
                  Text           = 'Nokia phone "%s" at %s:\n' '%s\n' % (Hostname, GetTimeString(), Text),
                  SuccessLogMsg  = 'send email message (subject=%s) to <%s>' % (Subject, smscnf.EMAILADDRESS),
                  Attachment = Attachment,
                  Attachhint = Attachhint):
            Revealed.append(smscnf.EMAILADDRESS)

    if not Revealed:
        LOGGER.info('Message sending disabled by user setting (%s).' % SendMethod)

#############################################
#    SMS
#############################################

def OnSmsReceived(PDUmessage, MessageCenter, SomeString, SenderNumber):
    """
    Receive SMS and try to execute as command.
    @return void
    """
    n = int(PDUmessage[1])
    n = n // 2 + n % 2 + 13

    MessageArray = PDUmessage[n:len(PDUmessage)]
    try:
        MessageText = DeOctifyMessage(MessageArray)     # decode SMS message to plain text
    except Exception, e:
        LOGGER.critical("DeOctifyMessage problem: %s", e)
        LOGGER.critical('SMS Message Bytes:' + ', '.join( ["0x%02X" % int(b) for b in MessageArray] ))
        return
    LOGGER.debug('SMS Message Chars:' + ', '.join( ["0x%02X[%s]" % (ord(b), b) for b in MessageText] ))
    LOGGER.debug("SMS received:'%s'" % MessageText)

    if (    MessageText.startswith(smscnf.SMS_COMPREFIX) and
            MessageText.  endswith(smscnf.SMS_COMSUFFIX)  ):
        ProcessCommand(MessageText[len(smscnf.SMS_COMPREFIX) : len(MessageText)-len(smscnf.SMS_COMSUFFIX)],
                       SenderPhoneNumber = SenderNumber)

def SMSsend(RecipientNumber, MessageList, SuccessLogMsg):
    """
    Send SMS message to RecipientNumber.
    @return bool False if the message has definitely not been sent. True otherwise.
    """
    if not RecipientNumber:
        return False
    MessageLengthMax = 160  # max. number of characters in one SMS message
    MessageText = ''.join(MessageList)

    if len(MessageText) <= MessageLengthMax:
        if SendToGSMmodem(RecipientNumber, MessageText, SuccessLogMsg):
            return True
        if smscnf.ENABLERESEND != 'yes':
            return False
        LOGGER.warning('Trying to re-send SMS message after %s minutes.' % float(smscnf.RESENDTIME))
        tryCount = { 'value': 0 }
        gobject.timeout_add(int(float(smscnf.RESENDTIME)) * 60000,
                            lambda args: not SendToGSMmodem(**args), dict(
                                RecipientNumber   = RecipientNumber,
                                Message           = MessageText,
                                SuccessLogMsg     = SuccessLogMsg,
                                TryCount          = tryCount,
                            ))
        return True

    # Message must be split
    if 0:
        # Fast and simple split but may require less SMS than sophisticated split:
        part = 0
        while len(MessageText) > 0:
            part += 1
            MessageText = ('-#%s-\n' % part) + MessageText
            SMSsend(RecipientNumber, [MessageText[0:MessageLengthMax]], SuccessLogMsg + (' [part %d]' % part))
            MessageText = MessageText[MessageLengthMax:]
        return True

    # Sophisticated split:
    # Calculates n of m part header and tries not to split within messages.
    PartHeader = lambda n,m: '#%d/%d\n' % (n,m)
    def SplitMessages(MessageList, MessageLengthMax, AssumeTotalPartCount):
        """
        Join the messages in the given list and split that to parts of MessageLengthMax.
        If possible to not split within a single message.
        @return list of messages each of which is no longer than MessageLengthMax chars.
        """
        SplittedMessages = []
        MessageParts = []
        MessageList = MessageList[:]
        while MessageList:
            Msg = MessageList.pop(0)
            if len(Msg) == 0:
                continue
            if len(MessageParts) == 0:
                MessageParts = [ PartHeader(len(SplittedMessages)+1, AssumeTotalPartCount) ]
                if len(MessageParts[0]) >= MessageLengthMax:
                    MessageParts[0] = '' # no header if header alone exceeds max length.
            if sum(map(len,MessageParts)) + len(Msg) <= MessageLengthMax:
                MessageParts.append(Msg)
                continue
            lenNow = 0
            if len(MessageParts) == 1:  # No change, must split within message.
                lenNow = MessageLengthMax - len(MessageParts[0])
                MessageParts.append(Msg[0:lenNow])
            MessageList = [Msg[lenNow:]] + MessageList     # unshift
            SplittedMessages.append(''.join(MessageParts))
            MessageParts = []
        if len(MessageParts):
            SplittedMessages.append(''.join(MessageParts))
        assert(max(map(len, SplittedMessages)) <= MessageLengthMax)
        return SplittedMessages

    # Since each message part has a header that contains the part number and the total number of
    # parts, splitting is a little complicated because the total number is not known in advance: 
    assumeTotalParts, totalParts = 0, 1
    while totalParts > assumeTotalParts:
        assumeTotalParts = (assumeTotalParts + 1) * 10 - 1    # 9, 99, 999, ...
        SplittedMessages = SplitMessages(MessageList, MessageLengthMax, assumeTotalParts)
        totalParts = len(SplittedMessages)
    if totalParts < assumeTotalParts:
        SplittedMessages = SplitMessages(MessageList, MessageLengthMax, totalParts)
        while len(SplittedMessages) < totalParts:
            SplittedMessages.append(PartHeader(len(SplittedMessages)+1, totalParts))
            
    LOGGER.info('SMS message is too long (%s chars); splitting message into %d parts.' % (len(MessageText), totalParts))

    if int(smscnf.SMS_PARTS_LIMIT) > 0 and totalParts > int(smscnf.SMS_PARTS_LIMIT):
        skipCount = totalParts - int(smscnf.SMS_PARTS_LIMIT)
        skipPart = 0
        while skipPart < skipCount:
            SplittedMessages[1+skipPart] = '' # at least 1 part is allowed, leave part 0 ("header info") unskipped
            skipPart += 1
        lenAvail = MessageLengthMax - len(SplittedMessages[0])
        if lenAvail > 0:
            SplittedMessages[0] += ("[%d/%d cut]" % (skipCount, totalParts))[:lenAvail]

    for part, msg in enumerate(SplittedMessages):
        if len(msg):
            SMSsend(RecipientNumber, msg, SuccessLogMsg + ' [part %d/%d, %d chars]' % (part+1, totalParts, len(msg)))
        else:
            LOGGER.warning(SuccessLogMsg + ' [part %d/%d, %d chars SKIPPED DUE TO SMS LIMIT]' % (part+1, totalParts, len(msg)))
        
    return True

def SendToGSMmodem(RecipientNumber, Message, SuccessLogMsg, TryCount = None):
    """
    Send SMS message to GSM modem.
    @return bool Success indicator
    """
    RetryInfo = ''
    if TryCount:
        TryCount['value'] += 1
        if int(smscnf.RESENDMAXTRY) > 0 and TryCount['value'] > int(smscnf.RESENDMAXTRY):
            LOGGER.error('(SMS) Tried to send %d times. Limit %d reached. Giving up recipient=%s.' %
                            (TryCount['value'], int(smscnf.RESENDMAXTRY), RecipientNumber))
            return True
        RetryInfo = " (%d. retry)" % TryCount['value']

    if SendSMSviaGSMmodem(RecipientNumber, Message):
        LOGGER.info(SuccessLogMsg.replace('{RECIPIENTNUMBER}', RecipientNumber))
        return True
    else:
        LOGGER.error('Failed to send SMS to "%s"%s.' % (RecipientNumber, RetryInfo))
        return False

def DeOctifyMessage(inputBytes):
    """
    Decode inputBytes from SMS message according to GSM 03.38, GSM 7 bit default alphabet.
    @return string.
    """
    outputCodes = []
    bitCount, lastByte = 7, 0
    for thisByte in inputBytes:
        outputCodes.append(0x7f & ((thisByte << (7 - bitCount)) | lastByte))
        lastByte = thisByte >> bitCount
        bitCount -= 1   # from the next byte, we need 1 bit less.
        if bitCount == 0:
            outputCodes.append(lastByte)    
            bitCount, lastByte = 7, 0

    # Note: for code pages below it is important that all code points evaluate to a unicode string (u-prefix):
    ESC = u'\x1b'
    GSM_03_38_codepage_std = (
        u'@' , u'£' , u'$' , u'¥' , u'è' , u'é' , u'ù' , u'ì' , u'ò' , u'Ç' , u'\n', u'Ø' , u'ø' , u'\r', u'Å' , u'å' ,
        u'Δ' , u'_' , u'Φ' , u'Γ' , u'Λ' , u'Ω' , u'Π' , u'Ψ' , u'Σ' , u'Θ' , u'Ξ' ,  ESC , u'Æ' , u'æ' , u'ß' , u'É' ,
        u' ' , u'!' , u'"' , u'#' , u'¤' , u'%' , u'&' , u'\'', u'(' , u')' , u'*' , u'+' , u',' , u'-' , u'.' , u'/' ,
        u'0' , u'1' , u'2' , u'3' , u'4' , u'5' , u'6' , u'7' , u'8' , u'9' , u':' , u';' , u'<' , u'=' , u'>' , u'?' ,
        u'¡' , u'A' , u'B' , u'C' , u'D' , u'E' , u'F' , u'G' , u'H' , u'I' , u'J' , u'K' , u'L' , u'M' , u'N' , u'O' ,
        u'P' , u'Q' , u'R' , u'S' , u'T' , u'U' , u'V' , u'W' , u'X' , u'Y' , u'Z' , u'Ä' , u'Ö' , u'Ñ' , u'Ü' , u'§' ,
        u'¿' , u'a' , u'b' , u'c' , u'd' , u'e' , u'f' , u'g' , u'h' , u'i' , u'j' , u'k' , u'l' , u'm' , u'n' , u'o' ,
        u'p' , u'q' , u'r' , u's' , u't' , u'u' , u'v' , u'w' , u'x' , u'y' , u'z' , u'ä' , u'ö' , u'ñ' , u'ü' , u'à' ,
    )

    GSM_03_38_codepage_esc = {
        0x0a: u'\f' , 
        0x14: u'^'  ,   0x1b: ESC   ,
        0x28: u'{'  ,   0x29: u'}'  ,   0x2f: u'\\' ,
        0x3c: u'['  ,   0x3d: u'~'  ,   0x3e: u']'  ,
        0x40: u'|'  ,
        0x65: u'€'  ,
    }

    outputText = []
    while outputCodes:
        code = outputCodes.pop(0)
        if code != ord(ESC) or not outputCodes:
            outputText.append(GSM_03_38_codepage_std[code])
        else:
            code = outputCodes.pop(0)
            if code in GSM_03_38_codepage_esc:
                outputText.append(GSM_03_38_codepage_esc[code])
            else:
                outputText.extend((ESC, GSM_03_38_codepage_std[code]))
    LOGGER.debug("DeOctifyMessage: have outputText: %s" % repr(outputText))
    return ''.join(outputText)

#############################################
#   send email
#############################################

def EMAILsend(EmailTo, Subject, Text, SuccessLogMsg, **kwargs):
    """
    Send email; retry after <RESENDTIME> minutes if send failure.
    @return bool False if the message has definitely not been sent. True otherwise.
    """
    if SendToSMTPserver(EmailTo, Subject, Text, SuccessLogMsg, **kwargs):
        return True
    if smscnf.ENABLERESEND != 'yes' or not EmailTo:
        return False
    LOGGER.warning('Trying to re-send email message after %s minutes.' % float(smscnf.RESENDTIME))
    tryCount = { 'value': 0 }
    gobject.timeout_add(int(float(smscnf.RESENDTIME)) * 60000,
                        lambda args: not SendToSMTPserver(**args), dict(
                            EmailTo       = EmailTo,
                            Subject       = Subject,
                            Text          = Text,
                            SuccessLogMsg = SuccessLogMsg,
                            TryCount      = tryCount,
                            **kwargs
                        ))
    return True

def SendToSMTPserver(EmailTo, Subject, Text, SuccessLogMsg, Attachment = None, Attachhint = None, TryCount = None):
    """
    Send email to SMTP server.
    @return bool Success indicator
    """
    RetryInfo = ''
    if TryCount:
        TryCount['value'] += 1
        if int(smscnf.RESENDMAXTRY) > 0 and TryCount['value'] > int(smscnf.RESENDMAXTRY):
            LOGGER.error('(EMAIL) Tried to send %d times. Limit %d reached. Giving up recipient=%s, subject=%s.' %
                            (TryCount['value'], int(smscnf.RESENDMAXTRY), EmailTo, Subject))
            return True
        RetryInfo = " (%d. retry)" % TryCount['value']

    if not EmailTo:
        LOGGER.error('(EMAIL) No recipient email address%s.' % RetryInfo)
        return False
    # convert string list to string
    if type(Text).__name__ == 'list': Text = ''.join(Text)

    # poor convertion of text to html text:
    MailBodyHtml = Text.replace('\n', '<br>')
    Hostname     = os.uname()[1]
    Message      = MIMEMultipart()

    if Attachment:
        Attachhint = Attachhint or 'text'
        ConvertToMIME         = MIMEText
        AttachmentCID         = None

        if Attachhint.startswith('image'):
            Message               = MIMEMultipart('related') 
            ConvertToMIME         = MIMEImage
            AltText               = '%s image' % os.path.basename(Attachment)
            AttachmentCID         = 'image%s' % random.randint(1, 100000)
            MailBodyHtml += '<br><img src="cid:%s" alt="%s" /><br>' % (AttachmentCID, AltText)

    Message['Subject'] = Subject
    Message['To']      = EmailTo
    Message['From']    = 'NokiaN900.SMSCON.%s <SMSCON@%s>' % (Hostname, Hostname)

    MailBodyHtml = '<html><head></head><body>\n'\
                    '<p>' + MailBodyHtml + '</p>\n'\
                    '</body></html>'
    Message.attach( MIMEText(MailBodyHtml, 'html') )

    if Attachment:
        try:
            File = open(Attachment, 'rb')
        except:
            LOGGER.error('(EMAIL) Attachment file "%s" not found.' % Attachment)
            AttachmentData = MIMEText('Attachment file "%s" not found' % Attachment)
            AttachmentCID = None
        else:
            try:
                AttachmentData = ConvertToMIME( File.read() )
            except:
                LOGGER.error('(EMAIL) Attachment file "%s" could not be read.' % Attachment)
                File.close()
                AttachmentData = MIMEText('Attachment file "%s" could not be read' % Attachment)
                AttachmentCID = None
            else:
                File.close()
        if AttachmentCID:
            AttachmentData.add_header('Content-ID', '<%s>' % AttachmentCID )
        else:
            AttachmentData.add_header('Content-Disposition', 'attachment; filename="%s"' % os.path.basename(Attachment))    
        Message.attach(AttachmentData)

    try:
        LOGGER.info('(EMAIL) Trying to connect to "%s:%s" (%s).' % ( smscnf.MAILSERVER, smscnf.MAILPORT, GetTimeString() ) )

        # connect to mail server
        Server = smtplib.SMTP(smscnf.MAILSERVER, smscnf.MAILPORT) 
        #Server.set_debuglevel(True)
    except (socket.error, smtplib.SMTPConnectError, smtplib.SMTPException):
        LOGGER.error('(EMAIL) Mailserver connection failure%s.' % RetryInfo)
        return False

    # identify to mail server
    Server.ehlo() 
        
    # if smtp server requires secure authentication
    if Server.has_extn('STARTTLS'):
        Server.starttls()
        Server.ehlo()

    # if smtp server requires user/password authentication
    if smscnf.USER != '' and smscnf.PASSWORD != '':
        try:
            LOGGER.info('(EMAIL) Trying to login to "%s".' % smscnf.MAILSERVER)
            Server.login(smscnf.USER, smscnf.PASSWORD) 
        except smtplib.SMTPAuthenticationError:
            LOGGER.error('(EMAIL) Wrong username/password%s.' % RetryInfo)
            Server.quit()
            return False

    try:
        LOGGER.info('(EMAIL) Trying to send email to "%s".' % smscnf.MAILSERVER)
        Server.sendmail(smscnf.EMAILFROM, EmailTo, Message.as_string()) 
    except smtplib.SMTPException, e:
        LOGGER.error('(EMAIL) [%s].' % e)
        Server.quit()
        LOGGER.error('(EMAIL) Mailserver connection closed%s.' % RetryInfo)
        return False

    LOGGER.info(SuccessLogMsg)
    LOGGER.info('(EMAIL) Mailserver connection closed successful.')
    Server.quit()
    return True

#############################################
#   save new number
#############################################

def SaveSenderPhoneNumber(PhoneNumber):
    """
    Save new SENDERNUMBER as smscnf.SENDERNUMBER in "smscon_config" file.
    """ 
    if smscnf.SENDERNUMBER == PhoneNumber:
        return

    if smscnf.update('SENDERNUMBER', PhoneNumber):
        LOGGER.info( 'New SMS SENDERNUMBER (%s) saved.' % PhoneNumber)
    else:
        LOGGER.error("Failed to save new SMS SENDERNUMBER (%s)." % PhoneNumber)

    # update smscnf.SENDERNUMBER in any case! We use it to send replies. 
    smscnf.SENDERNUMBER = PhoneNumber

#############################################
#    command processing
#############################################

def ProcessCommand(Command, SenderPhoneNumber=None, CommandOrigin=None, ExtraInfo='', LogIfCmdUnknown=False):
    """
    Process the received command message.
    """
    # Check if Command is a specific valid SMSCON command:
    CommandList = [smscnf[cmd] for cmd in smscnf.getSmsCommandList() if smscnf[cmd] != ""]
    if not Command in CommandList:
        if LogIfCmdUnknown:
            LOGGER.error( '"%s" was not a valid %s command.' % (Command, smscon.NAME) )
            LogBatteryChargeInfo()
        return False

    if SenderPhoneNumber:
        SaveSenderPhoneNumber(SenderPhoneNumber)
        CommandOrigin = CommandOrigin or "SMS"
        ExtraInfo = 'from %s (%s) %s' % (SenderPhoneNumber,
                                        IF(IsSamePhoneNumber(SenderPhoneNumber, smscnf.MASTERNUMBER), 'the master', 'NOT master'),
                                        ExtraInfo)
    else:
        CommandOrigin = str(CommandOrigin)

    LOGGER.info('Received %s command [%s] via %s %s.' % (smscon.NAME, Command, CommandOrigin, ExtraInfo))
    LogBatteryChargeInfo()

    # First of all, make phone silent
    EnableDeviceSilence(smscnf.SILENCEDEVICE == 'yes')

    # Command reply
    if smscnf.COMMANDREPLY == 'yes':
        RevealThat('reply', 'Command (%s) accepted.' % Command)
    else:
        LOGGER.info('Command (%s) accepted. Acknowledge message disabled.' % Command)

    # Explicit device locking
    if Command == smscnf.COM_UNLOCK:
        PerformDeviceUnlock()
        return

    if Command == smscnf.COM_LOCK:
        PerformDeviceLock()
        return

    # Automatic device locking
    if smscnf.AUTODEVICELOCK == 'yes':
        PerformDeviceLock()

    # Dispatch on other commands
    if Command == smscnf.COM_CHECK:
        ok = False
        if smscnf.MESSAGESEND in ['sms', 'both']:
            ok = ShowSMScommands('sms')
        if smscnf.MESSAGESEND in ['email', 'both']:
            ok = ShowSMScommands('email')
        if not ok:
            LOGGER.warning('No notification sent. Message sending is disabled by user setting: %s.' % smscnf.MESSAGESEND)

    elif Command == smscnf.COM_REBOOT:
        PerformReboot()

    elif Command == smscnf.COM_POWEROFF:
        PerformPoweroff()            

    elif Command == smscnf.COM_POWER:
        RevealThat('battery', GetBatteryChargeInfo())

    elif Command == smscnf.COM_LOCATION:
        behaviourChanged = False
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            behaviourChanged = SetAutoConnectBehaviour('GPRS') # disable switching to other network

        EnableGPStracking('single')

        if behaviourChanged:
            SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_REMOTEON:
        if smscnf.REMOTEUSER == '' and smscnf.REMOTEPASSWORD == '':
            LOGGER.warning('No user/password entered for enabling SSH connection.')
        else:
            if not IsInternetReachable():
                SetNetworkConnectionType('GPRS') # enable GPRS data network
                SetAutoConnectBehaviour('GPRS') # disable switching to other network
            EnableSSHcontrol()
            
    elif Command == smscnf.COM_REMOTEOFF:
        if smscnf.REMOTEUSER == '' and smscnf.REMOTEPASSWORD == '':
            LOGGER.warning('No user/password entered for disabling SSH connection.')
        else:
            DisableSSHcontrol()
            
    elif Command == smscnf.COM_CAMERA:
        behaviourChanged = False
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            behaviourChanged = SetAutoConnectBehaviour('GPRS') # disable switching to other network

        PerformCamera()

        if behaviourChanged:
            SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_CALL:
        PerformPhonecall(SenderPhoneNumber or smscnf.MASTERNUMBER)

    elif Command == smscnf.COM_TRACKON:
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            SetAutoConnectBehaviour('GPRS') # disable switching to other network
        EnableGPStracking('multi')

    elif Command == smscnf.COM_TRACKOFF:
        DisableGPStracking()
        SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_CUSTOM:
        PerformShellScript()

    elif Command == smscnf.COM_CUSTOMLOG:
        PerformShellScriptResult()

    elif Command == smscnf.COM_LOG:
        RevealThat('log', 'The smscon Log file', Attachment = LOGGER.LogFilename, Attachhint = 'text/plain')

    elif Command == smscnf.COM_SHELL:
        PerformDirectShellCommand(Command)

    elif Command == smscnf.COM_ALARM:
        PerformAlarm()

    elif Command == smscnf.COM_RESTART:
        RestartDaemon()

    else:
        LOGGER.debug('Unknown command: [%s].' % Command)

#############################################
#   play sound file as an alarm
#############################################

def PerformAlarm():
    """
    Make phone play sound file for alarm.
    """

    AudioFilename = os.path.join(InstPath, AudioFile)
    if not IsFileReadable(AudioFilename):
        LOGGER.error('(ALARM) Audiofile does not exist or is not readable: %s' % AudioFilename)
        RevealThat('alarm', 'Audiofile does not exist or is not readable.')        
        return

    PrePlayCommand = []
    PostPlayCommand = []
    
    # Before dealing with phone profiles and playing our alarm sound,
    # wait a little until the sounds are done that the phone will
    # play on receiving the sms message (the COM_ALARM message).... 
    PrePlayCommand.append("sleep 10")

    # get current active phone profile
    Output = os.popen('dbus-send --print-reply \
                      --session --dest=com.nokia.profiled \
                      --type=method_call \
                      /com/nokia/profiled \
                      com.nokia.profiled.get_profile').read()
    E = re.compile('string "(\S+)"')
    try:
        CurrentProfile = E.findall(Output)[0]
    except Exception, e:
        CurrentProfile = 'general'
        LOGGER.warning('(ALARM) Could not get current phone profile: %s' % e)

    SetPhoneProfileCommand = 'dbus-send --type=method_call' \
                              ' --dest=com.nokia.profiled' \
                              ' /com/nokia/profiled' \
                              ' com.nokia.profiled.set_profile string:%s'

    LOGGER.info('(ALARM) current profile is "%s".' % CurrentProfile)
    if CurrentProfile != 'general':
        LOGGER.info('(ALARM) Set phone profile temporarily to "General" for alarm.')
        PrePlayCommand.append(SetPhoneProfileCommand % shellArg('general'))
        RestoreProfileCommand.append(SetPhoneProfileCommand % shellArg(CurrentProfile))

    # set volume to maximum
    Volume = 100
    PrePlayCommand.append('dbus-send --type=method_call' \
               ' --dest=com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer' \
               ' /com/nokia/mafw/renderer/gstrenderer' \
               ' com.nokia.mafw.extension.set_extension_property' \
               ' string:volume variant:uint32:%s' % shellArg(Volume))
    
    # play audio
    ok = PlayAudio(AudioFilename,
                PrePlayCommand  = ';'.join(PrePlayCommand),
                PostPlayCommand = ';'.join(PostPlayCommand))
    RevealThat('alarm', IF(ok, 'Alarm sound activated.', 'Alarm sound activation failed.'))        

def PlayAudio(AudioFilename, PrePlayCommand = None, PostPlayCommand = None):
    """
    Play audio file in background.
    @return bool Success indicator.
    """ 
    PrePlayCommand  = PrePlayCommand  or 'true'
    PostPlayCommand = PostPlayCommand or 'true'
    try:
        os.system( '(%s; /usr/bin/aplay -q %s; %s)&' % (
            PrePlayCommand, shellArg(AudioFilename), PostPlayCommand) )
    except Exception, e:
        LOGGER.error('(ALARM) Failed to play audio file: %s' % e)
        return False
    LOGGER.info('(ALARM) Started playing audio file.')
    return True

#############################################
#   silence phone
#############################################

def EnableDeviceSilence(Enable = True):
    """
    Make the phone silent for email, instant-messaging, phone ringing & SMS.
    """    

    if not Enable:
        LOGGER.warning('(SILENCE) Silencing is disabled: Phone probably not silent while controlled remotely.')
        return

    KeyList = [('email.alert.volume'  , '0'),
               ('im.alert.volume'     , '0'),
               ('ringing.alert.volume', '0'),
               ('sms.alert.volume'    , '0')]

    # get current active phone profile
    Output = os.popen('dbus-send --print-reply \
                      --session --dest=com.nokia.profiled \
                      --type=method_call \
                      /com/nokia/profiled \
                      com.nokia.profiled.get_profile').read()
    E = re.compile('string "(\S+)"')
    try:
        CurrentProfile = E.findall(Output)[0]
    except Exception, e:
        LOGGER.error('(SILENCE) Could not get current phone profile: %s' % e)
        return
    else:
        LOGGER.info('(SILENCE) Current profile is "%s".' % CurrentProfile)

    # set items in current profile
    for Key, Value in KeyList:
        try:
            LOGGER.debug( '(SILENCE) Set current profile "%s" = "%s".' % (Key, Value) )
            os.system( 'dbus-send --session --dest=com.nokia.profiled \
                        --type=method_call \
                        /com/nokia/profiled \
                        com.nokia.profiled.set_value string:"%s" string:"%s" string:"%s"' % (CurrentProfile, Key, Value) )
        except:
            LOGGER.error( '(SILENCE) Could not set profile: %s = "%s".' % (Key, Value) )
            return
        
    LOGGER.info('(SILENCE) Phone is silenced in current "%s" profile.' % CurrentProfile)

#############################################
#   run custom scripts
#############################################

def PerformShellScript():
    """
    Execute user editable script.
    @return void
    """
    Filename = os.path.join(InstPath, smscon.ScriptFile)
    Logname  = os.path.join(InstPath, smscon.ScriptLog)
    # Delete old log (may be retrieved before by COM_CUSTOMLOG command)
    if pidof(Filename) == '':
        DeleteFile(Logname)

    if IsFileReadable(Filename):
        # run script file
        os.system( '"%s" 1>>"%s" 2>&1&' % (Filename, Logname) )
        LOGGER.info(         'Script file (%s) executes in BG.' % Filename)
        RevealThat('script', 'script file (%s) executes in BG.' % Filename)        
    else:
        LOGGER.error(        'script file (%s) not found.' % Filename)
        RevealThat('script', 'script file (%s) not found.' % Filename)        
        DeleteFile(Logname)

def PerformShellScriptResult():
    """
    Send back logfile of user editable script execution.
    @return void
    """
    Filename = os.path.join(InstPath, smscon.ScriptFile)
    Logname  = os.path.join(InstPath, smscon.ScriptLog)
    PIDs = pidof(Filename)
    if not IsFileReadable(Logname):
        RevealThat('script.log', 'Log file %s not present. Script processes running: (%s).' % (Logname, PIDs))
    else:
        RevealThat('script.log', 'Log file %s is attached. Script processes running: (%s).' % (Logname, PIDs),
                   Attachment = Logname, Attachhint = 'text/plain')

def PerformDirectShellCommand(Command):
    """
    Run direct shell command and reply with notification message containing the command's output.
    @return void.
    """
    try:
        Result = os.popen(Command).read()
    except:
        LOGGER.Error('Failed to run shell command (%s).' % Command)
        RevealThat('shell', ['command failed: %s.' % Command])        
    else:
        LOGGER.info('Shell command (%s) executed.' % Command)
        RevealThat('shell', ['command: %s\n' % Command, 'result:\n', Result])        

#############################################
#   show SMS commands
#############################################

def ShowSMScommands(SendMethod):
    """
    Log or SMS list of SMS commands.
    @return: bool Success status.
    """
    CommandVars = sorted(smscnf.getSmsCommandList())
    if SendMethod == 'sms':
        Format = '%s=%s\n'
        skip = len(smscnf.SmsCommandPreFix)  # Skip the command prefix to keep SMS message as short as possible
    else:
        Format = '%' + str(-smscnf.TabSize) + 's = %s\n'
        skip = 0
    Lines = []
    Lines.append(IF(len(smscnf.SMS_COMPREFIX), 'Prefix:%s\n' % smscnf.SMS_COMPREFIX, 'No prefix.\n'))
    Lines.append(IF(len(smscnf.SMS_COMSUFFIX), 'Suffix:%s\n' % smscnf.SMS_COMSUFFIX, 'No suffix.\n'))
    Lines.extend([Format % (CmdVar[skip:], smscnf[CmdVar]) for CmdVar in CommandVars])
    if SendMethod == 'log':
        for Line in Lines: 
            LOGGER.info(Line)
    else:
        RevealThat('commands', Lines, SendMethod = SendMethod)
    return True

#############################################
#   device lock
#############################################

def PerformDeviceLock():
    """
    Lock the phone.
    @return void
    """
    os.system ('run-standalone.sh \
               dbus-send --system --type=method_call \
               --dest=com.nokia.system_ui \
               /com/nokia/system_ui/request \
               com.nokia.system_ui.request.devlock_open \
               string:"com.nokia.mce" \
               string:"/com/nokia/mce/request" \
               string:"com.nokia.mce.request" \
               string:"devlock_callback" \
               uint32:"3"')

    LOGGER.info(             'Phone has been locked.')
    if smscnf.COMMANDREPLY == 'yes':
        RevealThat('locked', 'Phone has been locked.')
        
def PerformDeviceUnlock():
    """
    Unlock the phone.
    @return void
    """
    os.system ('run-standalone.sh \
               dbus-send --system --type=method_call \
               --dest=com.nokia.system_ui \
               /com/nokia/system_ui/request \
               com.nokia.system_ui.request.devlock_close \
               string:"com.nokia.mce" \
               string:"/com/nokia/mce/request" \
               string:"com.nokia.mce.request" \
               string:"devlock_callback" \
               uint32:"0"')

    LOGGER.info(         'Phone has been unlocked.')
    RevealThat('unlock', 'Phone has been unlocked.')

#############################################
#   system
#############################################

def PerformReboot():
    """
    Reboot the phone.
    """
    # disable resending of messages (would fail anyway when rebooting) 
    smscnf.ENABLERESEND = 'no' 
    DelaySeconds = 30

    LOGGER.info('Executing "%s" command: rebooting in %d seconds.' % (smscnf.COM_REBOOT, DelaySeconds))
    RevealThat('reboot',       'Phone is rebooting in %d seconds.' % DelaySeconds)

    time.sleep(DelaySeconds)
    os.system('reboot')

def PerformPoweroff():
    """
    Shutdown phone (switch off).
    """
    # disable resending of messages (would fail anyway when rebooting) 
    smscnf.ENABLERESEND = 'no' 
    DelaySeconds = 30 

    LOGGER.info('Executing "%s" command: power off in %d seconds.' % (smscnf.COM_POWEROFF, DelaySeconds))
    RevealThat('shutdown',   'Phone will power off in %d seconds.' % DelaySeconds)
    
    time.sleep(DelaySeconds)
    os.system('poweroff')

#############################################
#   camera
#############################################

def PerformCamera():
    """
    Get picture of front camera and send it to email address.
    """
    ImageFilename = TakeFrontCamPicture()
    if ImageFilename:
        Hostname = os.uname()[1]
        EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                  Subject =  'From "%s" command / N900[%s]' % (smscnf.COM_CAMERA, Hostname),
                  Text    =  ['Frontcam picture from Nokia N900 phone:\n',
                              'Taken on %s at %s\n' % (Hostname, GetTimeString())],
                  SuccessLogMsg  = '(EMAIL) camera picture send to "%s"' % smscnf.EMAILADDRESS,
                  Attachment =  ImageFilename,
                  Attachhint = 'image' )

def TakeFrontCamPicture():
    """
    Acquire front camera picture.
    @return string|bool Filename where image has been stored. False on errors.
    """
    ImageFilename       = os.path.join(LogPath, PhotoName)
    ImageFileEncoding   = 'jpegenc'     # TODO: encoding should depend on desired image type (png, jpeg, ...) 
    CamDevice           = '/dev/video1'
    DelaySeconds        = 1
    GStreamerPipeline   = (
        "v4l2camsrc device='%s' num-buffers=16" % CamDevice,
        "video/x-raw-yuv,width=640,height=480",
        "ffmpegcolorspace",
        "gamma gamma=1.8",
        "videobalance hue=0.1 brightness=0.2 saturation=1 contrast=1",
        ImageFileEncoding,
        "filesink location='%s'" % ImageFilename,
    )

    try:    os.remove(ImageFilename)
    except: pass

    if os.WEXITSTATUS(os.system('gst-launch ' + ' \! '.join(GStreamerPipeline))) != 0:
        LOGGER.error('Camera picture failed (image capture error).')
        return False

    time.sleep(DelaySeconds)        # i guess this this not really required... but to be on save side...

    if not os.path.isfile(ImageFilename):
        LOGGER.error('Taking camera picture failed (image file not found).')
        return False
    LOGGER.info('Camera picture acquired.')
    return ImageFilename

#############################################
#    phone call
#############################################

def PerformPhonecall(PhoneNumber):
    """
    Make phone call to PhoneNumber.
    @return void.
    """
    try:
        os.system('run-standalone.sh dbus-send --system --dest=com.nokia.csd.Call \
                   --type=method_call /com/nokia/csd/call com.nokia.csd.Call.CreateWith \
                   string:"%s" \
                   uint32:0' % PhoneNumber )        
        LOGGER.info(         'Making phonecall to "%s".' % PhoneNumber)          
    except:
        LOGGER.error('Failed to make phonecall to "%s".' % PhoneNumber)

#############################################
#    data connection
#############################################

def GetNetworkConnectionType(): 
    """
    Get info of current data connection type
    @return string WLAN|GPRS|NONE
    """
    Wlan = os.popen('ifconfig | grep "wlan0"').read()
    Gprs = os.popen('ifconfig | grep "gprs0"').read() 

    if   Wlan != '' and Gprs == '':     Type = 'WLAN'
    elif Wlan == '' and Gprs != '':     Type = 'GPRS'
    else:                               Type = 'NONE' 
    return Type

def GetGPRSConnectionStatus():
    """
    Get status of GPRS data network.
    @return: mixed True|False|None
    """
    Output = os.popen('dbus-send --print-reply --system --dest=com.nokia.csd.GPRS /com/nokia/csd/gprs com.nokia.csd.GPRS.GetStatus').read()

    E = re.compile(' boolean (\S+)')
    R = E.findall(Output)
    if R:
        if R[0] == 'false': return False
        if R[0]  == 'true': return True
    return None

def SetNetworkConnectionType(NewType):
    """
    Connect to GPRS data network.
    @param string Network type: GPRS|WLAN
    @return bool Success indicator
    """
    DelaySeconds = 3  # in sec. between connection status checks
    MaxT  = 30 # in sec. before timeout
    MaxR  = 3  # number of retries

    Type = GetNetworkConnectionType()

    if Type != 'NONE':
        if Type == NewType:
            if Type == 'GPRS':
                Type, IAP, Name = GetGPRSname()
            elif Type == 'WLAN':
                Type, IAP, Name = GetWLANname()
            else:
                LOGGER.error('Unknown network type "%s".' % Type)
                Name = 'unknown'
            LOGGER.info( '%s network (%s) already connected.' % (Type, Name) )
            return True 
        DisableConnection()

    if   NewType == 'GPRS': Type, IAP, Name = GetGPRSname()
    elif NewType == 'WLAN': Type, IAP, Name = GetWLANname()
    else:                   Type, IAP, Name = 'NONE', 'NONE', 'NONE'

    if Type == 'NONE':
        LOGGER.error('No network available.')
        return False

    # initiate network connecting
    LOGGER.info( 'Connecting to %s network (%s).' % (Type, Name) )
    os.system( 'dbus-send --system --type=method_call --dest=com.nokia.icd /com/nokia/icd com.nokia.icd.connect string:"%s" uint32:0' % IAP )

    Connected = IF(Type == 'GPRS', GetGPRSConnectionStatus, lambda:GetNetworkConnectionType() == Type)
    T, R = 0, 0
    while not Connected():
        if T >= MaxT:
            R += 1
            if R >= MaxR:
                LOGGER.error('Failed to connect to %s network after %s attempts.' % (Type, R))
                return False
            LOGGER.warning('No connection to %s network after %s sec. Retrying...' % (Type, T))
        time.sleep(DelaySeconds)
        T += DelaySeconds

    LOGGER.info( '%s network (%s) connection successful.' % (Type, Name) )
    return True

def DisableConnection(): 
    """
    Disconnect any current data connection.
    """
    DelaySeconds = 1

    os.system('dbus-send --system --dest=com.nokia.icd2 /com/nokia/icd2 com.nokia.icd2.disconnect_req uint32:0x8000')
    LOGGER.info('Current network disconnected.')

    time.sleep(DelaySeconds)

def GetGPRSname(): 
    """
    Get name of GPRS data connection from current inserted SIM card in Nokia phone.
    @return string tuple (Type, IAP, Name). Each of which is string 'NONE' if not found.
    """
    NetworkList = GetNetworks()
    CurrentIMSI = smscon.GetPhoneCode('IMSI')

    for Network in NetworkList:
        Type, IAP, Name, Imsi = Network[0:4]
        if Type == 'GPRS' and Imsi == CurrentIMSI:
            LOGGER.debug('original IAP=[%s]' % IAP)
            return Type, IAP.replace('@32@', ' '), Name

    return 'NONE', 'NONE', 'NONE'

def GetWLANname():
    """
    Get name of active WLAN data connection.
    @return string tuple (Type, IAP, Name). Each of which is string 'NONE' if not found.
    """
    NetworkList = GetNetworks()

    # get last used data network IAP
    LastUsedNetwork = os.popen('gconftool -g /system/osso/connectivity/IAP/last_used_network').read().strip('\n')
    LOGGER.debug('LastUsedNetwork = [%s].' % LastUsedNetwork)

    for Network in NetworkList:
        Type, IAP, Name = Network[0:3]
        if Type == 'WLAN' and LastUsedNetwork == IAP:
            LOGGER.debug('WLAN: %s' % Name)
            return Type, IAP, Name

    return 'NONE', 'NONE', 'NONE'

def GetNetworks():
    """
    Get list of all available data connections (stored under "Internet connections" in menu).
    @return list of (Type, IAP, Name [, Imsi, NameAP]) tuples.
    """
    Output = os.popen('gconftool -R /system/osso/connectivity/IAP | grep IAP').read()
    E = re.compile('/system/osso/connectivity/IAP/(\S+):')
    IAPlist = E.findall(Output) 

    DiscardList = ['wap', 'mms']
    NetworkList = []

    for IAP in IAPlist:
        # get connection type
        Type = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/type' % IAP).read().replace('_INFRA', '').strip('\n')

        if Type == 'GPRS':
            # get gprs_accespointname
            NameAP = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/gprs_accesspointname' % IAP).read().strip('\n')

            # discard unwanted connections
            if not [Discard for Discard in DiscardList if NameAP.find(Discard) >= 0]:
                # get connection name & it's IMSI number
                Name = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/name' % IAP).read().strip('\n')
                Imsi = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/sim_imsi' % IAP).read().strip('\n')

                # convert decimal value of ASCII character in IAP string to normal ASCII character.
                E = re.compile('@(\d+)@')
                DecList = E.findall(IAP)

                for Dec in DecList:
                    IAP = IAP.replace( '@%s@' % Dec, chr( int(Dec) ) )

                # add GPRS connection to list
                LOGGER.debug( 'Type=%s, IAP=%s, Name=%s, IMSI=%s, AP=%s' % (Type, IAP, Name, Imsi, NameAP) )
                NetworkList.append( (Type, IAP, Name, Imsi, NameAP) )
            
        elif Type == 'WLAN':
            Name = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/name' % IAP).read().strip('\n')
            # add WLAN connection to list
            LOGGER.debug( 'Type=%s, IAP=%s, Name=%s' % (Type, IAP, Name) )
            NetworkList.append( (Type, IAP, Name) )
        
    return NetworkList

def RememberNetworkSettings():
    """
    Save current active data network settings.
    """
    global ConnectionType
    global ConnectionIAP
    global ConnectionName

    Type = GetNetworkConnectionType()
    if   Type == 'WLAN':    ConnectionType, ConnectionIAP, ConnectionName = GetWLANname()
    elif Type == 'GPRS':    ConnectionType, ConnectionIAP, ConnectionName = GetGPRSname()
    else:                   ConnectionType, ConnectionIAP, ConnectionName = 'NONE', 'NONE', 'NONE'

    LOGGER.info( 'Saved current %s network (%s) for restore.' % (Type, ConnectionName) )

def RestoreNetworkSettings():
    """
    Re-enable previous stored data network settings.
    """
    global ConnectionType
    global ConnectionName

    if ConnectionType == 'GPRS' or ConnectionType == 'WLAN':
        LOGGER.info( 'Restoring saved %s network (%s)' % (ConnectionType, ConnectionName) )
        SetNetworkConnectionType(ConnectionType)
    elif ConnectionType != 'NONE':
        LOGGER.info( 'Disabling connection to %s network (%s).' % (ConnectionType, ConnectionName) )
        DisableConnection()

#############################################
#   connect behaviour
#############################################

def SetAutoConnectBehaviour(NewBehaviour):
    """
    Set Nokia phone data connection behavior.
    The old behaviour is saved in global ConnectionBehaviour and can be used to restore.
    @return bool True if behaviour changed.
    """    

    def _getConnectBehaviour():
        """
        Get current Nokia phone data connection behaviour.
        @return string ASK|ANY|GPRS|WLAN|None
        """ 
        Mode = os.popen('gconftool-2 -g --list-type string /system/osso/connectivity/network_type/auto_connect').read().strip('\n') 
    
        if Mode == '[]':            return 'ASK'
        if Mode == '[*]':           return 'ANY'
        if Mode == '[GPRS]':        return 'GPRS'    
        if Mode == '[WLAN_INFRA]':  return 'WLAN'
        LOGGER.error('Unexpected connect behaviour "%s".' % str(Mode))
        return None
    
    def _setConnectBehaviour(Behaviour):
        """
        Set new Nokia device data connection behaviour (ASK|WLAN|GPRS|ANY).
        @return void
        """
        DelaySeconds = 2
    
        # options: [WLAN_INFRA], [GPRS], [*] <=='any', [] <=='ask'
        if   Behaviour == 'ASK':     String = ''
        elif Behaviour == 'WLAN':    String = 'WLAN_INFRA'
        elif Behaviour == 'GPRS':    String = 'GPRS'
        elif Behaviour == 'ANY':     String = '*'
        else:
            LOGGER.error('Unknown connect behaviour "%s".' % str(Behaviour))
            return
    
        LOGGER.debug('Setting connect behaviour to "%s".' % Behaviour)
        # change setting "Connect automatically" to given Behaviour
        os.system('gconftool-2 -s --type list --list-type string /system/osso/connectivity/network_type/auto_connect [%s]' % String)
        time.sleep(DelaySeconds)

    global ConnectionBehaviour

    OldBehaviour = _getConnectBehaviour()
    if  ConnectionBehaviour == None:
        ConnectionBehaviour = OldBehaviour

    if NewBehaviour == None:
        LOGGER.info('Changing internet connection behaviour not enabled (is "%s").' % OldBehaviour)
        return False
    elif NewBehaviour == OldBehaviour:
        LOGGER.info('Internet connection behaviour already set to "%s".' % NewBehaviour)
        return False

    # New behaviour is different from current. Save the current and try to establish the new one:
    ConnectionBehaviour = OldBehaviour
    _setConnectBehaviour(NewBehaviour)

    # Check new behaviour setting
    if _getConnectBehaviour() == NewBehaviour:
        LOGGER.info('Internet connection behaviour set from "%s" to "%s".' % (OldBehaviour, NewBehaviour))
        return True
    else:
        LOGGER.error('Failed to set internet connection behaviour from "%s" to "%s".' % (OldBehaviour, NewBehaviour))
        return False

#############################################
#   power status
#############################################

def GetBatteryChargeInfo(TextOnly = True, ChargePercentMin = 10):
    """
    Determine battery charge info.
    @param bool TextOnly
    @param int ChargePercentMin Warn if battery change level is below this value.
    @return string Charge info text or
    @return (string, number, bool) if TextOnly is false.
    """
    global dev_obj

    # Try getting charge info, not all devices support all properties: 
    try:
        chargePercent = dev_obj.GetProperty('battery.charge_level.percentage')  # charge amount of battery in percentage
        chargePercentText = '%s%%' % chargePercent
        chargeHint = IF(chargePercent >= ChargePercentMin, '', ' (almost empty!)')
    except Exception, e:
        LOGGER.warning('%s' % e)
        chargePercent = None
        chargePercentText = 'unknown'
        chargeHint = ''

    try:
        isCharging = dev_obj.GetProperty('battery.rechargeable.is_charging') # charging: 1 / discharging: 0
        isChargingText = IF(isCharging, ', charging', ', discharging')
    except Exception, e:
        LOGGER.warning('%s' % e)
        isCharging = None
        isChargingText = ''

    Text = 'Phone battery charge is %s%s%s.' % (chargePercentText, isChargingText, chargeHint)
    if TextOnly: return Text
    return (Text, chargePercent, isCharging)

def LogBatteryChargeInfo():
    """
    Log battery charge info.
    """
    LOGGER.info(GetBatteryChargeInfo())

#############################################
#   SSH 
#############################################

def EnableSSHcontrol():
    """
    Start the ssh routine.
    @return void.
    """
    remote_sh = smscon.smscon_remote_sh(smscnf)
    
    if remote_sh.runs():
        LOGGER.info('ssh connection is already active.')
        return

    if not IsHostReachable(smscnf.REMOTEHOST):
        LOGGER.warning('No ping response from "%s", will try to connect anyway.' % smscnf.REMOTEHOST)

    LOGGER.info('Connecting to "%s" for ssh.' % smscnf.REMOTEHOST)

    (Output, ExitStatus) = remote_sh.start() 
    
    LOGGER.debug('ssh Output = %s' % Output.strip('\n') )
    LOGGER.debug('ssh ExitStatus = %s' % ExitStatus)

    if ExitStatus == 0:
        LOGGER.info(  'ssh connection to "%s" established.' % smscnf.REMOTEHOST)
        RevealThat('ssh', 'connection to "%s" established.' % smscnf.REMOTEHOST)
    else:
        LOGGER.error( 'ssh connection to "%s" failed (%s).' % (smscnf.REMOTEHOST, Output) )
        RevealThat('ssh', 'connection to "%s" failed.'       % smscnf.REMOTEHOST)
        PIDs = remote_sh.stop(Verbose=False)
        if PIDs:
            LOGGER.info('ssh stopped [PIDs=%s].' % PIDs)        

def DisableSSHcontrol(Verbose = True):
    """
    Stop the ssh routine.
    @return void.
    """
    remote_sh = smscon.smscon_remote_sh(smscnf)
    if remote_sh.stop(Verbose = Verbose):
        RevealThat('ssh', 'connection to "%s" terminated.' % smscnf.REMOTEHOST)

#############################################
#   GPS
#############################################

def EnableGPStracking(Mode):
    """
    Start the GPS device mainloop.
    """
    global GpsMode
    GpsMode = Mode
    
    global GpsIsActive
    if not GpsIsActive:
        GpsIsActive = True
        # enable GPS device control
        gobject.idle_add(GPSstart, GpsControl)
    else:
        LOGGER.warning('GPS device already active.')

def DisableGPStracking():
    """
    Stop the GPS device mainloop.
    """
    GpsData = GpsControl

    global GpsIsActive
    if GpsIsActive:
        GpsIsActive = False
        # stop the GPS device
        GpsData.stop()
        LOGGER.info('Stopped acquiering GPS coordinates.')
    else:
        LOGGER.info('GPS device already stopped.')

def GPSstart(GpsData):
    """
    Start the GPS device.
    """
    global GpsDataOK
    GpsDataOK = False

    if GpsMode == 'single':
        LOGGER.info('Starting GPS device in location mode.')
    elif GpsMode == 'multi':
        LOGGER.info('Starting GPS device in tracking mode.')

    global GpsStartTime
    GpsStartTime = time.time()
    GpsData.start()
    
    return False

def OnGPSstop(GpsControl, GpsData):
    """
    Stop the GPS device.
    """
    global GpsIsActive
    GpsIsActive = False

    global GpsIsTest
    if GpsIsTest:
        GpsIsTest = False
        RestoreNetworkSettings()
        ExitDaemon()        
    
    LOGGER.info('Stopping GPS device.')
    GpsData.quit()

def OnGPSchanged(GpsDevice, GpsData):
    """
    Get GPS device data.
    """
    global GpsCoordinateList
    global GpsDataOK
    global GpsStartTime

    if not GpsIsActive:
        return
    # check GPS device timout
    TimePassedSeconds = time.time() - GpsStartTime
    if not GpsDataOK:
        LOGGER.debug('GSP GpsStartTime = %s.' % GpsStartTime)
        if TimePassedSeconds >= float(smscnf.GPSTIMEOUT):
            LOGGER.error(     'GPS device timeout after %.1f sec.' % TimePassedSeconds)
            RevealThat('GPS', 'GPS device timeout after %.1f sec.' % TimePassedSeconds, smscnf.GPSSEND)                    
            # stop the GPS device
            GpsData.stop()
            return

    # adjust start time of GPS device
    GpsStartTime = time.time()
    LOGGER.debug('GpsDevice.status = %s' % GpsDevice.status) # LOCATION_GPS_DEVICE_STATUS_NO_FIX  or  LOCATION_GPS_DEVICE_STATUS_FIX

    if (GpsDevice.fix[1] & location.GPS_DEVICE_LATLONG_SET) == 0:
        LOGGER.info('Waiting for GPS device satellite fix.')
        return
    # GPS device has fix.
    Latitude  = GpsDevice.fix[4]
    Longitude = GpsDevice.fix[5]
    Accuracy  = GpsDevice.fix[6] / 100

    # Check if coordinates acquired by GPS device are valid
    GpsDataOK = str(Latitude) != 'nan' and str(Longitude) != 'nan' and str(Accuracy) != 'nan'
    if not GpsDataOK:
        LOGGER.info('Waiting for valid GPS coordinate (search time = %.1f sec.).' % TimePassedSeconds)
        return

    if GpsMode == 'single':
        if len(GpsCoordinateList) == 0:
            LOGGER.info('Start collecting %s GPS coordinate(s)...' % smscnf.GPSPOLLING)

        GpsCoordinateList.append( (Latitude, Longitude, Accuracy) )
        LOGGER.info( 'GPS coordinate #%d acquired (%f, %f, %.1f m.).'
                        % (len(GpsCoordinateList), Latitude, Longitude, Accuracy) )                
        if len(GpsCoordinateList) < int(smscnf.GPSPOLLING):
            return
        # sort GPS coordinate list on 'Accuracy'
        GpsCoordinateList = sorted( GpsCoordinateList, key=itemgetter(2) )
        Latitude, Longitude, Accuracy = GpsCoordinateList[0]
        LOGGER.info( 'Most accurate GPS coordinate out of %d = (%f, %f, %.1f m.).'
                        % (len(GpsCoordinateList), Latitude, Longitude, Accuracy) )

        GpsCoordinateList = []
        # stop the GPS device
        GpsData.stop()

    # tracking mode
    elif GpsMode == 'multi':
        LOGGER.info( 'GPS coordinate acquired (%f, %f, %.1f m.).' % (Latitude, Longitude, Accuracy) )

    GpsText = ['Lat: %f\n' % Latitude,
               'Long: %f\n' % Longitude,
               'Accuracy: %.1f mtr\n' % Accuracy,
               'Duration: %.1f sec\n' % TimePassedSeconds]

    LinkText = ['<a href="http://maps.google.com/maps?q=%f,%f">GoogleMap</a>\n'% (Latitude, Longitude)]

    RevealThat('GPS Location', GpsText + LinkText,
                SendMethod = smscnf.GPSSEND,
                Attachment = smscnf.GPSSEND in ['email', 'both'] and CreateGoogleMap(Latitude, Longitude),
                Attachhint = 'image')

def OnGPSerror(GpsControl, GpsError, GpsData):
    """
    GPS device error detected.
    """
    global GpsIsActive
    GpsIsActive = False

    if GpsError == location.ERROR_USER_REJECTED_DIALOG:
        LOGGER.error("GPS device error (User didn't enable requested methods).")
    elif GpsError == location.ERROR_USER_REJECTED_SETTINGS:
        LOGGER.error('GPS device error (User changed settings, which disabled location).')
    elif GpsError == location.ERROR_BT_GPS_NOT_AVAILABLE:
        LOGGER.error('GPS device error (Problems with BT GPS).')
    elif GpsError == location.ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
        LOGGER.error('GPS device error (Requested method is not allowed in offline mode).')
    elif GpsError == location.ERROR_SYSTEM:
        LOGGER.error('GPS device error (System error).')
    else:
        LOGGER.error('GPS device error (Unknown error).')
        
    LOGGER.error('Stopping failed GPS device.')
    GpsData.quit()

###############################

def CreateGoogleMap(Latitude, Longitude):
    """
    Create map of current GPS coordinate.
    @return string|False Filename of map or False if no map.
    """
    MapZoom     = 16
    MapSize     = (600,600)
    MarkerColor = 'red'
    MarkerLabel = 'N'

    MapFilename = os.path.join(LogPath, MapName)
    MapUrl = 'http://maps.google.com/maps/api/staticmap?center=%s,%s&zoom=%s&size=%sx%s&format=png&markers=color:%s|label:%s|%s,%s&sensor=false' \
              % (Latitude,
                 Longitude,
                 MapZoom,
                 MapSize[0],
                 MapSize[1],
                 MarkerColor,
                 MarkerLabel,
                 Latitude,
                 Longitude)
    try:
        urllib.urlretrieve( MapUrl, MapFilename )
    except:
        LOGGER.error('Failed to create GPS coordinate map (retrieve error).')
        return False
    if not os.path.isfile(MapFilename):
        LOGGER.error('Failed to create GPS coordinate map (file not found).')
        return False
    else:
        LOGGER.info('Created GPS coordinate map.')
        return MapFilename

#############################################
#   SIM card
#############################################

def GetOperatorName():
    """
    Get current SIM telecom operator name.
    """
    try:
        Output = os.popen('dbus-send --system --print-reply=literal --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_registration_status').read()
        E = re.compile('uint32 (\d+)')
        R = E.findall(Output)
        Output = os.popen( 'dbus-send --system --print-reply --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_operator_name byte:0 uint32:"%s" uint32:"%s"' % (R[1], R[2]) ).read()
        E = re.compile('string "(.*)"')
        R = E.findall(Output)[0]
    except:
        LOGGER.error('Could not get telecom network operator name.')
        return 'UNKNOWN'        

    return R

def ReadIMSI():
    """
    Load IMSI code(s) from file.
    @return list of IMSI codes.
    """
    imsi = smscon.IMSIConfiguration()
    if imsi.load(Silent = True):
        LOGGER.info('Found %d IMSI code(s) in "%s" file.' % (len(imsi.keys()), imsi.Filename))
    else:
        LOGGER.warning('Initalizing new "%s" file with current valid IMSI code.' % imsi.Filename)
        imsi.update(smscon.GetPhoneCode('IMSI'), '')
    return imsi.keys()

def WriteIMSI(newIMSI):
    """
    Save new IMSI code to file.
    @return void
    """
    imsi = smscon.IMSIConfiguration()
    if imsi.update(newIMSI, ''):
        LOGGER.info( 'Added IMSI code (%s) to "%s" file.' % (newIMSI, imsi.Filename) )
    else:
        LOGGER.error( 'Failed to add IMSI code (%s) to "%s" file.' % (newIMSI, imsi.Filename) )

def ValidateIMSI():
    """
    Check if current used IMSI code from SIM card is authorized.
    """
    CurrentIMSI = smscon.GetPhoneCode('IMSI')

    if CurrentIMSI == None:
        LOGGER.info('No SIM card present.')
        ExitDaemon()

    # compare current IMSI code with saved one or more in code file.
    elif CurrentIMSI in ReadIMSI():
        LOGGER.info('Authorized IMSI code found.')

    else:
        LOGGER.warning('IMSI code has changed to "%s".' % CurrentIMSI)

        # save new IMSI code to file
        WriteIMSI(CurrentIMSI)    

        # auto unlock phone after new SIM is inserted
        if smscnf.SIMUNLOCK == 'yes':
            PerformDeviceUnlock()                 

        SendMethod = smscnf.MESSAGESEND
        if   SendMethod == 'none':  SendMethod = 'sms'
        elif SendMethod == 'email': SendMethod = 'both'

        RevealThat('New SIM', '\n'.join([
                        'IMSI = %s' % CurrentIMSI,
                        'IMEI = %s' % smscon.GetPhoneCode('IMEI'),
                        'Telecom Operator = "%s".' % GetOperatorName()
                    ]), SendMethod)

#############################################

def DoCommandLine():
    """
    Check arguments for smscon_daemon and perform corresponding actions.
    @return bool True if ok
    """    
    def TellError(Msg):
        LOGGER.error(Msg)
        return False

    def PrepareNetwork():
        if  GetNetworkConnectionType() != 'WLAN':
            SetNetworkConnectionType('GPRS')
            SetAutoConnectBehaviour('GPRS') # disable switching to other network

    args = sys.argv[1:]
    if len(args) > 0:
        Mode = args.pop(0)
        if Mode == '-test':
            if len(args) == 0: return TellError('%s requires argument.' % Mode)
            Func = args.pop(0)
    
            if Func == 'gps1' or Func == 'gps2': 
                RememberNetworkSettings()
                PrepareNetwork();
    
                global GpsIsTest
                GpsIsTest = True
                
                if Func == 'gps1': # test mode for gps (location mode)
                    EnableGPStracking('single')
                elif Func == 'gps2': # test mode for gps (tracking mode)
                    EnableGPStracking('multi')
                return True
    
            if Func == 'ssh': # test mode for ssh connection
                RememberNetworkSettings()
                PrepareNetwork();
                DisableSSHcontrol(Verbose = False)
                EnableSSHcontrol()
                return True

            if Func == 'ssh-off': # test for ssh disconnect
                DisableSSHcontrol(Verbose = True)
                return True

            if Func == 'cam': # test mode for camera
                RememberNetworkSettings()
                PrepareNetwork();
                PerformCamera()
                RestoreNetworkSettings()
                SetAutoConnectBehaviour(ConnectionBehaviour)

            elif Func == 'sms': # test mode for sms
                SMSsend(smscnf.MASTERNUMBER, ['Test SMS from Nokia N900 phone'], 'send SMS message test to {RECIPIENTNUMBER}')

            elif Func == 'sms2': # test mode for sms
                SMSsend(smscnf.SENDERNUMBER, ['Test SMS from Nokia N900 phone'], 'send SMS message test to {RECIPIENTNUMBER}')

            elif Func in ('email1', 'email2'): 
                RememberNetworkSettings()
                PrepareNetwork();

                if Func == 'email1': # test mode1 for email (text only)
                    EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                              Subject = 'Test email / Nokia N900',
                              Text = 'Email sending from Nokia N900 phone is successful!',
                              SuccessLogMsg = 'send test email message to "%s"' % smscnf.EMAILADDRESS)
                elif Func == 'email2': # test mode2 for email + frontcam image
                    ImageFilename = TakeFrontCamPicture()
                    if ImageFilename:
                        EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                              Subject =  'Test email / Nokia N900',
                              Text =  'Frontcam picture from Nokia N900 phone:<br>%s<br>' % GetTimeString(),
                              SuccessLogMsg = 'send email camera picture send to "%s"' % smscnf.EMAILADDRESS,
                              Attachment =  ImageFilename,
                              Attachhint = 'image' )

                RestoreNetworkSettings()
                SetAutoConnectBehaviour(ConnectionBehaviour)
    
            elif Func == 'call': # test mode for phone call
                PerformPhonecall(smscnf.MASTERNUMBER)

            elif Func == 'script': # test mode for script
                PerformShellScript()

            else:
                LOGGER.error('Test option error (%s).' % Func)

        # /* ONLY FOR DEVELOPMENT USAGE */
        elif Mode == '-comtest':
            if len(args) == 0: return TellError('%s requires argument.' % Mode)
            Command = args.pop(0)

            LOGGER.warning('Simulating received SMS message with "%s" command.' % Command)
            ProcessCommand(Command, SenderPhoneNumber = smscnf.SENDERNUMBER,
                           ExtraInfo = 'SIMULATION!', LogIfCmdUnknown = True)    
        else:
            LOGGER.warning('Option error (%s) - ignored' % Mode)
            return True
    if len(args) > 0:
        LOGGER.warning('Options ignored: %s' % ' '.join(args))
    return True                

#############################################

def CheckBootLoader():
    """
    Check if smscon auto-loads at system boot.
    @return void.
    """
    if IsFileReadable(smscon.BootPath, smscon.BootFile):
        LOGGER.info('%s auto-loads at boot. OK.' % smscon.NAME)
    else:
        LOGGER.warning(    "%s does not auto-load at boot." % smscon.NAME)
        RevealThat('boot', "%s does not auto-load at boot." % smscon.NAME)        

def ExitDaemon(ExitReason = 'regularly'):
    """
    Exit smscon_daemon.
    @return NO RETURN.
    """
    LOGGER.critical('%s is stopping %s.' % (DaemonFile, ExitReason))
    sys.exit(1)

def RestartDaemon():
    """
    Restart smscon_daemon.
    @return void (does not return on success).
    """ 
    DelaySeconds = 10

    LOGGER.critical('(RESTART) %s tries to restart itself w/o regular stop.' % DaemonFile)
    try:
        time.sleep(DelaySeconds)
        os.system( '%s -restart' % shellArg(os.path.join(InstPath, smscon.NAME)) )
    finally:
        LOGGER.critical('(RESTART) Not restarted. Failed to kill %s.' % DaemonFile)

#############################################
#   keyboard detect (dbus trigger)
#############################################

def OnKeyboardSliderChange(Action, Type):
    """
    Check and reveal state of keyboard slider.
    @return void. 
    """
    try:
        SliderState = open('/sys/devices/platform/gpio-switch/slide/state').read().strip()
    except Exception, e:
        LOGGER.error('Failed to detect keyboard slider state: %s' % e)
        SliderState = 'in unknown state'
    LOGGER.info('Phone is being used: keyboard now %s.' % SliderState)
    RevealThat('Keyboard', 'Phone is being used: keyboard now %s.' % SliderState)            

#############################################
#   battery charge (dbus trigger)
#############################################

def OnBatteryStatusChange(BarLevel, BarLevelMax):
    """
    Check and reveal battery charge level.
    @return void. 
    """
    global LastBatteryStatusMessage
    ChargePercentMin = 10
    Message, ChargePercent, Charging = GetBatteryChargeInfo(TextOnly = False, ChargePercentMin = ChargePercentMin)        

    if Message != LastBatteryStatusMessage:
        LastBatteryStatusMessage = Message
        LOGGER.info(Message)
        if (Charging != None and not Charging) or (ChargePercent != None and ChargePercent < ChargePercentMin): 
            RevealThat('battery warning', Message)

#############################################
#   remote host file checking (remote smscon trigger)
#############################################

def PerformCheckHost(CheckHostURL):
    """
    Check remote host file for triggering smscon
    @return True
    """
    global CheckHostLastCommand, CheckHostStatusCount, CheckHostStatusTimestamp

    CommandList = {
        '01' : smscnf.COM_CHECK,
        '02' : smscnf.COM_REBOOT,
        '03' : smscnf.COM_POWEROFF,
        '04' : smscnf.COM_POWER,
        '05' : smscnf.COM_LOCATION,
        '06' : smscnf.COM_REMOTEON,
        '07' : smscnf.COM_REMOTEOFF,
        '08' : smscnf.COM_CAMERA,
        '09' : smscnf.COM_CALL,
        '10' : smscnf.COM_LOCK,
        '11' : smscnf.COM_UNLOCK,
        '12' : smscnf.COM_TRACKON,
        '13' : smscnf.COM_TRACKOFF,
        '14' : smscnf.COM_CUSTOM,
        '15' : smscnf.COM_SHELL,
        '16' : smscnf.COM_ALARM,
        '17' : smscnf.COM_RESTART,
        '18' : smscnf.COM_CUSTOMLOG,
        '19' : smscnf.COM_LOG,
    }  

    # Supply default protokol "http:"
    if  CheckHostURL.find(":/") < 0:
        CheckHostURL = 'http://' + CheckHostURL

    def eventsOf(what):
        return "%d %s since %s GMT" % (abs(CheckHostStatusCount), what, GetTimeString(CheckHostStatusTimestamp))

    LOGGER.debug('(CHECKHOST) Opening remote host file (%s).' % CheckHostURL)
    try:
        UrlFile = urllib.urlopen('%s?LASTMCD=%s&CHECKTIME=%s' % (CheckHostURL, CheckHostLastCommand, smscnf.CHECKTIME))
        Command = UrlFile.read(2)
        UrlFile.close()
    except Exception, e:
        if CheckHostStatusCount >= 0:
            LOGGER.error('(CHECKHOST) After %s: Failed to read remote file: %s' % (eventsOf('OK'), e))
            CheckHostStatusCount = 0
            CheckHostStatusTimestamp = time.gmtime()
        CheckHostStatusCount -= 1
        if -CheckHostStatusCount % 25 == 0:
            RevealThat('checkhost', 'Got %s: %s' % eventsOf('errors'), e)
        return True

    if CheckHostStatusCount <= 0:
        LOGGER.info('(CHECKHOST) After %s: Successfully read remote file.' % eventsOf('errors'))
        CheckHostStatusCount = 0
        CheckHostStatusTimestamp = time.gmtime()
    CheckHostStatusCount += 1

    LOGGER.debug('(CHECKHOST) Got command "%s". Last command was "%s".' % (Command, CheckHostLastCommand))
    if CheckHostLastCommand == Command:
        return True
    CheckHostLastCommand = Command
    
    if Command in CommandList:
        Command = CommandList[Command]
        LOGGER.info( '(CHECKHOST) %s is activated (command "%s") by remote host file.' % (DaemonFile, Command) )
        ProcessCommand(Command, CommandOrigin = 'CHECKHOST', ExtraInfo = CheckHostURL)

    elif Command == '00':
        LOGGER.info( '(CHECKHOST) %s is deactivated (%s) by remote host file.' % (DaemonFile, Command) )

    elif Command == '-1':
        LOGGER.info( '(CHECKHOST) %s will restart (%s) by remote host file.' % (DaemonFile, Command) )
        try:
            os.system( '"%s" -restart' % os.path.join(InstPath, smscon.NAME) )
        except:
            LOGGER.error('(CHECKHOST) Failed to kill %s' % DaemonFile)
    
    else:
        LOGGER.error( '(CHECKHOST) %s unknown command (%s) by remote host file.' % (DaemonFile, Command) )

    return True # run it continiously

#############################################

def InstallSignalReceivers():
    """
    Enable or disable signal receivers and polling timers to detect
    various device sensors and control command sources.
    @return void
    """
    
    global IsKeyboardSliderCheckActive
    global IsBatteryStatusCheckActive
    global IsCheckHostActive

    if (not IsKeyboardSliderCheckActive) == (smscnf.KEYBOARDDETECT == 'yes'):
            IsKeyboardSliderCheckActive   =  smscnf.KEYBOARDDETECT == 'yes'
            set_signal_receiver = IF(IsKeyboardSliderCheckActive, bus.add_signal_receiver, bus.remove_signal_receiver)
            set_signal_receiver(OnKeyboardSliderChange,
                                    path           = '/org/freedesktop/Hal/devices/platform_slide',
                                    dbus_interface = 'org.freedesktop.Hal.Device',
                                    signal_name    = 'Condition')
    LOGGER.info('Keyboard slider change detection is %s.' % IF(IsKeyboardSliderCheckActive, 'on', 'off'))

    if (not IsBatteryStatusCheckActive) == (smscnf.AUTOBATREPORT == 'yes'):
            IsBatteryStatusCheckActive   =  smscnf.AUTOBATREPORT == 'yes'
            set_signal_receiver = IF(IsBatteryStatusCheckActive, bus.add_signal_receiver, bus.remove_signal_receiver)
            set_signal_receiver(OnBatteryStatusChange, 
                                    path           = '/com/nokia/bme/signal',
                                    dbus_interface = 'com.nokia.bme.signal',
                                    signal_name    = 'battery_state_changed')
    LOGGER.info('Battery status change detection is %s.' % IF(IsBatteryStatusCheckActive, 'on', 'off'))

    def CheckHost(checktime):
        global IsCheckHostActive
        if smscnf.ENABLECHECKHOST == 'yes':
            IsCheckHostActive = True
            PerformCheckHost(smscnf.CHECKHOST)
            if smscnf.CHECKTIME == checktime:
                return True
            LOGGER.info('(CHECKHOST) Checkhost is on. Checking each %s minutes.' % float(smscnf.CHECKTIME))
            gobject.timeout_add(int(float(smscnf.CHECKTIME) * 60000), CheckHost, smscnf.CHECKTIME)
        else:
            IsCheckHostActive = False
            LOGGER.info('(CHECKHOST) Checkhost is off.')
        return False
    if not IsCheckHostActive:
        CheckHost(None)

def LoadConfiguration():
    """"
    Load and establish user settings from smscon_config file
    @return bool Success indicator
    """
    global smscnf
    isReload = not not smscnf;
    LastKnownGoodConfigPath = os.path.join(InstPath, LastKnownGoodConfigFile)

    theConf = smscon.UserConfiguration()
    if  theConf.load():
        theConf.upgrade()
    else:
        LOGGER.critical( 'Failed to load "%s" file.' % theConf.Filename)
        if isReload:
            return False
        LOGGER.warning('Trying last known good config.')
        theConf = smscon.UserConfiguration(Filename = LastKnownGoodConfigPath)
        if not theConf.load():
            return False
        theConf.upgrade(Permanent = False)  # Only in RAM; Do not change last known good on disk.

    # For convinience code make config settings available as attributes:
    for key in theConf.getKnownConfigVars():
        if not key in theConf:
            LOGGER.critical( 'Failed to load "%s" file.' % theConf.Filename)
            return False
        setattr(theConf, key, theConf[key])

    LOGGER.info('Successfully loaded "%s" file.' % theConf.Filename)
    theConf.backup(LastKnownGoodConfigPath, Silent = 'last known good config')

    smscnf = theConf
    if isReload:
        InstallSignalReceivers()
    return True

def main():
    """
    Main function of smscon daemon. Does not return to caller.
    """
    global smscnf, bus, dev_obj, GpsControl

    LOGGER.critical('%s is starting.' % DaemonFile)
    
    signal.signal(smscon.smscon_daemon.TELL_TERMINATE,
                    lambda signum, stack_frame: ExitDaemon("on signal %s" % str(signum))  )
    signal.signal(smscon.smscon_daemon.TELL_RELOAD,
                    lambda signum, stack_frame: LoadConfiguration()                       )
    
    LoadConfiguration() or ExitDaemon('on config load error')
    ValidateIMSI()
    CheckBootLoader()
    
    # Connect to various signal triggers:
    try:            
        DBusGMainLoop(set_as_default = True)
    
        bus = dbus.SystemBus()
    
        bus.add_signal_receiver(OnSmsReceived,
                                path           = '/com/nokia/phone/SMS',
                                dbus_interface = 'Phone.SMS',
                                signal_name    = 'IncomingSegment')
    
        # set battery charge measurment
        hal_obj = bus.get_object('org.freedesktop.Hal', '/org/freedesktop/Hal/Manager')
        hal = dbus.Interface(hal_obj, 'org.freedesktop.Hal.Manager')
    
        uids = hal.FindDeviceByCapability('battery')
        dev_obj = bus.get_object('org.freedesktop.Hal', uids[0])
        LogBatteryChargeInfo()
    
        # set GPS device control
        GpsControl = location.GPSDControl.get_default()
        GpsDevice  = location.GPSDevice()
    
        # set GPS method & most suitable interval
        gps_interval = int(float(smscnf.GPSINTERVAL))
        if   gps_interval <= 10:  gps_interval = location.INTERVAL_10S
        elif gps_interval <= 20:  gps_interval = location.INTERVAL_20S
        elif gps_interval <= 30:  gps_interval = location.INTERVAL_30S
        elif gps_interval <= 60:  gps_interval = location.INTERVAL_60S
        else:                     gps_interval = location.INTERVAL_120S
        GpsControl.set_properties(preferred_method   = location.METHOD_GNSS|location.METHOD_AGNSS,
                                  preferred_interval = gps_interval)
    
        GpsControl.connect('error-verbose', OnGPSerror, gobject.MainLoop() )
        GpsDevice.connect('changed', OnGPSchanged, GpsControl )
        GpsControl.connect('gpsd-stopped', OnGPSstop, gobject.MainLoop() )
    
        if not DoCommandLine():
            ExitDaemon('bad command line')
    
        InstallSignalReceivers()
    
        gobject.MainLoop().run()
    
    except KeyboardInterrupt:
        ExitDaemon('on KeyboardInterrupt')
    
    except Exception, e:
        LOGGER.critical('<<< SMSCON FATAL ERROR:\n%s >>>' % e)
        # Disable resending of messages (would fail anyway when exiting): 
        smscnf.ENABLERESEND = 'no' 
        RevealThat('fatal error', ['smscon_daemon crashed.\n', '%s' % e])        
        ExitDaemon('on fatal error')
    
    ExitDaemon('on mainloop exit')

#############################################

"""
"""