# -*- coding: utf-8 -*-
##############################################################
# smscon_master - remote control master for Nokia N900 phone #
# $Id: smscon_master.py 117 2011-12-29 15:24:00Z yablacky $
##############################################################

import os
import sys
import time                
import re

from smsconlib import *

##############################################################
# For help text, assemble pretty display names of our various files:

ScriptDisplayName    = ScriptFile.replace(NAME, '').strip('_')
ConfigDisplayName    = ConfigFile.replace(NAME, '').strip('_')
CodeDisplayName      = CodeFile  .replace(NAME, '').strip('_')
BootDisplayName      = BootFile  .replace(NAME, '').strip('_')
ScriptLogDisplayName = ScriptLog .replace(NAME, '').strip('_')
LogDisplayName       = LogFile   .replace(NAME, '').strip('_')

##############################################################

def ShowFile(*File):
    """
    Show file. Very verbose.
    @return bool True iff file exists and is non empty.
    """
    Filename = os.path.join(*File)
    try:
        f = open(Filename, 'r')
    except:
        LOGGER.warning('File "%s" not found.' % Filename)
        return False
    Lines = f.readlines() 
    f.close()
    if Lines == []:
        LOGGER.info('File "%s" is empty.' % Filename)
        return False
    for Line in Lines:
        print Line,
    return True

def ShowSmsCommands(theConf):
    """
    Show commands that are recognized if sent via SMS.
    @return bool True iff commands could be shown.
    """
    for SmsCommand in sorted(theConf.getSmsCommandList()):
        print theConf._genConfigLine(SmsCommand, theConf['SMS_COMPREFIX'] + theConf[SmsCommand] + theConf['SMS_COMSUFFIX'])
    return True

def ShowIMSICodes():
    """
    Show IMSI numbers from IMSI database.
    @return bool True iff IMSI codes available and listed.
    """
    imsi = IMSIConfiguration()
    if not imsi.load():
        return False
    for n, Text in enumerate(imsi):
        print 'IMSI_%02d = %s' % ( n + 1, Text )
    return True;

def SaveIMSIcode():
    """
    Get current inserted SIM IMSI code & save to IMSI database.
    @return bool True iff IMSI code written to IMSI database.
    """
    imsi = IMSIConfiguration()
    return imsi.update(GetPhoneCode('IMSI'), '')

def RemoveIMSIcode():
    """
    Remove current inserted SIM IMSI code from IMSI database.
    @return bool True iff IMSI code no longer present in IMSI database.
    """
    imsi = IMSIConfiguration()
    return imsi.update(GetPhoneCode('IMSI'), None)

##############################################################
# Command line interface

def ShowOptions(Level = 1):
    """
    Show options for smscon.
    """
    if Level >= 1:
        print '== %s %s - Nokia N900 remote control utility ==' % (NAME, VERSION)
        print 'usage: %s [Options]...' % NAME
        print ' Options:'
        print '   -start             : Start %s' % DaemonFile
        print '   -restart           : Stop & restart %s' % DaemonFile
        print '   -stop              : Stop %s' % DaemonFile
        print '   -status            : Get %s status' % DaemonFile
        print '   -log               : Show log file'
        print '   -del log           : Delete log file'
        print '   -sms               : Show SMS commands'
        print '   -set name "value"  : Set user setting (name = "value")'
        print '   -config            : Show %s file' % ConfigDisplayName
        print '   -script            : Show %s file' % ScriptDisplayName
        print '   -scriptlog         : Show %s file' % ScriptLogDisplayName
        print '   -del scriptlog     : Delete %s file' % ScriptLogDisplayName
        print '   -boot              : Enable %s to start at phone boot' % DaemonFile
        print '   -unboot            : Disable %s to start at phone boot' % DaemonFile
        print '   -add imsi          : Add IMSI code of current SIM to %s file' % CodeDisplayName
        print '   -remove imsi       : Remove IMSI code of current SIM from %s file' % CodeDisplayName
        print '   -version           : Show version number'    
        print '   -help              : This help menu'
        print '   -help2             : More help (special options, rarely used).'
    if Level >= 2:
        print ' Special options (normally not needed!):'
        print '   -init              : Ensure %s files exist; upgrades existing files' % ' & '.join((ConfigDisplayName, ScriptDisplayName))
        print '   -reset             : Restore factory settings: Stop all tasks,'
        print '                        disable boot starter and delete all the'
        print '                        %s-,' % '-, '.join((BootDisplayName, ConfigDisplayName, CodeDisplayName,
                                                        ScriptDisplayName, ScriptLogDisplayName, LogDisplayName))
        print '                        %s-, %s- and %s-files.' % (PhotoName, MapName, AudioFile)
        print '   -export filename   : Export user settings into filename.'
        print '                        Use -export! to write even existing file.'
        print '   -import filename   : Import user settings from filename.'
        print '                        Use -import! to continue import on errors.'
        print '   -imsi              : Show IMSI code file'
        print '   -del imsi          : Delete %s file' % CodeDisplayName
        print '   -del config        : Delete %s file' % ConfigDisplayName
        print '   -del script        : Delete %s file' % ScriptDisplayName
        print '   -sendsms number message : Send one SMS message to number. Number may be verbatim "MASTERNUMBER"'
    
        if os.geteuid() != 0:
            print 'WARNING: %s must run as "root". You are currently not "root".' % NAME

def ShowHiddenOptions():
    """
    Show hidden options for smscon.
    """
    print ' Developer options:'
    print '   -devhelp           : This extended help menu'
    print '   -branch            : Show branch number. Deprecated. Removed in future. Shows -version now.'
    print '   -build             : Show build number.  Deprecated. Removed in future. Shows -version now.'
    print '   -init!             : Create default %s files. Do not upgrade existing ones.' % ' & '.join((ConfigDisplayName, ScriptDisplayName))
    print '   -get name "default": Get user setting'
    print '   -test "test"       : Run developer test (gps1, gps2, ssh, ssh-off, cam, sms, email1, email2, call, script)'
    print '   -comtest "command" : Run %s command as if received as SMS message (no prefix/suffix required)' % NAME
    print '   -devconfig password: Show uncrypted %s file. MASTERNUMBER is password' % ConfigDisplayName

##############################################################

def DoCommandLine():
    """
    Parse command line and perform actions.
    @return bool Success indicator.
    """
    def TellError(Msg):
        LOGGER.error(Msg)
        return False

    global ourConf, daemon, settingsChangeCount
    script = smscon_script()
    remosh = smscon_remote_sh(ourConf)

    ClientUsrID = OurUsrID = os.geteuid()
    ClientGrpID = OurGrpID = os.getegid()
    DelaySeconds = 0.5
    args = sys.argv[1:]
    ok = len(args) > 0
    if not ok:
        ShowOptions()  
    while (ok and len(args) > 0):
        LOGGER.debug("Command Line args: %s" % ' '.join(args))
        OriMode = Mode = args.pop(0)
        # Allow all options to start with the usual double dash as well:
        if len(Mode) > 3 and Mode[0:2] == '--' and Mode[3:1] != '-':
            Mode = Mode[1:] 

        if   Mode == '-help':       ShowOptions(1) 
        elif Mode == '-help2':      ShowOptions(2) 
        elif Mode == '-devhelp':    ShowOptions(2); ShowHiddenOptions()
        elif Mode == '-version':    print VERSION        
        elif Mode == '-branch':     print VERSION
        elif Mode == '-build':      print VERSION
        elif Mode == '-status':     print "Daemon %s is %s" % (DaemonFile, IF(daemon.runs(), 'running', 'off'))
        elif Mode == '-log':        ok = ShowFile(LOGGER.LogFilename)    
        elif Mode == '-imsi':       ok = ShowIMSICodes()
        elif Mode == '-sms':        ok = ShowSmsCommands(ourConf)
        elif Mode == '-config':     ok = ShowFile(ourConf.Filename)
        elif Mode == '-script':     ok = ShowFile(script.Filename)
        elif Mode == '-scriptlog':  ok = ShowFile(InstPath, ScriptLog)
        elif Mode == '-devconfig':
            if len(args) == 0: return TellError("%s requires argument" % Mode)
            Password = args.pop(0)
            ok = UserConfiguration().show(Password = Password)

        # The next commands potentially modify data and therefore need root permissions:
        elif os.geteuid() != 0 and not XDEBUG:
            return TellError('%s must run as "root" for this command.' % NAME)

        elif Mode == '-usrid':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            try:    ClientUsrID = int(args.pop(0))
            except: return TellError("%s argument must be integer." % Mode)

        elif Mode == '-grpid':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            try:    ClientGrpID = int(args.pop(0))
            except: return TellError("%s argument must be integer." % Mode)

        elif Mode in ( '-init', '-init!'):
            Force = Mode == '-init!'
            if Force and daemon.stop() + script.stop():
                time.sleep(DelaySeconds)

            if ourConf.create(Force = Force):
                LOGGER.warning('Do not edit "%s" directly. Use "%s -set OPTION VALUE" instead!' % (ourConf.Filename, NAME))
                settingsChangeCount += 1
            else:
                ok = False
            ok = script.init(Force) and ok

        elif Mode in ('-export', '-export!') :
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Filename = args.pop(0)
            if not IsFileReadable(ourConf.Filename):
                return TellError('Export to "%s" failed: No source configuration file.' % Filename)
            if Mode != '-export!' and IsFileReadable(Filename):
                return TellError('Export to "%s" failed: file already exists.' % Filename)
            # Perform export operation in client user/group privilege context.
            # This prevents creation or overwriting of files where user has
            # normally no permission:   
            try:
                os.setegid(ClientGrpID) # must set group first!
                os.seteuid(ClientUsrID)
                ok = ourConf.backup(Filename = Filename) and ok
            except Exception, e:
                ok = TellError('Export to "%s" failed: %s.' % (Filename, e))
            finally:
                os.seteuid(OurUsrID)
                os.setegid(OurGrpID)

        elif Mode in ('-import', '-import!') :
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Filename = args.pop(0)
            restoreCount = ourConf.restore(Filename = Filename, Force = Mode == '-import!')
            settingsChangeCount += abs(restoreCount)
            ok = restoreCount > 0 and ok

        elif Mode in ('-start', '-restart'):
            if not IsFileReadable(InstPath, ConfigFile):
                return TellError('Config file %s not present. Use -init.' % ConfigFile)
            if not IsFileReadable(InstPath, ScriptFile):
                return TellError('Script file %s not present. Use -init.' % ScriptFile)
    
            if daemon.runs():
                if Mode == '-start':
                    LOGGER.info('Daemon %s already active.' % DaemonFile)
                    continue
                if daemon.stop():
                    time.sleep(DelaySeconds)
            daemon.start()
            time.sleep(DelaySeconds)
            PIDdaemon = daemon.runs()
            if not PIDdaemon:
                return TellError('Daemon %s failed to start.' % DaemonFile)
            LOGGER.info('Daemon %s started [PID=%s].' % (DaemonFile, PIDdaemon))

        elif Mode == '-stop':
            daemon.stop()
            remosh.stop()
            script.stop()

        elif Mode == '-reset':
            if (daemon.stop() + remosh.stop() + script.stop()):
                time.sleep(DelaySeconds) 
            ok = DeleteFile(BootPath, BootFile) and ok
            ok = DeleteFile(InstPath, CodeFile) and ok
            ok = DeleteFile(InstPath, ConfigFile) and ok
            ok = DeleteFile(InstPath, ScriptFile) and ok
            ok = DeleteFile(InstPath, ScriptLog) and ok
            ok = DeleteFile(InstPath, PhotoName) and ok
            ok = DeleteFile(InstPath, MapName) and ok
            ok = DeleteFile(InstPath, AudioFile) and ok
            ok = DeleteFile(LogPath, LogFile) and ok
            ok = DeleteFile(LogPath, BootLogFile) and ok

        elif Mode == '-boot':       ok = CreateBootfile()
        elif Mode == '-unboot':     ok = DeleteFile(BootPath, BootFile)

        elif Mode in('-test', '-comtest'):
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if not daemon.runs():
                daemon.start(Mode, Func)
            elif not daemon.writeCliCommand(Mode, Func):
                return TellError('%s %s failed to start.' % (Mode[1:], Func))
            LOGGER.info('%s %s started...' % (Mode[1:], Func))

        elif Mode == '-del':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if   Func == 'log':         ok = DeleteFile(LogPath, LogFile)
            elif Func == 'imsi':        ok = DeleteFile(InstPath, CodeFile)
            elif Func == 'config':      ok = DeleteFile(InstPath, ConfigFile)
            elif Func == 'script':      ok = DeleteFile(script.Filename)
            elif Func == 'scriptlog':   ok = DeleteFile(InstPath, ScriptLog)
            else:   return TellError('%s: unknown option (%s).' % (Mode, Func))

        elif Mode == '-add':
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if   Func == 'imsi':    ok = SaveIMSIcode()
            else:   return TellError('%s: unknown option (%s).' % (Mode, Func))

        elif Mode == '-remove': # add current IMSI code to smscon_code file
            if len(args) == 0: return TellError("%s requires argument." % Mode)
            Func = args.pop(0)
            if   Func == 'imsi':    ok = RemoveIMSIcode()
            else:   return TellError('%s: unknown option (%s).' % (Mode, Func))

        elif Mode == '-set':
            if len(args) < 2: return TellError("%s requires 2 arguments." % Mode)
            ConfigVar   = args.pop(0)
            ConfigValue = args.pop(0)
            ok = ourConf.update(ConfigVar, ConfigValue)
            if ok: settingsChangeCount += 1

        elif Mode == '-get':
            if len(args) < 2: return TellError("%s requires 2 arguments." % Mode)
            ConfigVar   = args.pop(0)
            ConfigValue = args.pop(0)
            print ourConf.get(ConfigVar, ConfigValue)

        elif Mode == '-sendsms':
            if len(args) < 2: return TellError("%s requires 2 arguments." % Mode)
            TargetNumber = Recipient = args.pop(0).upper()
            if TargetNumber == 'MASTERNUMBER':
                TargetNumber = ourConf.get('MASTERNUMBER', '')
            Message = args.pop(0)
            if re.match('^\+?0*$', TargetNumber):
                return TellError('Sending SMS to %s failed: no recipient number.' % Recipient)
            if not re.match('^\+?[0-9]+$', TargetNumber):
                return TellError('Sending SMS to %s failed: wrong recipient number.' % Recipient)
            if not SendSMSviaGSMmodem(TargetNumber, Message[0:160]):
                LOGGER.error('Sending SMS to %s failed while sending.' % Recipient)
            elif len(Message) > 160: 
                LOGGER.warning('SMS send to %s but truncated to 160 chars.' % Recipient)
            else:
                LOGGER.info('SMS send to %s.' % Recipient)

        else:
            return TellError('Unknown option (%s).' % OriMode)
                
    return ok;

##############################################################

def main():
    """
    Main function of smscon master utility. Does not return to caller.
    """
    global LOGGER, ourConf, daemon, settingsChangeCount

    if os.geteuid() != 0 and not XDEBUG:
        os.execvp("sudo", ["sudo"] + sys.argv[0:1]
                    + ["-usrid", str(os.geteuid()), "-grpid", str(os.getegid())]
                    + sys.argv[1:])
        # We get here on failure only!
        print 'ERROR: %s must run as "root". Try sudo command prefix to elevate your rights.' % NAME
        sys.exit(1)

    LOGGER = StartLogging('SMSCON', withConsole = True)

    ourConf = UserConfiguration()
    if ourConf.load(Silent = True) or ourConf.needsUpgrade():
        ourConf.upgrade()

    daemon = smscon_daemon()
    settingsChangeCount = 0

    ok = DoCommandLine()

    if settingsChangeCount:
        daemon.tell(daemon.TELL_RELOAD)

    sys.exit(IF(ok, 0, 1))

##############################################################
