# -*- coding: utf-8 -*-
# Sleeper Desktop Widget for Maemo 5
# Copyright (c) 2010 The Authors
#
# Authors:
#   Thomas Waelti <twaelti@gmail.com> (Original idea and implementation, settings/configuration, packaging)
#   Thomas Perl <thp@thpinfo.com> (Panucci support, code cleanup, testing&feedback)
#
# To run it manually:
# python2.5 /usr/lib/hildon-desktop/sleeper.py

APP_TITLE = "Sleeper"
APP_VERSION = "1.1"

import gtk
import pygtk
import hildon
import hildondesktop
import dbus
import os
import time
import gobject
import gconf
import osso
import string

APP_NAME = APP_TITLE.lower()
LOGO = '/usr/share/icons/hicolor/48x48/hildon/%s.png' % APP_NAME
GCONFPATH = "/apps/Maemo/tomch.com/%s" % APP_NAME

class MafwRenderer(object):
    DBUS_NAME = 'com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer'
    DBUS_PATH = '/com/nokia/mafw/renderer/gstrenderer'
    DBUS_INTF = 'com.nokia.mafw.renderer'

class Panucci(object):
    DBUS_NAME = 'org.panucci.panucciInterface'
    DBUS_PATH = '/panucciInterface'
    DBUS_INTF = 'org.panucci.panucciInterface'


class SleeperHomePlugin(hildondesktop.HomePluginItem):
    defaultPRESETS = [15,30,60]
    defaultLAYOUT = 'H'   #Or 'V' for a vertical bar
    defaultICON = True
    defaultSHOWBANNER = True
     
    BUTTON_SIZE = 64
    FIRSTRUN = True
    
    osso_context = osso.Context("com.tomch.%s" % APP_NAME, str(APP_VERSION), False)
    osso_rpc = osso.Rpc(osso_context)

    def __init__(self):
        hildondesktop.HomePluginItem.__init__(self)
        self.set_settings(True)
        self.connect("show-settings", self.show_options)
    
        self.bus = dbus.SessionBus()
        self.timer = None
        
        #Load the settings using GConf
        conf_client = gconf.client_get_default()
        conf_client.add_dir(GCONFPATH, gconf.CLIENT_PRELOAD_NONE)
        self.PRESETS=conf_client.get_list(GCONFPATH+"/presets",gconf.VALUE_INT)
        self.LAYOUT=conf_client.get_string(GCONFPATH+"/layout")
        self.ICON=conf_client.get_bool(GCONFPATH+"/drawicon")
        self.SHOWBANNER=conf_client.get_bool(GCONFPATH+"/showbanner")

        #Fallback to default on first launch
        if not self.PRESETS:
          self.PRESETS=self.defaultPRESETS
        if self.LAYOUT == None:
          self.LAYOUT=self.defaultLAYOUT    
        if self.ICON == None:
          self.ICON=self.defaultICON    
        if self.SHOWBANNER == None:
          self.SHOWBANNER=self.defaultSHOWBANNER  
        #Draw the widget
        self.draw()

    #Draw the widget
    def draw(self, widget=None):
        #Would be nice if we could destroy ourselves for complete re-draw after layout change
        #Alas, not working (yet?)
        if self.FIRSTRUN == False:
          self.box.destroy()
          self.container.queue_draw()
        #Layout direction
        self.container = gtk.VBox(True)
        if self.LAYOUT == 'V':
          self.box = gtk.VBox(True)
        else:
          self.box = gtk.HBox(True)
        
        #First the widget icon
        if self.ICON:
          self.image = gtk.Image()
          self.image.set_from_file(LOGO)         
          self.box.pack_start(self.image)    
        #Then a button for each minute value
        for minutes in self.PRESETS:
            button = gtk.ToggleButton(str(minutes))
            button.set_name('HildonButton-finger')
            button.set_size_request(self.BUTTON_SIZE, self.BUTTON_SIZE)
            button.connect('toggled', self.on_toggled, minutes)
            self.box.pack_start(button)

        self.container.add(self.box)
        self.box.show_all()
        if self.FIRSTRUN:
          self.FIRSTRUN = False
          self.add(self.container)
        self.container.show_all()
        self.set_opacity(0.5)
      
    #Log and/or display banner     
    def log(self, message, *args, **kwargs):
        print time.strftime('%H:%M:%S'), '- '+APP_TITLE+':', message % args
        #if kwargs.get('show_banner', False) or self.SHOWBANNER:
        if self.SHOWBANNER:
            banner = hildon.hildon_banner_show_information(self, '', APP_TITLE+': ' + message%args)
            banner.set_timeout(2000)

    #A minute value button was toggled, therefore disable all timers, start new one
    def on_toggled(self, button, minutes):
        # Abort any running timers
        if self.timer:
            gobject.source_remove(self.timer)
            self.timer = None
            self.log('Timer aborted.', show_banner=True)

        if button.get_active():
            # Button has been activated - deactivate other buttons
            for child in self.box.get_children():
                if isinstance(child, gtk.ToggleButton) and child != button:
                    child.set_active(False)

            self.timer = gobject.timeout_add_seconds(minutes*60, \
                    self.on_sleeptimer_fired, minutes, button)

            self.log('%d minutes to go - sleep well...', minutes, \
                    show_banner=True)

    #Time is up, pause media player and evtl. panucci
    def on_sleeptimer_fired(self, minutes, button):
        # Abort the timer + deactivate the button
        button.set_active(False)
        self.timer = None

        # Send pause signal to all known players
        self.pause_media_player()
        self.pause_panucci()

        self.log('Good night...', show_banner=True)

        return False

    def pause_media_player(self):
        try:
            gstrenderer = self.bus.get_object(MafwRenderer.DBUS_NAME, MafwRenderer.DBUS_PATH)
            gstrenderer.pause(dbus_interface=MafwRenderer.DBUS_INTF)
            self.log('Mediaplayer paused successfully.')
        except Exception, e:
            self.log('Mediaplayer already paused.')

    def pause_panucci(self):
        if self.bus.name_has_owner(Panucci.DBUS_NAME):
            panucci = self.bus.get_object(Panucci.DBUS_NAME, Panucci.DBUS_PATH)
            panucci.pause(dbus_interface=Panucci.DBUS_INTF)
            self.log('Panucci paused successfully.')
    
    #Setup the main options/settings dialogue (the one accessible through the tool icon when in Hildon Desktop Edit mode)
    def show_options(self, widget):
        dialog = gtk.Dialog("%s Options" % APP_TITLE, None, gtk.DIALOG_DESTROY_WITH_PARENT)

        settings_button = hildon.Button(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        settings_button.set_text("Settings", "Change how %s looks and works" % APP_TITLE)
        settings_button.set_alignment(0,0,0,0)
        settings_button.connect('clicked', self.show_settings)
                
        help_button = hildon.Button(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        help_button.set_text("Help", "Get help on how to use %s" % APP_TITLE)
        help_button.set_alignment(0,0,0,0)
        help_button.connect('clicked', self.open_help) 
        
        about_button = hildon.Button(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        about_button.set_text("About", "More about Author, Copyright and License")
        about_button.set_alignment(0,0,0,0)
        about_button.connect("clicked", self.show_about)

        donate_button = hildon.Button(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        donate_button.set_text("Donate", "Like the software? Support the effort :-)")
        donate_button.set_alignment(0,0,0,0)
        donate_button.connect('clicked', self.open_donate)

        line1 = gtk.HBox()
        line2 = gtk.HBox()        
        line1.pack_start(settings_button, True, True, 0)
        line1.pack_start(help_button, True, True, 0)
        line2.pack_start(about_button, True, True, 0)
        line2.pack_start(donate_button, True, True, 0)
        dialog.vbox.pack_start(line1, True, True, 0)  
        dialog.vbox.pack_start(line2, True, True, 0)  
         
        dialog.show_all()
        dialog.run()
        dialog.destroy()
    
    #Link to a help page on the maemo wiki    
    def open_help(self, widget): 
        self.osso_rpc.rpc_run_with_defaults("osso_browser", "open_new_window", ("http://wiki.maemo.org/"+APP_TITLE,))

    #Link to a donation page using PayPal    
    def open_donate(self, widget):
        self.osso_rpc.rpc_run_with_defaults("osso_browser", "open_new_window", ("https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=NJ4HVUZKB2CZY",))                
    
    #Setup the settings UI , store changed parameters  
    def show_settings(self, widget):
        dialog = gtk.Dialog("Settings", None, gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR)
        search_button = dialog.add_button(gtk.STOCK_SAVE, gtk.RESPONSE_OK)
        
        main_vbox = gtk.VBox()
        
        ts = hildon.TouchSelector()
        ts.add(main_vbox)
        
        hboxPresets = gtk.HBox()
        presetsInput = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        presetsInput.set_text(",".join(["%s" % el for el in self.PRESETS]))
        hboxPresets.pack_start(gtk.Label("Sleep timers"), False, False, 0)
        hboxPresets.pack_start(presetsInput)

        hboxCheckboxes = gtk.HBox()
        self.iconButton = hildon.CheckButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.iconButton.set_label("Show icon")
        if self.ICON:
          self.iconButton.set_active(True)
        hboxCheckboxes.pack_start(self.iconButton)

        self.bannerButton = hildon.CheckButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.bannerButton.set_label("Show banners")
        if self.SHOWBANNER:
          self.bannerButton.set_active(True)
        hboxCheckboxes.pack_start(self.bannerButton)
                
        h_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, None)
        h_button.set_label("Horizontal")
        h_button.set_mode(False)
        v_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, h_button)
        v_button.set_label("Vertical")
        v_button.set_mode(False)
        
        if self.LAYOUT == 'V':
           v_button.set_active(True)
           self.box = gtk.VBox(True)
        else:
           h_button.set_active(True)
           self.box = gtk.HBox(True)
               
        hBoxLayout = gtk.HBox()
        hBoxLayout.pack_end(h_button, True, True, 0)
        hBoxLayout.pack_end(v_button, True, True, 0)

        main_vbox.pack_start(hboxPresets)
        main_vbox.pack_start(hBoxLayout, True, True, 10)
        main_vbox.pack_start(hboxCheckboxes)      

        dialog.vbox.add(ts)
        dialog.show_all()
        response = dialog.run()
        
        #Store the old settings temporarily
        oldPRESETS = self.PRESETS
        oldICON = self.ICON
        oldSHOWBANNER = self.SHOWBANNER
        oldLAYOUT = self.LAYOUT
        
        #Parse the updated settings        
        if response == gtk.RESPONSE_OK:
           icon_buttons = h_button.get_group()
           for button in icon_buttons:
              selected = button.get_active()
              if selected == True:
                 label = button.get_label()
                 if label == "Horizontal":
                    self.LAYOUT = 'H'
                 else:
                    self.LAYOUT = 'V'
                 break
                 
           if (self.iconButton.get_active()):
             self.ICON = True
           else:
             self.ICON = False

           if (self.bannerButton.get_active()):
             self.SHOWBANNER = True
           else:
             self.SHOWBANNER = False

           try:
             self.PRESETS = presetsInput.get_text().split(",")
             self.PRESETS = [int(column) for column in self.PRESETS]           
           except Exception:
             self.PRESETS = self.defaultPRESETS
             self.log('Invalid presets. Reset to default values.')
           
           #Store the updated settings
           conf_client = gconf.client_get_default()
           conf_client.add_dir(GCONFPATH, gconf.CLIENT_PRELOAD_NONE)
           try:
             conf_client.set_list(GCONFPATH+"/presets", gconf.VALUE_INT, self.PRESETS)
             conf_client.set_string(GCONFPATH+"/layout", self.LAYOUT)
             conf_client.set_bool(GCONFPATH+"/drawicon", self.ICON)
             conf_client.set_bool(GCONFPATH+"/showbanner", self.SHOWBANNER)
           except Exception:
             self.log('Unable to save settings in GConf.')
           self.draw()
        
        if oldPRESETS != self.PRESETS or oldICON != self.ICON or oldLAYOUT != self.LAYOUT:
          banner = hildon.hildon_banner_show_information(self, '', 'Delete the widget and add it again to see the changes.')
          banner.set_timeout(2000)

        dialog.destroy()
    
    #Show the about dialog    
    def show_about(self, widget):
        dialog = gtk.AboutDialog()
        dialog.set_title("About")
        dialog.set_name(APP_TITLE)
        dialog.set_version(APP_VERSION)
        dialog.set_copyright("Copyright 2010 The authors")
        dialog.set_authors(["Tom Waelti <twaelti@gmail.com>","Thomas Perl <thp@thpinfo.com>"])
        dialog.set_logo(gtk.gdk.pixbuf_new_from_file(LOGO))
        dialog.set_comments("All logos and trademarks are property of their respective owners and are used for informational purposes only.")
        #dialog.set_license("GPL 3")
        dialog.set_license("""This program is free software: you can redistribute it and/or
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation, either version 3 of
  the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied 
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public 
  License along with this program.  If not, see
  <http://www.gnu.org/licenses/>.""")
        dialog.set_wrap_license(True)
        dialog.show_all()
        dialog.run()
        dialog.destroy() 

hd_plugin_type = SleeperHomePlugin

# The code below is just for testing purposes.
# It allows to run the widget as a standalone process.
if __name__ == "__main__":
    gobject.type_register(hd_plugin_type)
    obj = gobject.new(hd_plugin_type, plugin_id="plugin_id")
    obj.show_all()
    gtk.main()

