#include "dbmanager.h"
#include "global.h"
#include <QtDebug>
#include <QSqlQuery>
#include <QVariant>
#include <QStandardItem>
#include <QUuid>


DBManager::DBManager()
{
        initDB();
}

DBManager::~DBManager()
{
        closeDB();
}

void DBManager::initDB()
{
        bool create;

        QString dirPath(QDir::home().path());
        dirPath.append(QDir::separator()).append(DATA_FOLDER);

        QString filePath(dirPath);
        filePath.append(QDir::separator()).append(DB_FILENAME);

        create = !QDir::home().exists(filePath);

        if (!QDir::home().exists(dirPath)) {
                QDir::home().mkdir(dirPath);
        }

        db = QSqlDatabase::addDatabase("QSQLITE");
        db.setDatabaseName(filePath);
        openDB();

        if (create) {
                qDebug() << "\nDatabase empty. Creating tables";
                if (!createTables()) {
                        qCritical() << "\nError Initializing database! Exiting...";
                        exit(0);
                }
        }
}

bool DBManager::createTables()
{
        bool ret;
        QSqlQuery query;

        ret = query.exec("create table account ("
                         "id integer primary key autoincrement, "
                         "name text unique not null, "
                         "budget real, "
                         "password text)");

        if (!ret) {
                qDebug() << "\nError creating table account" << query.lastError();
        }

        ret = query.exec("create table month (year integer not null, "
                         "month integer not null, "
                         "account integer references account(id), "
                         "budget real not null, "
                         "primary key (year,month,account))");
        if (!ret) {
                qDebug() << "\nError creating table month" << query.lastError();
                return false;
        }

        ret = query.exec("create table concept ("
                         "name text primary key, "
                         "price real not null)");
        if (!ret) {
                qDebug() << "\nError creating table concept" << query.lastError();
                return false;
        }

        ret = query.exec("create table expense (id integer primary key autoincrement,"
                         "year integer, "
                         "month integer, "
                         "account integer, "
                         "day integer not null, "
                         "amount real not null, "
                         "concept text references concept(name), "
                         "description text, "
                         "uuid text, "
                         "foreign key (year,month,account) references month(year,month,account))");

        if (!ret) {
                qDebug() << "\nError creating table expense" << query.lastError();
        }

        ret = query.exec("create table scheduled (id integer primary key autoincrement,"
                         "account integer references account(id), "
                         "day integer, "
                         "amount real not null, "
                         "concept text references concept(name), "
                         "description text, "
                         "frecuency integer)");

        if (!ret) {
                qDebug() << "\nError creating table scheduled" << query.lastError();
        }

        return ret;
}

/*----------- Account operations -------------*/

bool DBManager::create(Account *a)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into account (id,name,budget) "
                                  "values (NULL,'%1', %2)")
                         .arg(a->name())
                         .arg(a->budget()));

        if (!ret) {
                qDebug() << "\nError adding account to the database" << query.lastError();
        }
        return ret;
}

bool DBManager::loadAccounts(QStandardItemModel *accounts)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select id,name,budget from account order by name collate nocase"));

        if (!ret) {
                qDebug() << "\nError loading accounts from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                QList<QStandardItem*> row;
                QStandardItem *item;
                item = new QStandardItem (query.value(0).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(1).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(2).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem ();
                item->setEditable(false);
                row.append(item);
                accounts->appendRow(row);
        }

        return true;
}

bool DBManager::update(Account *a)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update account set budget=%1 where id=%2")
                         .arg(a->budget())
                         .arg(a->id()));

        if (!ret) {
                qDebug() << "\nError updating account in database" << query.lastError();
        }
        return ret;
}

bool DBManager::remove(Account *a)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from scheduled where account=%1")
                         .arg(a->id()));

        if (!ret) {
                qDebug() << "\nError deleting account related scheduled from database" << query.lastError();
                return ret;
        }

        ret = query.exec(QString ("delete from expense where account=%1")
                         .arg(a->id()));

        if (!ret) {
                qDebug() << "\nError deleting account related expenses from database" << query.lastError();
                return ret;
        }

        ret = query.exec(QString ("delete from month where account=%1")
                         .arg(a->id()));

        if (!ret) {
                qDebug() << "\nError deleting account related months from database" << query.lastError();
                return ret;
        }

        ret = query.exec(QString ("delete from account where id=%1")
                         .arg(a->id()));

        if (!ret) {
                qDebug() << "\nError deleting account from database" << query.lastError();
                return ret;
        }

        return ret;
}

bool DBManager::exists(Account *a)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select * from account where name='%1'")
                         .arg(a->name()));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        return query.next();
}

bool DBManager::load(Account *a)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select name, budget from account where id=%1")
                         .arg(a->id()));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        if (!query.next()) {
                qDebug() << "Error: active account doesn't exist in database";
                return false;
        }

        a->setName(query.value(0).toString());
        a->setBudget(query.value(1).toDouble());
        return true;
}


/*----------- MonthData operations -------------*/

bool DBManager::create(MonthData *month)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec (QString("insert into month (year, month, account, budget) "
                                  "values (%1, %2, %3, %4)")
                          .arg(month->year())
                          .arg(month->month())
                          .arg(month->account()->id())
                          .arg(month->budget()));

        if (!ret) {
                qDebug() << "\nError adding month to database";
        }
        return ret;
}

bool DBManager::load(MonthData *month, bool expensesOnly)
{
        bool ret;
        QSqlQuery query;

        if (!expensesOnly) {
                ret = query.exec(QString ("select budget from month where year=%1 and month=%2 and account=%3")
                                 .arg(month->year())
                                 .arg(month->month())
                                 .arg(month->account()->id()));
                if (!ret) {
                        qDebug() << "\nError loading month budget from database" << query.lastError();
                        return ret;
                }

                if (query.next()) {
                        month->setBudget(query.value(0).toDouble());
                } else {
                        qDebug() << "\nError: Selected month doesn't exist";
                        return false;
                }
        }

        ret = query.exec(QString ("select id, day, amount, concept, description from expense "
                                  "where year=%1 and month=%2 and account=%3 order by day")
                         .arg(month->year())
                         .arg(month->month())
                         .arg(month->account()->id()));

        if (!ret) {
                qDebug() << "\nError loading month expenses from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                month->addExpense(new Expense(query.value(0).toInt(),
                                              query.value(1).toInt(),
                                              query.value(2).toDouble(),
                                              query.value(3).toString(),
                                              query.value(4).toString()));
        }

        return true;
}

bool DBManager::exists(MonthData *month)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select * from month where year=%1 and month=%2 and account=%3")
                         .arg(month->year())
                         .arg(month->month())
                         .arg(month->account()->id()));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        return query.next();
}

/*----------- Expense operations -------------*/

bool DBManager::create(int year, int month, int account, Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into expense (id, year, month, account, day, amount, concept, description, uuid) "
                                  "values (NULL, %1, %2, %3, %4, %5, '%6', '%7', '%8')")
                         .arg(year)
                         .arg(month)
                         .arg(account)
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(QUuid::createUuid().toString()));

        if (!ret) {
                qDebug() << "\nError adding expense to the database" << query.lastError();
        }

        return ret;
}

bool DBManager::update(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update expense set day=%1, amount=%2, concept='%3',"
                                  "description='%4' where id=%5")
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError updating expense in database" << query.lastError();
        }
        return ret;
}

bool DBManager::remove(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from expense where id=%1")
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError deleting expense from database" << query.lastError();
        }

        return ret;
}

/*----------- Concept operations -------------*/

bool DBManager::create(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into concept (name,price) "
                                  "values ('%1', %2)")
                         .arg(c->name())
                         .arg(c->price()));

        if (!ret) {
                qDebug() << "\nError adding concept to the database" << query.lastError();
        }
        return ret;
}

bool DBManager::loadConcepts(QStandardItemModel *concepts)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select name,price from concept order by name collate nocase"));

        if (!ret) {
                qDebug() << "\nError loading concepts from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                QList<QStandardItem*> row;
                QStandardItem *item;
                item = new QStandardItem (query.value(0).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(1).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                concepts->appendRow(row);
        }

        return true;
}

bool DBManager::update(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update concept set price=%1 where name='%2'")
                         .arg(c->price())
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError updating concept in database" << query.lastError();
        }
        return ret;
}

bool DBManager::remove(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from concept where name='%1'")
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError deleting concept from database" << query.lastError();
                return ret;
        }

        ret = query.exec(QString ("update expense set concept='' where concept='%1'")
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError disassociating expenses from removed concept" << query.lastError();
        }

        ret = query.exec(QString ("update scheduled set concept='' where concept='%1'")
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError disassociating scheduled expenses from removed concept" << query.lastError();
        }

        return ret;
}

bool DBManager::exists(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select * from concept where name='%1'")
                         .arg(c->name()));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        return query.next();
}

/*----------- Budget operations -------------*/

bool DBManager::updateBudget(int year, int month, int account, double budget)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update month set budget=%1 where year=%2 and month=%3 and account=%4")
                         .arg(budget)
                         .arg(year)
                         .arg(month)
                         .arg(account));
        if (!ret) {
                qDebug() << "\nError saving month budget to database" << query.lastError();
                return ret;
        }

        return true;
}

/*----------- Scheduled expense operations -------------*/

bool DBManager::createScheduled(int account, Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into scheduled (id, account, day, amount, concept, description, frecuency) "
                                  "values (NULL, %1, %2, %3, '%4', '%5', %6)")
                         .arg(account)
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(e->frecuency()));

        if (!ret) {
                qDebug() << "\nError adding scheduled expense to the database" << query.lastError();
        }

        return ret;
}

bool DBManager::removeScheduled(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from scheduled where id=%1")
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError deleting scheduled expense from database" << query.lastError();
        }

        return ret;
}

bool DBManager::loadScheduled(int account, QStandardItemModel *scheduled)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select id,day,amount,concept,description,frecuency "
                                  "from scheduled where account=%1 order by day")
                         .arg(account));

        if (!ret) {
                qDebug() << "\nError loading scheduled expenses from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                QList<QStandardItem*> row;
                QStandardItem *item;
                item = new QStandardItem (query.value(0).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(1).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(2).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(3).toString());
                item->setTextAlignment(Qt::AlignLeft | Qt::AlignVCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(4).toString());
                item->setTextAlignment(Qt::AlignLeft | Qt::AlignVCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(5).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (Expense::frecuencyAsString(query.value(5).toInt()));
                item->setTextAlignment(Qt::AlignLeft | Qt::AlignVCenter);
                item->setEditable(false);
                row.append(item);
                scheduled->appendRow(row);
        }

        return true;
}

bool DBManager::updateScheduled(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update scheduled set day=%1, amount=%2, concept='%3', "
                                  "description='%4', frecuency=%5 where id=%6")
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(e->frecuency())
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError updating scheduled in database" << query.lastError();
        }
        return ret;
}

/*---------------- summary related functions ---------------*/

bool DBManager::getDataInterval(int account, int &minYear, int &minMonth, int &maxYear, int &maxMonth)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select year, month from month where account=%1 order by year, month limit 0,1")
                         .arg(account));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        if (query.next()) {
                minYear = query.value(0).toInt();
                minMonth = query.value(1).toInt();
        } else {
                qDebug() << "\nError: No month stored in database";
                return false;
        }

        ret = query.exec(QString ("select year, month from month where account=%1 order by year desc , month desc limit 0,1")
                         .arg(account));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        if (query.next()) {
                maxYear = query.value(0).toInt();
                maxMonth = query.value(1).toInt();
        } else {
                qDebug() << "\nError: No month stored in database";
                return false;
        }
        return true;
}

bool DBManager::load(int account, SummaryData *summary)
{
        bool ret;
        QSqlQuery query;

        if (summary->startYear() == summary->endYear()) {
                ret = query.exec(QString ("select month.budget,sum(amount) from month left outer join expense "
                                          "on (month.year=expense.year and month.month=expense.month "
                                          "and month.account=expense.account) "
                                          "where month.account=%4 "
                                          "group by month.year,month.month,month.account "
                                          "having (month.year=%1 and month.month>=%2 and month.month<=%3)")
                                 .arg(summary->startYear())
                                 .arg(summary->startMonth())
                                 .arg(summary->endMonth())
                                 .arg(account));
        } else {
                ret = query.exec(QString ("select month.budget,sum(amount) from month left outer join expense "
                                          "on (month.year=expense.year and month.month=expense.month "
                                          "and month.account=expense.account) "
                                          "where month.account=%5 "
                                          "group by month.year,month.month,month.account "
                                          "having (month.year=%1 and month.month>=%2) or (month.year>%1 and month.year<%3) "
                                          "or (month.year=%3 and month.month<=%4)")
                                 .arg(summary->startYear())
                                 .arg(summary->startMonth())
                                 .arg(summary->endYear())
                                 .arg(summary->endMonth())
                                 .arg(account));
        }

        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                summary->addMonthData(query.value(0).toInt(),
                                      query.value(1).toDouble());
        }

        if (summary->startYear() == summary->endYear()) {
                ret = query.exec(QString ("select concept,sum(amount) from expense "
                                          "where ((year=%1 and month>=%2 and month<=%3)) "
                                          "and account=%4 "
                                          "group by concept order by concept collate nocase")
                                 .arg(summary->startYear())
                                 .arg(summary->startMonth())
                                 .arg(summary->endMonth())
                                 .arg(account));
        } else {
                ret = query.exec(QString ("select concept,sum(amount) from expense "
                                          "where ((year=%1 and month>=%2) "
                                          "or (year>%1 and year<%3) "
                                          "or (year=%3 and month<=%4)) "
                                          "and account=%5 "
                                          "group by concept order by concept collate nocase")
                                 .arg(summary->startYear())
                                 .arg(summary->startMonth())
                                 .arg(summary->endYear())
                                 .arg(summary->endMonth())
                                 .arg(account));
        }

        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                summary->addConcept(new Concept (query.value(0).toString(),
                                                 query.value(1).toDouble()));
        }

        return true;
}

bool DBManager::openDB()
{
        return db.open();
}

void DBManager::closeDB()
{
        db.close();
}
