#include "mainwindow.h"
#include "settingsdlg.h"
#include "helpdialog.h"
#include "Settings.h"
#include "SiBApplication.h"
#include "videofile.h"
#include "filelist.h"
#include "QScrollBar"
#include <QModelIndex>
#include "QAbstractKineticScroller"


MainWindow::MainWindow()
{
    QFont buttonFont;
    buttonFont.setPixelSize( 25 );

    int buttonWidth = 150;
    int buttonHeight = 70;
    int buttonLeftOffset = 15;

    m_playbutton = new QPushButton( tr( "Play" ), this );
    m_playbutton->setFont( buttonFont );
    m_playbutton->setGeometry( buttonLeftOffset, 40, buttonWidth, buttonHeight );
    m_playbutton->setEnabled( false );
    m_resumebutton = new QPushButton( tr( "Resume" ), this );
    m_resumebutton->setFont( buttonFont );
    m_resumebutton->setGeometry( buttonLeftOffset, 135, buttonWidth, buttonHeight );
    m_resumebutton->setEnabled( false );
    QPushButton* settingsbutton = new QPushButton( tr( "Folders" ), this );
    settingsbutton->setFont( buttonFont );
    settingsbutton->setGeometry( buttonLeftOffset, 230, buttonWidth, buttonHeight );
    QPushButton* helpButton = new QPushButton( tr( "Info" ), this );
    helpButton->setFont( buttonFont );
    helpButton->setGeometry( buttonLeftOffset, 325, buttonWidth, buttonHeight );

    m_fileList = new QTableView( this );
    m_fileList->setItemDelegate( new QItemDelegate( this ) );
    m_fileList->setGeometry( 190, 5, 605, 415 );

    m_fileList->setSelectionBehavior( QAbstractItemView::SelectRows );
    m_fileList->setSelectionMode( QAbstractItemView::SingleSelection );
    m_fileList->horizontalHeader()->setResizeMode( 0, QHeaderView::Stretch );
    m_fileList->horizontalHeader()->setResizeMode( 1, QHeaderView::ResizeToContents );
    m_fileList->horizontalHeader()->setResizeMode( 2, QHeaderView::ResizeToContents );
    m_fileList->verticalHeader()->hide();
    m_fileList->setShowGrid(false);

    connect( m_playbutton, SIGNAL( clicked() ),
             this, SLOT( playButtonPressed() ) );

    connect( m_resumebutton, SIGNAL( clicked() ),
             this, SLOT( resumeButtonPressed() ) );

    connect( settingsbutton, SIGNAL( clicked() ),
             this, SLOT( settingsButtonPressed() ) );

    connect( helpButton, SIGNAL( clicked() ),
             this, SLOT( helpButtonPressed() ) );

    connect( m_fileList, SIGNAL( doubleClicked( const QModelIndex & ) ),
        this, SLOT( fileSelected() ) );

    connect( m_fileList, SIGNAL( clicked( const QModelIndex & ) ),
        this, SLOT( cellClicked() ) );

    connect( m_fileList->horizontalHeader(), SIGNAL( sectionClicked(int) ),
             this, SLOT( resort( int ) ) );

    m_sortByColumn = 0;
    m_sortOrder = Qt::AscendingOrder;
}

MainWindow::~MainWindow()
{
}

// ------------------------------------------------------------------------------
// Called on key events. Derived from QWidget.
// ------------------------------------------------------------------------------
//
void MainWindow::keyPressEvent ( QKeyEvent* event )
{
	SiBApplication::instance()->handleKeyEvent( event );
}

void MainWindow::updateFileList( const QList<VideoFile*>& files )
{
    QStandardItemModel* model = new QStandardItemModel( 0, 0 );

    int fileCount = files.count();

    for ( int i = 0; i < fileCount; i++ )
    {
        QList<QStandardItem*> row;

        QStandardItem *fileNameItem = new QStandardItem( files[i]->file().fileName() );
        fileNameItem->setBackground( QBrush( QColor( ( 100 * i / fileCount ) << 16 | 50 ) ) );
        fileNameItem->setFlags( fileNameItem->flags() ^ Qt::ItemIsEditable );
        row.append( fileNameItem );

        int size = int( 0.5 + files[i]->file().size() / ( 1024 * 1024 ) );
        QStandardItem* sizeItem = new QStandardItem(  tr("%1 MB").arg( size ) );
        sizeItem->setBackground( QBrush( QColor( ( 100 * i / fileCount ) << 16 | 50 ) ) );
        sizeItem->setFlags( sizeItem->flags() ^ Qt::ItemIsEditable );
        row.append( sizeItem );

        QStandardItem* resumePosItem = new QStandardItem( tr("%1").arg( toHHMMSS( files[i]->resumePos() ) ) );
        resumePosItem->setBackground( QBrush( QColor( ( 100 * i / fileCount ) << 16 | 50 ) ) );
        resumePosItem->setFlags( resumePosItem->flags() ^ Qt::ItemIsEditable );
        row.append( resumePosItem );

        model->appendRow( row );
    }

    model->setHeaderData( 0, Qt::Horizontal, QVariant( tr( "File" ) ) );
    model->setHeaderData( 1, Qt::Horizontal, QVariant( tr( "Size" ) ) );
    model->setHeaderData( 2, Qt::Horizontal, QVariant( tr( "Resume" ) ) );

    m_fileList->setModel( model );

    QHeaderView* header = m_fileList->horizontalHeader();

    header->setResizeMode( 0, QHeaderView::Stretch );
    header->setResizeMode( 1, QHeaderView::ResizeToContents );
    header->setResizeMode( 2, QHeaderView::ResizeToContents );
}

void MainWindow::playButtonPressed()
{
    int currentRow = m_fileList->currentIndex().row();

    if ( !SiBApplication::instance()->isPlayerRunning() )
    {
        if ( currentRow >= 0 ) {
                SiBApplication::instance()->restartFile( currentRow );
        }
    }
    else
    {
        SiBApplication::instance()->stopPlayer();
    }
}

void MainWindow::resumeButtonPressed()
{
    fileSelected();
}

void MainWindow::settingsButtonPressed()
{
    SettingsDlg settingsDlg;
    settingsDlg.exec();
    SiBApplication::instance()->updateFileList();
}

void MainWindow::helpButtonPressed()
{
    HelpDialog help;
    help.exec();
}

void MainWindow::fileSelected()
{
    int row = m_fileList->currentIndex().row();

    if ( row >= 0 ) {
        SiBApplication::instance()->playFile( row );
    }
}

void MainWindow::cellClicked()
{
    int row = m_fileList->currentIndex().row();

    if ( row >= 0 ) {
        if ( SiBApplication::instance()->hasFileResumePos( row ) ) {
            m_resumebutton->setEnabled( true );
        } else {
            m_resumebutton->setEnabled( false );
        }
        m_playbutton->setEnabled( true );
    } else {
        m_playbutton->setEnabled( false );
        m_resumebutton->setEnabled( false );
    }
}

void MainWindow::resort( int column )
{
    if ( m_sortByColumn != column ) {
        m_sortByColumn = column;
        if ( column == 2 ) { // if sort by resume pos is selected for the first time sort descending
            m_sortOrder = Qt::DescendingOrder;
        } else {
            m_sortOrder = Qt::AscendingOrder;
        }
    } else {
        if ( m_sortOrder == Qt::AscendingOrder ) {
            m_sortOrder = Qt::DescendingOrder;
        } else {
            m_sortOrder = Qt::AscendingOrder;
        }
    }

    FileList::SortCriteria criteria = FileList::ESortByName;
    switch( column ) {
        case 0: // sort by name
            criteria = FileList::ESortByName;
            break;
        case 1: // sort by file size
            criteria = FileList::ESortBySize;
            break;
        case 2: // sort by resume pos
            criteria = FileList::ESortByResumePos;
            break;
        default: // won't happen
            break;
    }

    SiBApplication::instance()->sortFileList( criteria, m_sortOrder );
}

void MainWindow::playerStart()
{
	m_playbutton->setText( tr( "Stop" ) );
	m_resumebutton->setEnabled( false );
}

void MainWindow::playerStop()
{
	m_playbutton->setText( tr( "Start" ) );
#if 0
	cellClicked( m_fileList->currentRow(), m_fileList->currentColumn() );
#endif
}

QString MainWindow::toHHMMSS( unsigned int tenthOfASec )
{
    unsigned int secs = tenthOfASec / 10;
    unsigned int hoursFrac = secs / 3600;
    secs -= hoursFrac * 3600;
    unsigned int minsFrac = secs / 60;
    secs -= minsFrac * 60;
    unsigned int secsFrac = secs;
    return QString( "%1:%2:%3" ).arg( hoursFrac, 2, 10, QChar( '0' ) ).arg( minsFrac, 2, 10, QChar( '0' ) ).arg( secsFrac, 2, 10, QChar( '0' ) );
}
