/*
 * This file is part of sharing-plugin-blipfoto
 *
 * Copyright (C) 2010 Dave Elcock. All rights reserved.
 *
 * This code is licensed under a MIT-style license, that can be
 * found in the file called "COPYING" in the root directory.
 *
 */

#ifndef __BLIP_P_H__
#define __BLIP_P_H__

#include <glib.h>

G_BEGIN_DECLS

/**
 * Builds up the necessary url to access the required resource.
 * Prefer one of the ready-made methods like BlipGetToken(), as
 * they are likely to be less error-prone.
 *
 * @param [in] resource
 *   The resource to access
 * @return
 *   The returned string must be freed with g_free().
 *   The returned string is NULL if there were problems with the
 *   arguments, or the resulting string was too long.
 */
gchar* blip_build_api_string(const gchar* resource, ...);


/**
 * Given a valid blipfoto.com api url, this fetches the raw response
 * @param [in] url
 *   A Blipfoto api url
 * @param [out, optional] oError
 *   If you pass in a non-null error pointer, any error will be
 *   recorded there. You must then free its resources with
 *   blip_error_free().
 * @return
 *   Returned string must be freed with g_free().
 */
gchar* blip_fetch_response(const gchar* url, BlipError * oError);

/**
 * Given some data, return the md5 checksum.
 * @return
 *   Returned string must be freed with g_free().
 */
gchar* blip_md5(BlipFoto blip, const void* data, size_t size);

/**
 * Generates a nonce for url signing.
 * @return
 *   Returned string must be freed with g_free().
 */
gchar* blip_generate_nonce(BlipFoto blip);

/**
 * Generates the signature for a url, consisting of
 * timestamp, nonce, auth token and signature.
 * @return
 *   Returned string must be freed with g_free().
 */
gchar* blip_generate_signature(BlipFoto blip);

gchar* blip_sign_url(BlipFoto blip, const gchar* url);

G_END_DECLS

#endif // __BLIP_P_H__
