#include "blip_defines.h"
#include "blip_error.h"

struct BlipError_s
{
    BlipErrorDomain domain;
    gint code;
    gchar* message;
};

static const gchar* DomainNames[]= {
    "Blipfoto library",
    "Blipfoto API",
    "Blipfoto XML",
    "XML subsystem",
    "Transport subsystem",
    "",
};

const gchar* blip_error_domain_name(BlipError error)
{
    BlipErrorDomain domain= blip_error_domain(error);
    if (domain <= BLIP_DOMAIN_NONE)
        return DomainNames[domain];
    else
        return "Undefined domain";
}

BlipError CreateCommon(BlipErrorDomain domain,
                       gint code)
{
    BlipError error= g_try_malloc(sizeof(struct BlipError_s));
    if (error)
    {
        error->domain= domain;
        error->code= code;
    }
    return error;
}

BlipError blip_error_create(BlipErrorDomain domain,
                            gint code,
                            const gchar* message)
{
    BlipError error= CreateCommon(domain, code);
    if (error)
        error->message= g_strdup(message);
    return error;
}

BlipError blip_error_create_no_copy(BlipErrorDomain domain,
                                    gint code,
                                    gchar* message)
{
    BlipError error= CreateCommon(domain, code);
    if (error)
        error->message= message;
    return error;
}

void blip_error_free(BlipError error)
{
    if (error)
    {
        g_free(error->message);
        g_free(error);
    }
}

gboolean blip_error_match(BlipError       error,
                          BlipErrorDomain domain,
                          ...)
{
    gboolean result= FALSE;
    if (error && error->domain == domain)
    {
        gint code;
        va_list ap;
        va_start(ap, domain);
        code= va_arg(ap, gint);
        while (!result && code != BLIP_ERROR_MATCH_END)
        {
            if (error->code == code)
                result= TRUE;
            else
                code= va_arg(ap, gint);
        }
        va_end(ap);
    }
    return result;
}

BlipErrorDomain blip_error_domain(BlipError error)
{
    if (error == NULL) return BLIP_DOMAIN_NONE;
    return error->domain;
}

gint blip_error_code(BlipError error)
{
    if (error == NULL) return BLIP_ERROR_NONE;
    return error->code;
}

const gchar* blip_error_message(BlipError error)
{
    if (error == NULL) return NULL;
    return error->message;
}
