#include <glib/gprintf.h>

#include "blip_entry.h"
#include "blip_connection.h"

struct BlipEntry_s
{
    /**
     * The image to upload. Note that the maximum allowed filesize of
     * the uploaded image may depend on the status of the
     * authenticated contributor.
     *
     * @required No
     */
    gchar * image_fullpath;
    /**
     * To use the contributor's most recently uploaded image, set this
     * parameter to '1'. This might happen after an image was uploaded
     * successfully but there was a problem with the entry data, or if
     * the post/image resource was used to create a 2-step upload
     * process.
     *
     * If an image is supplied to the image_upload parameter, this
     * parameter is ignored.
     *
     * @required No
     */
    gboolean use_previous_image;
    /**
     * The title of the entry. If omitted, any IPTC Headline
     * information within the image is used instead.
     * @required No
     */
    gchar* title;
    /**
     * The description of the entry. The description may contain the
     * following custom tags:[b]...[/b], [u]...[/u], [i]...[/i],
     * [url=...]...[/url]. If omitted, any IPTC Description
     * information within the image is used instead.
     *
     * @required No
     */
    gchar* description;
    /**
     * A comma separated list of tags to assign to the entry. If
     * omitted, any IPTC Keywords information within the image is used
     * instead.
     *
     * @required No
     */
    gchar* tags;
    /**
     * The entry date in YYYY-MM-DD format. If omitted, the EXIF Date
     * within the image is used instead.
     *
     * @required No
     */
    gchar* date;
};

BlipEntry blip_entry_create()
{
    BlipEntry params;
    params= (BlipEntry) g_try_malloc0(sizeof(struct BlipEntry_s));
    return params;
}

void blip_entry_free(BlipEntry entry)
{
    if (entry)
    {
        g_free(entry->image_fullpath);
        g_free(entry->title);
        g_free(entry->description);
        g_free(entry->tags);
        g_free(entry->date);
    }	
}

void blip_entry_set_image_full_path(BlipEntry entry,
                                    const gchar* fullpath)
{
    if (entry)
    {
        g_free(entry->image_fullpath);
        entry->image_fullpath= g_strdup(fullpath);
    }
}

void blip_entry_set_title(BlipEntry entry,
                          const gchar* title)
{
    if (entry)
    {
        g_free(entry->title);
        entry->title= g_strdup(title);
    }
}

void blip_entry_set_description(BlipEntry entry,
                                const gchar* description)
{
    if (entry)
    {
        g_free(entry->description);
        entry->description= g_strdup(description);
    }
}

void blip_entry_set_tags(BlipEntry entry,
                         const gchar* tags)
{
    if (entry)
    {
        g_free(entry->tags);
        entry->tags= g_strdup(tags);
    }
}

void blip_entry_set_date(BlipEntry entry,
                         guint year,
                         guint month,
                         guint day)
{
    if (entry)
    {
        if (!entry->date)
        {
            entry->date= (gchar*) g_try_malloc0(sizeof(gchar) * 11);
            if (!entry->date)
                return;
        }
        g_sprintf(entry->date,
                  "%04d-%02d-%02d",
                  year,
                  month,
                  day);
    }
}

void blip_entry_set_connection_options(BlipEntry entry,
                                       BlipConnection connection)
{
    if (entry)
    {
        blip_connection_post_text(connection, "entry_title", entry->title);
        blip_connection_post_text(connection, "entry_description", entry->description);
        blip_connection_post_text(connection, "entry_tags", entry->tags);
        blip_connection_post_text(connection, "entry_date", entry->date);
        if (entry->image_fullpath)
        {
            blip_connection_post_file(connection, "image_upload", entry->image_fullpath);
        }
        else
        {
            blip_connection_post_text(connection, "use_previous_image",
                                      entry->use_previous_image ? "1" : "0");
        }
    }
}
