#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
===============================================================
    Contains definition of class Structure
===============================================================
"""

__author__ = "Mezhenin Artoym <mezhenin@cs.karelia.ru>"
__version__ = "$Revision: 19 $"
__date__ = "$Date: 2010/02/07 $"
__copyright__ = ""
__license__ = "GPLv2"

import sys
#sys.path.append("..")

from ScriboExc import WarningExc
from SqlDriver import sql_driver


class Structure(object):
    """
        Class Structure is base class for Message, Account and Profile.
    It contains common methods and attributes to work with any table in 
    database. 
    
    @exception you must use unicode strings as params to all methods
    @author Mezhenin Artoym <mezhenin@cs.karelia.ru> 
    @version 0.4
    @date 2010/01/12
    """


    def __init__(self, table, id=None):
        """
            Common __init__ for classes that disctibes some table in DB. 
        Table with that name must exist, otherwise excaption will be
        raised. if there is no id given, new record will be 
        created. So we can access to stored data and create new records.

        @param table (string) Table name in database. It will be saved
                              and used to create queries to database.
        @param id (int) id of structure in table. It will be 
                            saved and used to create queries.
        @return None
        @exception exceptions from sqlite database
        """

        self._table = table

        if not id:
            id = 1 + sql_driver.select_cmd("SELECT max(id) FROM " + table)
            sql_driver.commit_cmd("INSERT INTO %s (id) VALUES (%d)" %
                                  (table, id,))

            self._id = int(id)
            self.misc = {}

        else:
            self._id = int(id)
            try:
                self._get_field("id")
            except WarningExc:
                raise WarningExc("Structure __ini__", "no such record in DB")



    def get_id(self):
        """
        Get private attribute _id

        @return (string) id, None

        """
        return self._id

    id = property(get_id)

    def __eq__(self, other):
        """
        Compare this class with other object.

        @param other  object
        @return (bool) if the types and id match, return True, 
                       otherwise False
        """

        if type(self) == type(other) and self.id == other.id:
            return True

        return False


    def _get_field(self, field):
        """
        Get field from DB. 

        @param field (string) field name 
        @return (string) field value
        """

        query = "SELECT %s FROM %s WHERE id=%d" % \
                 (field, self._table, self._id)
        return sql_driver.select_cmd(query)


    def _set_field(self, field, value):
        """
        Change field value in DB.

        @param field  (string) field name 
        @param value (string) new value
        @return None 
        """
        query = "UPDATE %s SET %s=? WHERE id=%d" % \
                 (self._table, field, self._id)
        sql_driver.commit_cmd(query, (value,))


    def get_misc(self):
        """
        Get misc value from table.
         
        @return (dict) dictionary with misc data
        """
        tmp = eval(self._get_field("misc"))
        if tmp == None:
	    tmp = {}
	return tmp

    def set_misc(self, value):
        """
        Change misc value in table.
        
        @param value (dict) dictionary with misc data
        @return None
        """

        if type(value) != dict:
            raise WarningExc("Structure setter", "misc value should be dict")
        self._set_field("misc", repr(value))


    ## property
    misc = property(get_misc, set_misc)

    def _del_struct(self):
        """
        Delete record from DB.
        
        @return None
        """

        query = "DELETE FROM %s WHERE id=%d" % (self._table, self._id,)
        sql_driver.commit_cmd(query)

        self._id = None
        self._table = None


    def set_option(self, name, value):
        """
        save additional options for this record in database.
        
        @param name (string) option name, must be unique. You should use prefix
               like "ui_<name u like>" to prevent errors.
        @param value (representable type) value of option. should be int, 
               string, float, list, typle or dict.
        @return None 
        """
        misc = self.misc
        misc[name] = value
        self.misc = misc


    def get_option(self, name):
        """
        get additional options for this record from database.
        
        @param name (string) name of existiong option
        @return value of option. if you set list to this option name, this 
                function will return list.
        """
        return self.misc[name]


    def _copy_fields(self, field_list):
        """
        copy fields from current record to another record.
        
        @param field_list (list) list with field names (strings) that we should 
                                 copy
        @return (int) id of new record  
        """
        fields = ''
        for i in field_list:
            fields += i + ', '

        fields = '(' + fields[:-2] + ')'

        query = "INSERT INTO " + self._table + ' ' + fields + ' SELECT ' + \
                fields[1:-1] + ' FROM ' + self._table + ' WHERE id=' + \
                str(self._id)

        sql_driver.commit_cmd(query)

        id = sql_driver.select_cmd("SELECT max(id) FROM " + self._table)
        return int(id)

