#!/usr/bin/env python

"""
===============================================================
    Contains deifinition of class Account and id's for field
service_id in database (table Accounts)
===============================================================
"""

__author__ = "Mezhenin Artoym <mezhenin@cs.karelia.ru>"
__version__ = "$Revision: 20 $"
__date__ = "$Date: 2010/02/07 $"
__copyright__ = ""
__license__ = "GPLv2"

import sys

from ScriboExc import WarningExc
from Structure import Structure
from Message import Message, POST, INBOX, OUTBOX
from SqlDriver import sql_driver, SQL_ACCOUNT_FIELDS



class Account(Structure):

    """
        Class Account contain information about one account on some service 
    provider and list of post and comments.
    
    @exception WarningExc All methods can raise WarningExc.
    @author Mezhenin Artoym <mezhenin@cs.karelia.ru> 
    @version 0.2
    @date 2009/13/08 
    """

    def __new__(cls, id=None):
        """
        Account is base class for objects with secvices API. 
        We must choose object type before memory allocation.
        """
        if id:
            """
            if id given it meens we need to load existing account 
            """
            from services.Dispatcher import dispatcher
            service_id = sql_driver.select_cmd("SELECT service_id FROM "\
                                               "Accounts WHERE id=?", (id,))
            cls = dispatcher.get_type(service_id)

        return super(Account, cls).__new__(cls)


    def __init__(self, id):
        """
        Constuctor for class.
        
        @param id (string) id of account in table.
        """

        super(Account, self).__init__("Accounts", id)
        if not id:
            self.group = False
            self.friends = []
            self.readers = []
            self.journals = []


    """
    ===============================================================
        Getter methods
    ===============================================================
    """

    def get_service_id(self):
        """
        Get field from Accounts table.

        @return (string) service id
        """
        return self._get_field("service_id")

    def get_username(self):
        """
        Get field from Accounts table.

        @return (string) username
        """
        return self._get_field("username")

    def get_passwd(self):
        """
        Get field from Accounts table.

        @return (string) password
        """
        return self._get_field("passwd")

    def get_nick(self):
        """
        Get field from Accounts table.

        @return (string) nickname on this account
        """
        return self._get_field("nick")

    def get_path_to_ava(self):
        """
        Get field from Accounts table.

        @return (string) path to avatar 
        """
        return self._get_field("path_to_ava")

    def get_date_of_birth(self):
        """
        Get field from Accounts table.

        @return (string) user birthdate
        """
        return self._get_field("date_of_birth")

    def get_city(self):
        """
        Get field from Accounts table.

        @return (string) city 
        """
        return self._get_field("city")

    def get_prof(self):
        """
        Get field from Accounts table.

        @return (Profile) profile for this account or None
        """
        from Profile import Profile

        res = self._get_field("prof")
        if res:
            return Profile(int(res))

        return None


    def get_date(self):
        """
        Get field from Accounts table.

        @return (string) date of last update
        """
        return self._get_field("date")


    def get_group(self):
        """
        Get field from Accounts table.

        @return (string) type of journal
        """
        return bool(self._get_field("is_group"))


    def get_friends(self):
        """
        Get field from Accounts table.

        @return (list) friends
        """

        return eval(self._get_field("friends"))


    def get_readers(self):
        """
        Get field from Accounts table.

        @return (list) readers
        """
        return eval(self._get_field("readers"))

    def get_journals_list(self):
        """
        Get field from Accounts table.

        @return (list) journals, to which this user can write posts
        """
        return eval(self._get_field("journals"))


    def get_posts(self):
        query = "journal='%s' AND type=%d" % (self.id, POST)
        posts = Message.find(fetch_one=False, where=query)
        return posts


    def get_inbox(self):
        query = "journal='%s' AND type=%d" % (self.id, INBOX)
        inbox = Message.find(fetch_one=False, where=query)
        return inbox


    def get_outbox(self):
        query = "poster='%s' AND type=%d" % (self.id, OUTBOX)
        outbox = Message.find(fetch_one=False, where=query)
        return outbox


    """ 
    ===============================================================
        Setter methods
    ===============================================================
    """

    def set_service_id(self, value):
        """
        Change field value in Accounts table.

        @param value (string) service id
        @return None
        """
        self._set_field("service_id", value)

    def set_username(self, value):
        """
        Change field value in Accounts table.

        @param value (string) username
        @return None
        """
        self._set_field("username", value)

    def set_passwd(self, value):
        """
        Change field value in Accounts table.

        @param value (string) password 
        @return None
        """
        self._set_field("passwd", value)

    def set_nick(self, value):
        """
        Change field value in Accounts table.

        @param value (string) nickname
        @return None
        """
        self._set_field("nick", value)

    def set_path_to_ava(self, value):
        """
        Change field value in Accounts table.

        @param value (string) path to avatar
        @return None
        """
        self._set_field("path_to_ava", value)

    def set_date_of_birth(self, value):
        """
        Change field value in Accounts table.

        @param value (string) birthdate
        @return None
        """
        self._set_field("date_of_birth", value)

    def set_city(self, value):
        """
        Change field value in Accounts table.

        @param value (string) city
        @return None
        """
        self._set_field("city", value)

    def set_prof(self, value):
        """
        Change field value in Accounts table.

        @param value (Profile) profile for this account 
        @return None
        """
        from Profile import Profile

        if type(value) == Profile:
            self._set_field("prof", value.id)
        elif value == None:
            self._set_field("prof", None)
        else:
            raise WarningExc("Account setter",
                             "value is not None or Profile")


    def set_date(self, value):
        """
        Change field value in Accounts table.

        @param value (string) date of last update
        @return None
        """
        self._set_field("date", value)

    def set_group(self, value):
        """
        Change field value in Accounts table.

        @param value (bool) type of journal
        @return None
        """
        if type(value) == bool:
            self._set_field("is_group", value)
        else:
            raise WarningExc("Account setter",
                             "value is not bool")

    def set_readers(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) readers 
        @return None
        """
        if type(value) != list:
            raise WarningExc("Account setter", "readers value should be list")
        self._set_field("readers", repr(value))

    def set_friends(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) friends 
        @return None
        """
        if type(value) != list:
            raise WarningExc("Account setter", "friends value should be list")
        self._set_field("friends", repr(value))


    def set_journals_list(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) journals, to which this user can write posts
        @return None
        """
        if type(value) != list:
            raise WarningExc("Account setter", "journals value should be list")
        self._set_field("journals", repr(value))


    """
    ===============================================================
        Init getter/setter  methods
    ===============================================================
    """

    ## property
    service_id = property(get_service_id, set_service_id)
    ## property
    username = property(get_username, set_username)
    ## property
    passwd = property(get_passwd, set_passwd)
    ## property
    nick = property(get_nick, set_nick)
    ## property
    path_to_ava = property(get_path_to_ava, set_path_to_ava)
    ## property
    date_of_birth = property(get_date_of_birth, set_date_of_birth)
    ## property
    city = property(get_city, set_city)
    ## property
    prof = property(get_prof, set_prof)
    ## property
    date = property(get_date, set_date)
    ## property
    friends = property(get_friends, set_friends)
    ## property
    group = property(get_group, set_group)
    ## property
    readers = property(get_readers, set_readers)
    ## property
    journals_list = property(get_journals_list, set_journals_list)
    ## property
    posts = property(get_posts)
    ## property
    inbox = property(get_inbox)
    ## property
    outbox = property(get_outbox)

    """
    ===============================================================
        Other methods
    ===============================================================
    """

    def add_post(self, msg):
        """
        Add post to this account.
        
        @param msg (Message) new post 
        @return None
        @exception (WarningExc) add new post to group without poster id in msg 
        """

        if type(msg) != Message:
            raise WarningExc("Add post", "msg should be Message instance")

        msg.journal = self
        if not self.group:
            msg.poster = self
        elif not msg.poster:
            raise WarningExc("Add post", "adding in group, poster id required")

        msg.type = POST
        msg.read = False
        msg.parent = None


    def del_account(self):
        """
        Delete account and all his posts and comments from DB
        
        @return None
        """

        if self.is_default():
            self.prof.def_account = None

        self.del_posts()

        self._del_struct()

    def del_posts(self):
        """
        Delete all posts from some journal
        
        @return None
        """
        query = 'SELECT id FROM Messages WHERE journal=? AND type=?'
        id_list = sql_driver.select_mcmd(query, (self._id, POST))

        for i in id_list:
            Message(i[0]).del_msg_tree()


    def set_default(self):
        """
        Set this account, account by default in profile
         
        @return None 
        @exception WarningExc no profile
        """

        prof = self.prof
        if prof:
            prof.def_account = self
        else:
            raise WarningExc("Account set_default",
                             "Account can't be set")


    def is_default(self):
        """
        Is this accout, account by default?

        @return (bool) True if this is accout by default, otherwise False
        """
        prof = self.prof
        if prof and prof.def_account == self:
            return True

        return False


    def add_friend(self, user):
        """
        Add new friend to this account. Make changes only in local cache.
        
        @param user (child of Account) new friend
        @return (bool) True if account was successfully added, False otherwise 
        """

        list = self.friends
        list.append(user.id)
        self.friends = list
        return True


    def remove_friend(self, user):
        """
        Remove friend from this account. Make changes only in local cache. 
        
        @param user (child of Account) our friend, if friend is not in our 
                    list of friends nothing happens

        @return (bool) True if account was successfully removed, False otherwise        
        """

        list = self.friends
        try:
            list.remove(user.id)
        except ValueError:
            return False

        self.friends = list
        return True


    '''
    def add_reader(self, user):
        list = self.readers
        list.append(user.id)
        self.readers = list
    '''


    @staticmethod
    def find(service, username, profile=None):
        """
        find account by some values
        
        @return (child of Account) first account that match query or None, if
                no account match
        """

        req = 'SELECT id FROM Accounts WHERE service_id="%s" AND username=?' % \
              (service.__name__,)

        if profile:
            req = req + ' AND prof="%d"' % profile.id

        try:
            res = sql_driver.select_cmd(req, (username,))
            return service(res)
        except WarningExc:
            return None


    def add_inbox(self, msg):
        """
        Add inbox to this account.
        
        @param msg (Message) new inbox message 
        @return None
        @exception (WarningExc) poster is not set
        """

        if type(msg) != Message:
            raise WarningExc("Add inbox", "msg should be Message instance")

        if not msg.poster:
            raise WarningExc("Account add_inbox",
                             "message has no poster field")
        msg.journal = self
        msg.type = INBOX
        msg.read = False


    def add_outbox(self, msg):
        """
        Add outbox to this account.
        
        @param msg (Message) new outbox message 
        @return None
        @exception (WarningExc) journal is not set
        """
        if type(msg) != Message:
            raise WarningExc("Add outbox", "msg should be Message instance")

        if not msg.journal:
            raise WarningExc("Account add_outbox",
                             "message has no journal field")
        msg.poster = self
        msg.type = OUTBOX
        msg.read = True


    def copy(self):
        """
        Make full copy of this Account.
        
        @return (child of Account) new account with exact values
        """

        return Account(self._copy_fields(SQL_ACCOUNT_FIELDS))


    def get_journals(self):

        list = self.journals_list
        accounts = []

        for i in list:
            accounts.append(Account(i))

        return accounts
