/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        panel.cpp ( Live Once Library, C++ )
 *
 * COMMENTS:
 */

#include "panel.h"

#include "rand.h"
#include "gfxengine.h"

#include <ctype.h>
#include <stdlib.h>
#include <memory.h>

PANEL::PANEL(int w, int h, bool recordhistory)
{
    int		i;

    myX = 0;
    myY = 0;
    myW = w;    
    myH = h;
    myIndent = 0;
    myRightMargin = 1;
    myRecordHistory = recordhistory;
    
    myLines = new char *[h];
    for (i = 0; i < h; i++)
    {
	myLines[i] = new char [w+10];
	myLines[i][0] = '\0';
    }

    myAttrMap = new ATTR_NAMES *[h];
    for (i = 0; i < h; i++)
    {
	myAttrMap[i] = new ATTR_NAMES [w];
    }

    myCurLine = 0;
    myCurPos = 0;

    setAttr(ATTR_NORMAL);
}

PANEL::~PANEL()
{
    int		i;

    for (i = 0; i < myH; i++)
	delete [] myLines[i];
    delete [] myLines;
    for (i = 0; i < myH; i++)
	delete [] myAttrMap[i];
    delete [] myAttrMap;

    clearHistory();
}

void
PANEL::clearHistory()
{
    int		i;

    for (i = 0; i < myHistory.entries(); i++)
	delete [] myHistory(i);
    myHistory.clear();
}

int
PANEL::getHistoryLine()
{
    return myCurLine + myHistory.entries();
}

void
PANEL::scrollToHistoryLine(int line)
{
    // Determine how many lines of history we need to scroll into
    // our window...
    int		netnew;

    netnew = getHistoryLine() - line;
    if (netnew <= 0)
    {
	// Either trying to go forward in time or a no-op
	return;
    }

    while (netnew--)
    {
	scrollDown();
    }
}

void
PANEL::setCurrentLine(const char *text)
{
    // Clear out the current line.
    myLines[myCurLine][0] = '\0';
    myCurPos = 0;

    // And append normally
    appendText(text);
}

void
PANEL::appendText(const char *text, int linecount)
{
    // Start copying text into myLines[myCurLine][myCurPos], doing
    // wordwrap at myW.
    const char		*start;
    char		*dst;
    int			 dstpos;

    start = text;
    dst = myLines[myCurLine];
    dstpos = myCurPos;

    while (*text)
    {
	// Check for control characters.
	if (*text == '\n')
	{
	    dst[dstpos] = '\0';
	    linecount++;
	    newLine();
	    if (myH <= 2)
	    {
		// Cannot scroll with prompt!
		linecount = 0;
	    }
	    else if (linecount >= myH-2) 
	    {
		awaitKey();
		linecount = 0;
	    }
	    appendText(text+1, linecount);
	    return;
	}
	
	// Check to see if we have hit myW.
	// Note we always scroll one character early!  This is more
	// visually pleasing than filling to the boundrary.
	if (dstpos >= myW - myRightMargin)
	{
	    // If this is a space, eat all succeeding spaces.
	    if (ISSPACE(*text))
	    {
	    }
	    else
	    {
		const char *fallback;

		fallback = text;
		
		// Not a space, we want to roll back until the next space.
		while (text > start)
		{
		    text--;
		    dstpos--;
		    if (ISSPACE(*text))
			break;
		}
		if (text == start)
		{
		    // Oops, no breakpoint! Hyphenate arbitrarily.
		    // Restore ourselves to our fallback position.
		    // The exception is if we did not start with
		    // an empty line!  In that case, a new line
		    // may suffice.
		    if (dstpos == myIndent)
		    {
			dstpos += (int)(fallback - text);
			text = fallback;
		    }
		}
	    }

	    // Write in a null and new line.
	    dst[dstpos] = '\0';

	    // Go forward in text removing all spaces.
	    while (*text && ISSPACE(*text))
	    {
		text++;
	    }
	    linecount++;
	    newLine();
	    if (myH <= 2)
	    {
		// Cannot scroll with prompt!
		linecount = 0;
	    }
	    else if (linecount >= myH-2) 
	    {
		awaitKey();
		linecount = 0;
	    }
	    // Append the remainder.
	    appendText(text, linecount);
	    return;
	}

	// All good for normal addition.
	dst[dstpos] = *text;
	dst[dstpos+1] = '\0';

	dstpos++;
	text++;
	myCurPos = dstpos;
    }
}

void
PANEL::awaitKey()
{
    appendText("-- MORE --\n");
    redraw();
    gfx_update();
    gfx_getKey();
}

void
PANEL::getString(const char *prompt, char *buf, int maxlen)
{
    appendText(prompt);

    redraw();
    gfx_getString(myCurPos + myX, myCurLine + myY, myAttrMap[myCurLine][myCurPos], buf, maxlen);
    // Ensure the buffer is inside our panel's memory!
    appendText(buf);
    // Always do a new line!
    newLine();
}

void
PANEL::setAttr(ATTR_NAMES attr)
{
    fillRect(0, 0, myW, myH, attr);
}

void
PANEL::fillRect(int x, int y, int w, int h, ATTR_NAMES attr)
{
    int		i, j;

    // Clip...
    if (x < 0)
    {
	w += x;
	x = 0;
    }
    if (y < 0)
    {
	h += y;
	y = 0;
    }

    for (i = y; i < y + h; i++)
    {
	if (i >= myH)
	    break;
	for (j = x; j < x + w; j++)
	{
	    if (j >= myW)
		break;

	    myAttrMap[i][j] = attr;
	}
    }
}

void
PANEL::newLine()
{
    myCurLine++;
    if (myCurLine == myH)
    {
	scrollUp();
    }
    myCurPos = 0;
    int		i;
    for (i = 0; i < myIndent; i++)
	appendText(" ");
}

void
PANEL::clear()
{
    int		i;

    for (i = 0; i < myH; i++)
    {
	myLines[i][0] = '\0';
    }
    myCurPos = 0;
    myCurLine = 0;
}

void
PANEL::move(int x, int y)
{
    myX = x;
    myY = y;
}

void
PANEL::redraw()
{
    int		x, y;

    for (y = 0; y < myH; y++)
    {
	if (y + myY >= 0 && y + myY < SCR_HEIGHT)
	{
	    for (x = 0; x < myW; x++)
	    {
		// Hit end of line.
		if (!myLines[y][x])
		    break;

		if (x + myX < 0 || x + myX >= SCR_WIDTH)
		    continue;
		gfx_printchar(x + myX, y + myY, myLines[y][x], myAttrMap[y][x]);
	    }

	    // Pad with spaces.
	    for (; x < myW; x++)
	    {
		if (x + myX < 0 || x + myX >= SCR_WIDTH)
		    continue;
		gfx_printchar(x + myX, y + myY, ' ', myAttrMap[y][x]);
	    }
	}
    }
}

void
PANEL::setIndent(int indent)
{
    myIndent = indent;
}

void
PANEL::setRigthMargin(int margin)
{
    myRightMargin = margin;
}

void
PANEL::scrollUp()
{
    int		y;

    if (myRecordHistory)
    {
	char		*line;

	// It is a sign of an incompetent programmer to have an oversized
	// overflow like this.  Either you know the behaviour, so can
	// have an exact bonus, or you don't, so can't really give any
	// bonus.
	line = new char [myW+10];
	memcpy(line, myLines[0], myW+1);
	myHistory.append(line);
    }

    for (y = 1; y < myH; y++)
    {
	memcpy(myLines[y-1], myLines[y], myW+1);
    }
    myLines[myH-1][0] = '\0';

    myCurLine--;
    if (myCurLine < 0)
	myCurLine = 0;
}

void
PANEL::scrollDown()
{
    int		y;

    for (y = myH-1; y >= 1; y--)
    {
	memcpy(myLines[y], myLines[y-1], myW+1);
    }

    // Pull out the history...
    if (myHistory.entries())
    {
	memcpy(myLines[0], myHistory.top(), myW+1);
	delete [] myHistory.pop();
    }
    else
	myLines[y][0] = '\0';

    myCurLine++;
    if (myCurLine > myH-1)
	myCurLine = myH-1;
}
