/*
 * Licensed under BSD license.  See LICENCE.TXT  
 *
 * Produced by:	Jeff Lait
 *
 *      	7DRL Development
 *
 * NAME:        mob.h ( Live Once Library, C++ )
 *
 * COMMENTS:
 */

#ifndef __mob__
#define __mob__

#include "glbdef.h"

#include "grammar.h"
#include "dpdf.h"
#include "ptrlist.h"

#include <iostream>
using namespace std;

class MAP;
class ITEM;

enum AI_STATE
{
    AI_STATE_EXPLORE,		// The default.
    AI_STATE_NEXTLEVEL,
    AI_STATE_HUNT,
    AI_STATE_PACKRAT,
};

class MOB
{
public:
    static MOB		*create(MOB_NAMES def);

    static MOB		*createNPC(int depth);

    // Creates a copy.  Ideally we make this copy on write so
    // is cheap, but mobs probably move every frame anyways.
    MOB			*copy() const;
    
    ~MOB();

    int			 getX() const { return myX; }
    int			 getY() const { return myY; }

    int			 getHP() const { return myHP; }

    void		 gainHP(int hp);

    void		 setHome(int x, int y) { myHX = x; myHY = y; }
    int			 getHX() const { return myHX; }
    int			 getHY() const { return myHY; }

    bool		 isSwallowed() const { return myIsSwallowed; }
    void		 setSwallowed(bool isswallowed) { myIsSwallowed = isswallowed; }

    // Returns true if we die!
    bool		 applyDamage(MOB *src, int hp, const DPDF &dpdf);

    MOB_NAMES		 getDefinition() const { return myDefinition; }

    const MOB_DEF	&defn() const { return defn(getDefinition()); }
    static const MOB_DEF &defn(MOB_NAMES mob) { return glb_mobdefs[mob]; }

    bool		 isAvatar() const { return this == getAvatar(); }
    static MOB		*getAvatar() { return theAvatar; }
    static void		 setAvatar(MOB *avatar) { theAvatar = avatar; };

    static DPDF		&getAvatarHP_DPDF() { return theAvatarHP_DPDF; }

    VERB_PERSON		 getPerson() const;

    // Symbol and attributes for rendering.
    void		 getLook(u8 &symbol, ATTR_NAMES &attr) const;

    // Retrieve the raw name.
    const char		*getName() const;
    const char		*getRawName() const;

    MAP			*getMap() const { return myMap; }
    static MAP		*getAvatarMap() { theAvatar ? theAvatar->getMap() : 0; }

    ITEM		*getInventory() const { return myInventory; }

    // Returns true if we have the item in question.
    bool		 hasItem(ITEM_NAMES itemname) const;
    ITEM 		*lookupItem(ITEM_NAMES itemname) const;
    ITEM		*getRandomItem() const;

    int			 getMeleeDamage() const;
    const char		*getMeleeVerb() const;
    DPDF		 getMeleeDPDF() const;
    int			 getRangedDamage() const;
    DPDF		 getRangedDPDF() const;
    int			 getRangedRange() const;

    void		 getRangedLook(u8 &symbol, ATTR_NAMES &attr) const;

    bool		 canMove(int x, int y, bool checkmob = true) const;

    void		 move(int x, int y);

    bool		 isFriends(MOB *other) const;
    void		 viewDescription() const;

    void		 addItem(ITEM *item);
    void		 removeItem(ITEM *item, bool quiet = false);

    void		 clearBoredom() { myBoredom = 0; }

    //
    // High level AI functions.
    //

    AI_NAMES		 getAI() const;

    // Determines if there are any mandatory actions to be done.
    // Return true if a forced action occured, in which case the
    // player gets no further input.
    bool		 aiForcedAction();

    // Runs the normal AI routines.  Called if aiForcedAction failed
    // and not the avatar.
    bool		 aiDoAI();

    // Runs the magical avatar AI that can do everything
    bool		 aiAvatar();

    // Searches for unexplored locations.
    bool		 aiAvatarExplore();

    // Runs to next level
    bool		 aiAvatarNextLevel();

    // Finds the tastiest enemy & attacks
    bool		 aiAvatarHunt();

    // Picks up item stored in HX/HY
    bool		 aiAvatarPackrat();

    // Updates our lock on the avatar - tracking if we know
    // where he is.  Position of last known location is in myTX.
    bool		 aiAcquireAvatar();
    bool		 aiAcquireTarget(MOB *foe);

    // Determines who we hate most.
    MOB			*aiFindEnemy() const;

    // Finds the closest item that is mapped.
    ITEM		*aiFindCloseMappedItem() const;

    // Returns the mob that we hate more, a or b
    MOB 		*aiHateMore(MOB *a, MOB *b) const;

    // Determine which item we like more.
    ITEM		*aiLikeMore(ITEM *a, ITEM *b) const;

    // Hugs the right hand walls.
    bool		 aiDoMouse();

    // Runs in straight lines until it hits something.
    bool		 aiStraightLine();

    // Charges the listed mob if in FOV
    bool		 aiCharge(MOB *foe, AI_NAMES aitype, bool orthoonly = false);

    // Can find mob anywhere on the mob - charges and kills.
    bool		 aiKillPathTo(MOB *target);

    // Attempts a ranged attack against the avatar
    bool		 aiRangeAttack(MOB *target = 0);

    // Runs away from (x, y).  Return true if action taken.
    bool		 aiFleeFrom(int x, int y);
    bool		 aiFleeFromAvatar();

    // Runs straight towards (x, y).  Return true if action taken.
    bool		 aiMoveTo(int x, int y, bool orthoonly = false);

    // Does a path find to get to x/y.  Returns false if blocked
    // or already at x & y.
    bool		 aiPathFindTo(int x, int y);

    // Tries to go to (x, y) by flanking them.
    bool		 aiFlankTo(int x, int y);
    
    bool		 aiRandomWalk(bool orthoonly = false);

    // Action methods.  These are how the AI and user manipulates
    // mobs.
    // Return true if the action consumed a turn, else false.
    bool		 actionBump(int dx, int dy);
    bool		 actionChat(int dx, int dy);
    bool		 actionMelee(int dx, int dy);
    bool		 actionWalk(int dx, int dy);
    bool		 actionFire(int dx, int dy);
    bool		 actionPickup();

    bool		 actionClimb();

    // The following are used at the map level..
    MOB			*getNext() const { return myNext; }
    void		 setNext(MOB *next) { myNext = next; }
    void		 setMap(MAP *map) { myMap = map; }

    void		 save(ostream &os) const;
    static MOB		*load(istream &is);

    void		 getVisibleEnemies(PTRLIST<MOB *> &list) const;
    bool		 hasVisibleEnemies() const;
    
protected:
    MOB();

    MOB_NAMES		 myDefinition;
    static MOB		*theAvatar;
    static DPDF		 theAvatarHP_DPDF;

    int			 myX, myY;
    MOB			*myNext;

    ITEM		*myInventory;

    // Current target
    int			 myTX, myTY;
    int			 myFleeCount;
    int			 myBoredom;

    bool		 myIsSwallowed;

    // State machine
    int			 myAIState;

    // My home spot.
    int			 myHX, myHY;

    // Hitpoints
    int			 myHP;

    MAP			*myMap;
};

#endif

