/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "profile.h"
#include "profile-private.h"
#include <errno.h>
#include <glib/gstdio.h>
#include <libprofile.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "debug.h"
#include "utils.h"


/* This file is a very hackish way to try to interoperate with the
 * default ringtone stuff.
 *
 * When a ringtone FOO.BAR is set, an uncompressed version of it
 * is created in ~/.local/share/sounds/FOO.BAR.wav.
 * What ringtoned does is to move away that file and replace it with
 * an invalid empty one, so the default ringtone is not played.
 */

#define RINGING_TONE "ringing.alert.tone"
#define RINGING_VOLUME "ringing.alert.volume"
 
#define NOKIA_TUNE_PATH "/usr/share/sounds/NokiaTune.aac"
#define RINGTONED_CACHED_NOKIA_TUNE "NokiaTune.aac.wav.ringtoned"

/* Was an initialization function called? */
static gboolean initialized = FALSE;
/* Only the ringtoned process moves files around */
static gboolean is_daemon = FALSE;
/* The ringtone set in the UI */
static gchar *ringtone_path = NULL;
/* The cached uncompressed ringtone file (that we change to be invalid) */
static gchar *cached_path = NULL;
/* The renamed uncompressed ringtone file (that we use when we want to play
 * the default ringtone) */
static gchar *ringtoned_cached_path = NULL;
/* The volume of the ringtone [0; 100] */
static gint ringtoned_volume = -1;

#define return_val_if_not_initialized(v) g_return_val_if_fail (initialized, (v))

gint
ringtoned_profile_get_current_volume (void)
{
    if (ringtoned_volume < 0)
    {
        g_critical ("Volume has not been initialized, "
                "falling back to full volume");
        return 100;
    }
    else
        return ringtoned_volume;
}

const gchar *
ringtoned_profile_get_default_ringtone (void)
{
    return_val_if_not_initialized ("");
    /* This should not happen, but just in case be nice and don't
     * return NULL */
    g_return_val_if_fail (ringtone_path != NULL, "");

    return ringtone_path;
}

const gchar *
ringtoned_profile_get_cached_default_ringtone (void)
{
    return_val_if_not_initialized ("");
    /* This should not happen, but just in case be nice and don't
     * return NULL */
    g_return_val_if_fail (ringtoned_cached_path != NULL, "");

    return ringtoned_cached_path;
}

gboolean
ringtoned_profile_is_default_ringtone (const gchar *path)
{
    return_val_if_not_initialized (FALSE);
    g_return_val_if_fail (path, FALSE);

    return strcmp (path, ringtone_path) == 0 ||
           strcmp (path, cached_path) == 0 ||
           strcmp (path, ringtoned_cached_path) == 0;
}

static gchar *
get_chached_file (const gchar *original)
{
    gchar *original_basename;
    gchar *cached_basename;
    gchar *cached;

    original_basename = g_path_get_basename (original);
    cached_basename = g_strconcat (original_basename, ".wav", NULL);
    cached = g_build_filename (g_get_user_data_dir (), "sounds",
            cached_basename, NULL);

    g_free (cached_basename);
    g_free (original_basename);

    return cached;
}

/* If update is true then this function is called for the normal update
 * process.
 * If false, the function just has to manage a ringtone (i.e. the default
 * Nokia tune) without actually setting it as default. */
static void
move_ringtones (const gchar *new_ringtone_path,
                gboolean     update)
{
    gchar *new_cached_path = NULL;
    gchar *new_ringtoned_cached_path = NULL;
    struct stat stat_buf;
    GIOChannel *io_channel = NULL;

    if (update && ringtone_path &&
        strcmp (ringtone_path, new_ringtone_path) == 0)
    {
        /* Nothing changed, but we still verify that the ringtones are ok */
        update = TRUE;
    }

    new_cached_path = get_chached_file (new_ringtone_path);
    new_ringtoned_cached_path = g_strconcat (new_cached_path, ".ringtoned", NULL);

    if (!is_daemon)
        /* The renaming and moving of file is done only by the main ringtoned
         * process. Other uses of the library just get the notifications about
         * which ringtone is set */
        goto finally;

    if (g_stat (new_cached_path, &stat_buf) != 0)
    {
        /* The cached file is not there :-/
         * If it appears later we will end up playing both our custom ringtone
         * and the default one */
        g_warning ("Cannot find the cached file for ringtone %s",
                new_cached_path);
        goto finally;
    }

    if (stat_buf.st_size == 0)
        goto finally;

    if (ringtoned_cached_path && update &&
        !g_str_has_suffix (ringtoned_cached_path, RINGTONED_CACHED_NOKIA_TUNE))
    {
        g_unlink (ringtoned_cached_path);
    }

    if (g_rename (new_cached_path, new_ringtoned_cached_path))
    {
        g_critical ("Cannot move the cached ringtone from %s to %s. "
                "Error %d: %s", new_cached_path, new_ringtoned_cached_path,
                errno, g_strerror (errno));
        goto finally;
    }

    io_channel = g_io_channel_new_file (new_cached_path, "w", NULL);
    if (!io_channel)
    {
        g_critical ("Cannot open the fake cached ringtone %s",
                new_cached_path);
        goto finally;
    }

finally:
    if (update)
    {
        DEBUG ("Ringtone changed\n"
                "    New path: %s\n"
                "        (was: %s)\n"
                "    New fake cached path: %s\n"
                "        (was: %s)\n"
                "    New ringtoned cached path: %s\n"
                "        (was: %s)",
                new_ringtone_path, ringtone_path,
                new_cached_path, cached_path,
                new_ringtoned_cached_path, ringtoned_cached_path);

        g_free (ringtone_path);
        ringtone_path = g_strdup (new_ringtone_path);

        g_free (cached_path);
        cached_path = new_cached_path;

        g_free (ringtoned_cached_path);
        ringtoned_cached_path = new_ringtoned_cached_path;
    }
    else
    {
        g_free (new_cached_path);
        g_free (new_ringtoned_cached_path);
    }

    if (io_channel)
        g_io_channel_unref (io_channel);
}

static void
update_ringtone (const gchar *new_ringtone_path)
{
    /* First we take care about the actual ringtone */
    move_ringtones (new_ringtone_path, TRUE);

    /* And then about the default Nokia Tune. This is treated in a
     * different way by default ringtone programs; its cached version
     * is never deleted and, if missing, is recreated.
     * We always have to ensure it exists and it's non-playable or
     * the default ringtone program will fall back to use it */
    if (is_daemon)
        move_ringtones (NOKIA_TUNE_PATH, FALSE);
}

static gboolean
update_ringtone_async_cb (gpointer user_data)
{
    const gchar *new_ringtone_path = user_data;

    update_ringtone (new_ringtone_path);

    return FALSE;
}

static void
update_ringtone_async (const gchar *new_ringtone_path)
{
    g_timeout_add (1500, update_ringtone_async_cb,
            (gpointer) new_ringtone_path);
}

static void
update_volume (gint volume)
{
    DEBUG ("Set volume to %d", volume);
    if (volume < 0 || volume > 100)
    {
        g_critical ("Volume not in the expected range!");
        volume = CLAMP (volume, 0, 100);
    }

    ringtoned_volume = volume;
}

static void
update_volume_from_active_profile (void)
{
    update_volume (profile_get_value_as_int (NULL, RINGING_VOLUME));
}

static void
active_profile_changed_cb (const gchar *profile_name,
                           gpointer     user_data)
{
    gchar *value;

    DEBUG ("Active profile changed to %s", profile_name);

    value = profile_get_value (NULL, RINGING_TONE);
    update_ringtone (value);
    g_free (value);

    update_volume_from_active_profile ();
}

static void
active_profile_data_changed_cb (const gchar *profile_name,
                                const gchar *key,
                                const gchar *val,
                                const gchar *type,
                                gpointer     user_data)
{
    DEBUG ("New value for %s, %s:%s (type %s)",
            profile_name, key, val, type);

    if (strcmp (key, RINGING_TONE) == 0)
        update_ringtone_async (val);
    else if (strcmp (key, RINGING_VOLUME) == 0)
        update_volume (profile_parse_int (val));
}

static gboolean
verify_cached_ringtones_cb (gpointer user_data)
{
    DEBUG ("Verify ringtones after boot");

    move_ringtones (ringtone_path, FALSE);
    move_ringtones (NOKIA_TUNE_PATH, FALSE);

    return FALSE;
}

static void
ringtoned_profile_init_real (void)
{
    gchar *ringtone;

    initialized = TRUE;

    profile_track_add_profile_cb (active_profile_changed_cb, NULL, NULL);
    profile_track_add_active_cb (active_profile_data_changed_cb, NULL, NULL);

    profile_tracker_init ();

    ringtone = profile_get_value (NULL, RINGING_TONE);
    DEBUG ("Setting initial ringtone to %s", ringtone);
    update_ringtone (ringtone);
    g_free (ringtone);

    update_volume_from_active_profile ();

    /* It looks like the system detects that the ringtones are not valid
     * at boot time, so it regenerates them.
     * The *very* ugly workaround for this is to check again after a
     * timeout.
     * With the normal ringtones the system seems to take about half a
     * minute before the ringtones are regenerated, so we do a first
     * check after 1 minute. */
    g_timeout_add (1 * 60 * 1000, verify_cached_ringtones_cb, NULL);
    /* Just to be sure we try again in some minutes: regenerating the
     * ringtones could take more if the load at boot time is really heavy
     * or it the ringtones are really big */
    g_timeout_add (5 * 60 * 1000, verify_cached_ringtones_cb, NULL);
}

static void
initialization_critical (void)
{
    g_critical ("Trying to initialize again the profile watching but "
            "in a different mode");
}

void
ringtoned_profile_init (void)
{
    if (initialized)
    {
        /* Calling the init function multiple times is fine, but not if
         * trying to change mode */
        if (is_daemon)
            initialization_critical ();
        return;
    }

    ringtoned_profile_init_real ();
}

void
ringtoned_profile_init_daemon (void)
{
    if (initialized)
    {
        /* Calling the init function multiple times is fine, but not if
         * trying to change mode */
        if (!is_daemon)
            initialization_critical ();
        return;
    }

    is_daemon = TRUE;
    ringtoned_profile_init_real ();
}
