/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "profile.h"
#include "profile-private.h"
#include <errno.h>
#include <glib/gstdio.h>
#include <libprofile.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "debug.h"
#include "utils.h"


/* This file is a very hackish way to try to interoperate with the
 * default ringtone stuff.
 *
 * When a ringtone FOO.BAR is set, an uncompressed version of it
 * is created in ~/.local/share/sounds/FOO.BAR.wav.
 * What ringtoned does is to move away that file and replace it with
 * an invalid empty one, so the default ringtone is not played.
 */

#define PROFILE_GENERAL "general"
#define RINGING_TONE "ringing.alert.tone"
 
#define NOKIA_TUNE_PATH "/usr/share/sounds/NokiaTune.aac"
#define RINGTONED_CACHED_NOKIA_TUNE "NokiaTune.aac.wav.ringtoned"

/* Was an initialization function called? */
static gboolean initialized = FALSE;
/* Only the ringtoned process moves files around */
static gboolean is_daemon = FALSE;
/* The ringtone set in the UI */
static gchar *ringtone_path = NULL;
/* The cached uncompressed ringtone file (that we change to be invalid) */
static gchar *cached_path = NULL;
/* The renamed uncompressed ringtone file (that we use when we want to play
 * the default ringtone) */
static gchar *ringtoned_cached_path = NULL;

#define return_val_if_not_initialized(v) g_return_val_if_fail (initialized, (v))

const gchar *
ringtoned_profile_get_default_ringtone (void)
{
    return_val_if_not_initialized ("");
    /* This should not happen, but just in case be nice and don't
     * return NULL */
    g_return_val_if_fail (ringtone_path != NULL, "");

    return ringtone_path;
}

gboolean
ringtoned_profile_is_default_ringtone (const gchar *path)
{
    return_val_if_not_initialized (FALSE);
    g_return_val_if_fail (path, FALSE);

    return strcmp (path, ringtone_path) == 0 ||
           strcmp (path, cached_path) == 0 ||
           strcmp (path, ringtoned_cached_path) == 0;
}

const gchar *
ringtoned_profile_get_playable_path (const gchar *path)
{
    return_val_if_not_initialized (path);

    if (ringtoned_profile_is_default_ringtone (path))
        /* Let's try to play the uncompressed one if possible */
        path = ringtoned_cached_path;

    /* The point of this function is to just return something that
     * should be playable and avoid not playing any sound just because
     * the file was deleted or is in an unmounted MyDocs if the device
     * is in mass storage mode.
     * Of course this does't protect us from invalid audio files,
     * but that's not the point of this function */

#define RETURN_IF_EXISTS(p) \
        if (!IS_EMPTY(p) && g_file_test (p, G_FILE_TEST_EXISTS)) \
            return p;

    /* The file that the user wanted, great! */
    RETURN_IF_EXISTS (path);

    /* Fall back to the default ringtone */
    RETURN_IF_EXISTS (ringtoned_cached_path);
    RETURN_IF_EXISTS (ringtone_path);

    /* Fall back to the original default ringtone */
    RETURN_IF_EXISTS ("/usr/share/sounds/NokiaTune.aac");

#undef RETURN_IF_EXISTS

    /* Seriously, WTF? This should really not be reached, in any case
     * let's return something that is not NULL. */
    return "/dev/null";
}

static gchar *
get_chached_file (const gchar *original)
{
    gchar *original_basename;
    gchar *cached_basename;
    gchar *cached;

    original_basename = g_path_get_basename (original);
    cached_basename = g_strconcat (original_basename, ".wav", NULL);
    cached = g_build_filename (g_get_user_data_dir (), "sounds",
            cached_basename, NULL);

    g_free (cached_basename);
    g_free (original_basename);

    return cached;
}

/* If update is true then this function is called for the normal update
 * process.
 * If false, the function just has to manage a ringtone (i.e. the default
 * Nokia tune) without actually setting it as default. */
static void
move_ringtones (const gchar *new_ringtone_path,
                gboolean     update)
{
    gchar *new_cached_path = NULL;
    gchar *new_ringtoned_cached_path = NULL;
    struct stat stat_buf;
    GIOChannel *io_channel = NULL;

    if (update && ringtone_path &&
        strcmp (ringtone_path, new_ringtone_path) == 0)
    {
        /* Nothing changed, but we still verify that the ringtones are ok */
        update = TRUE;
    }

    new_cached_path = get_chached_file (new_ringtone_path);
    new_ringtoned_cached_path = g_strconcat (new_cached_path, ".ringtoned", NULL);

    if (!is_daemon)
        /* The renaming and moving of file is done only by the main ringtoned
         * process. Other uses of the library just get the notifications about
         * which ringtone is set */
        goto finally;

    if (g_stat (new_cached_path, &stat_buf) != 0)
    {
        /* The cached file is not there :-/
         * If it appears later we will end up playing both our custom ringtone
         * and the default one */
        g_warning ("Cannot find the cached file for ringtone %s",
                new_cached_path);
        goto finally;
    }

    if (stat_buf.st_size == 0)
        goto finally;

    if (ringtoned_cached_path && update &&
        !g_str_has_suffix (ringtoned_cached_path, RINGTONED_CACHED_NOKIA_TUNE))
    {
        g_unlink (ringtoned_cached_path);
    }

    if (g_rename (new_cached_path, new_ringtoned_cached_path))
    {
        g_critical ("Cannot move the cached ringtone from %s to %s. "
                "Error %d: %s", new_cached_path, new_ringtoned_cached_path,
                errno, g_strerror (errno));
        goto finally;
    }

    io_channel = g_io_channel_new_file (new_cached_path, "w", NULL);
    if (!io_channel)
    {
        g_critical ("Cannot open the fake cached ringtone %s",
                new_cached_path);
        goto finally;
    }

finally:
    if (update)
    {
        DEBUG ("Ringtone changed\n"
                "    New path: %s\n"
                "        (was: %s)\n"
                "    New fake cached path: %s\n"
                "        (was: %s)\n"
                "    New ringtoned cached path: %s\n"
                "        (was: %s)",
                new_ringtone_path, ringtone_path,
                new_cached_path, cached_path,
                new_ringtoned_cached_path, ringtoned_cached_path);

        g_free (ringtone_path);
        ringtone_path = g_strdup (new_ringtone_path);

        g_free (cached_path);
        cached_path = new_cached_path;

        g_free (ringtoned_cached_path);
        ringtoned_cached_path = new_ringtoned_cached_path;
    }
    else
    {
        g_free (new_cached_path);
        g_free (new_ringtoned_cached_path);
    }

    if (io_channel)
        g_io_channel_unref (io_channel);
}

static void
update_ringtone (const gchar *new_ringtone_path)
{
    /* First we take care about the actual ringtone */
    move_ringtones (new_ringtone_path, TRUE);

    /* And then about the default Nokia Tune. This is treated in a
     * different way by default ringtone programs; its cached version
     * is never deleted and, if missing, is recreated.
     * We always have to ensure it exists and it's non-playable or
     * the default ringtone program will fall back to use it */
    if (is_daemon)
        move_ringtones (NOKIA_TUNE_PATH, FALSE);
}

static gboolean
update_ringtone_async_cb (gpointer user_data)
{
    const gchar *new_ringtone_path = user_data;

    update_ringtone (new_ringtone_path);

    return FALSE;
}

static void
update_ringtone_async (const gchar *new_ringtone_path)
{
    g_timeout_add (1500, update_ringtone_async_cb,
            (gpointer) new_ringtone_path);
}

static void
active_profile_changed_cb (const gchar *profile_name,
                           gpointer     user_data)
{
    DEBUG ("Active profile changed to %s", profile_name);

    if (strcmp (profile_name, PROFILE_GENERAL))
    {
        gchar *value;

        value = profile_get_value (PROFILE_GENERAL, RINGING_TONE);
        update_ringtone (value);
        g_free (value);
    }
}

static void
active_profile_data_changed_cb (const gchar *profile_name,
                                const gchar *key,
                                const gchar *val,
                                const gchar *type,
                                gpointer     user_data)
{
    DEBUG ("New value for %s, %s:%s (type %s)",
            profile_name, key, val, type);

    if (strcmp (profile_name, PROFILE_GENERAL) == 0 &&
        strcmp (key, RINGING_TONE) == 0)
    {
        update_ringtone_async (val);
    }
}

static gboolean
verify_cached_ringtones_cb (gpointer user_data)
{
    DEBUG ("Verify ringtones after boot");

    move_ringtones (ringtone_path, FALSE);
    move_ringtones (NOKIA_TUNE_PATH, FALSE);

    return FALSE;
}

static void
ringtoned_profile_init_real (void)
{
    gchar *value;

    initialized = TRUE;

    profile_track_add_profile_cb (active_profile_changed_cb, NULL, NULL);
    profile_track_add_active_cb (active_profile_data_changed_cb, NULL, NULL);

    profile_tracker_init ();

    value = profile_get_value (NULL, RINGING_TONE);
    DEBUG ("Setting initial value to %s", value);
    update_ringtone (value);
    g_free (value);

    /* It looks like the system detects that the ringtones are not valid
     * at boot time, so it regenerates them.
     * The *very* ugly workaround for this is to check again after a
     * timeout.
     * With the normal ringtones the system seems to take about half a
     * minute before the ringtones are regenerated, so we do a first
     * check after 1 minute. */
    g_timeout_add (1 * 60 * 1000, verify_cached_ringtones_cb, NULL);
    /* Just to be sure we try again in some minutes: regenerating the
     * ringtones could take more if the load at boot time is really heavy
     * or it the ringtones are really big */
    g_timeout_add (5 * 60 * 1000, verify_cached_ringtones_cb, NULL);
}

static void
initialization_critical (void)
{
    g_critical ("Trying to initialize again the profile watching but "
            "in a different mode");
}

void
ringtoned_profile_init (void)
{
    if (initialized)
    {
        /* Calling the init function multiple times is fine, but not if
         * trying to change mode */
        if (is_daemon)
            initialization_critical ();
        return;
    }

    ringtoned_profile_init_real ();
}

void
ringtoned_profile_init_daemon (void)
{
    if (initialized)
    {
        /* Calling the init function multiple times is fine, but not if
         * trying to change mode */
        if (!is_daemon)
            initialization_critical ();
        return;
    }

    is_daemon = TRUE;
    ringtoned_profile_init_real ();
}
