/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "ringtone-chooser.h"
#include <ctype.h>
#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>
#include <libringtoned/ringtoned.h>
#include <string.h>
#include "contact.h"


enum
{
    PROP_0,
    PROP_RINGTONE
};

enum
{
    COL_TEXT,
    COL_PATH,
};

#define RESPONSE_MORE 1

struct _PCRRingtoneChooserPrivate
{
    GtkListStore *store;
    GtkWidget *selector;

    gchar *ringtone;
    RingtonedPlayer *player;
};

#define PCR_RINGTONE_CHOOSER_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), \
                                  PCR_TYPE_RINGTONE_CHOOSER, \
                                  PCRRingtoneChooserPrivate))

G_DEFINE_TYPE (PCRRingtoneChooser, pcr_ringtone_chooser, GTK_TYPE_DIALOG)

GtkWidget *
pcr_ringtone_chooser_new (GtkWindow   *parent,
                          const gchar *ringtone)
{
    GtkWidget *dialog;

    dialog = g_object_new (PCR_TYPE_RINGTONE_CHOOSER,
            "ringtone", ringtone,
            NULL);

    if (parent)
        gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);

    return dialog;
}

static gboolean
get_selected_iter (PCRRingtoneChooser *self,
                   GtkTreeIter        *iter)
{
    GList *rows;
    gboolean success = FALSE;

    rows = hildon_touch_selector_get_selected_rows (
            HILDON_TOUCH_SELECTOR (self->priv->selector), 0);
    if (!rows || rows->next)
    {
        g_critical ("%d ringtones selected, while 1 was expected",
                g_list_length (rows));
        goto finally;
    }

    if (gtk_tree_model_get_iter (GTK_TREE_MODEL (self->priv->store),
                iter, rows->data))
    {
        success = TRUE;
    }

finally:
    g_list_foreach (rows, (GFunc) gtk_tree_path_free, NULL);
    g_list_free (rows);

    return success;
}

static void
stop_preview (PCRRingtoneChooser *self)
{
    if (self->priv->player)
    {
        DEBUG ("Stop current preview");

        ringtoned_player_stop (self->priv->player);
        g_object_unref (self->priv->player);
        self->priv->player = NULL;
    }
}

static void
selector_changed_cb (HildonTouchSelector *widget,
                     gint                 column,
                     PCRRingtoneChooser  *self)
{
    GtkTreeIter iter;
    gchar *preview_path;

    if (!get_selected_iter (self, &iter))
        return;

    stop_preview (self);

    gtk_tree_model_get (GTK_TREE_MODEL (self->priv->store), &iter,
            COL_PATH, &preview_path, -1);

    if (g_strcmp0 (preview_path, PCR_RINGTONE_DEFAULT) == 0)
        DEBUG ("Not playing preview for default ringtone");
    else if (g_strcmp0 (preview_path, PCR_RINGTONE_SILENT) == 0)
        DEBUG ("Not playing preview for silent");
    else
    {
        DEBUG ("Starting preview of %s", preview_path);
        self->priv->player = ringtoned_player_new (preview_path, TRUE);
        ringtoned_player_start (self->priv->player);
    }

    g_free (preview_path);
}

static void
pcr_ringtone_chooser_size_request (GtkWidget      *widget,
                                   GtkRequisition *requisition)
{
    GTK_WIDGET_CLASS (pcr_ringtone_chooser_parent_class)->size_request (
            widget, requisition);
    /* Pretend this didn't happen */
    requisition->height = 294;
}

typedef struct
{
    PCRRingtoneChooser *self;
    GtkTreeIter iter;
    gboolean found;
} FindRingtoneData;

static gboolean
find_ringtone_cb (GtkTreeModel *model,
                  GtkTreePath  *path,
                  GtkTreeIter  *iter,
                  gpointer      user_data)
{
    FindRingtoneData *data = user_data;
    gchar *ringtone_path;

    gtk_tree_model_get (model, iter, COL_PATH, &ringtone_path, -1);
    if (g_strcmp0 (ringtone_path, data->self->priv->ringtone) == 0)
    {
        data->iter = *iter;
        data->found = TRUE;
    }

    g_free (ringtone_path);

    return data->found;
}

static void
ensure_current_ringtone (PCRRingtoneChooser *self)
{
    FindRingtoneData *find_ringtone_data;
    gchar *display;
    gchar *extension;

    find_ringtone_data = g_new0 (FindRingtoneData, 1);
    find_ringtone_data->self = self;

    gtk_tree_model_foreach (GTK_TREE_MODEL (self->priv->store),
            find_ringtone_cb, find_ringtone_data);

    if (!find_ringtone_data->found)
    {

        DEBUG ("Adding current ringtone (%s)", self->priv->ringtone);

        display = g_path_get_basename (self->priv->ringtone);
        extension = strrchr (display, '.');
        if (extension)
            *extension = '\0';

        gtk_list_store_insert_with_values (self->priv->store,
                &find_ringtone_data->iter,
                2, /* After the default one and silent */
                COL_TEXT, display,
                COL_PATH, self->priv->ringtone,
                -1);

        g_free (display);
    }

    hildon_touch_selector_select_iter (
            HILDON_TOUCH_SELECTOR (self->priv->selector), 0,
            &find_ringtone_data->iter, TRUE);

    g_free (find_ringtone_data);
}

static void
file_chooser_response_cb (GtkWidget          *chooser,
                          gint                response_id,
                          PCRRingtoneChooser *self)
{
    if (GTK_RESPONSE_OK == response_id)
    {
        g_free (self->priv->ringtone);
        self->priv->ringtone = gtk_file_chooser_get_filename (
                GTK_FILE_CHOOSER (chooser));
        ensure_current_ringtone (self);
        g_object_notify (G_OBJECT (self), "ringtone");
    }

    gtk_widget_destroy (GTK_WIDGET (chooser));
}

static void
show_file_chooser (PCRRingtoneChooser *self)
{
    GtkWidget *chooser;
    GtkFileFilter *filter;
    gchar *sounds;

    chooser = hildon_file_chooser_dialog_new_with_properties (
            GTK_WINDOW (self),
            "title", "Add tone",
            "action", GTK_FILE_CHOOSER_ACTION_OPEN,
            "local-only", TRUE,
            NULL);

    filter = gtk_file_filter_new ();
    gtk_file_filter_add_mime_type (filter, "audio/*");
    gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (chooser), filter);

    sounds = g_build_filename (g_get_home_dir (), "MyDocs", ".sounds", NULL);
    if (g_file_test (sounds, G_FILE_TEST_IS_DIR))
        gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), sounds);

    g_signal_connect (chooser, "response",
            G_CALLBACK (file_chooser_response_cb), self);

    gtk_widget_show (chooser);

    g_free (sounds);
}

static void
set_ringtone_from_selection (PCRRingtoneChooser *self)
{
    GtkTreeIter iter;

    if (!get_selected_iter (self, &iter))
        return;

    g_free (self->priv->ringtone);
    gtk_tree_model_get (GTK_TREE_MODEL (self->priv->store), &iter,
            COL_PATH, &self->priv->ringtone, -1);
}

static void
response_cb (GtkDialog *dialog,
             gint       response_id,
             gpointer   data)
{
    PCRRingtoneChooser *self = PCR_RINGTONE_CHOOSER (dialog);

    stop_preview (self);

    switch (response_id)
    {
        case GTK_RESPONSE_OK:
            set_ringtone_from_selection (self);
            break;
        case RESPONSE_MORE:
            g_signal_stop_emission_by_name (dialog, "response");
            show_file_chooser (self);
            break;
    }
}

static void
add_ringtone (PCRRingtoneChooser *self,
              const gchar        *text,
              const gchar        *path)
{
    GtkTreeIter tree_iter;

    gtk_list_store_append (self->priv->store, &tree_iter);
    gtk_list_store_set (self->priv->store, &tree_iter,
            COL_TEXT, text,
            COL_PATH, path,
            -1);
}

typedef struct
{
    gchar *text;
    gchar *path;
} SortableRingtone;

static gint
sortable_ringtone_cmp (gconstpointer a,
                       gconstpointer b)
{
    const SortableRingtone *sr1 = a;
    const SortableRingtone *sr2 = b;

    g_assert (sr1);
    g_assert (sr2);

    return strcmp (sr1->text, sr2->text);
}

static void
load_default_ringtones (PCRRingtoneChooser *self)
{
    static const gchar ringtones_file[] = "/etc/ringtones";
    static const gchar ringtone_section[] = "[RINGTONE]\n";

    gchar *buff;
    gchar *start;
    gchar *end;
    gchar *tmp;
    gboolean done;
    gchar *display;
    gchar *filename;
    GList *ringtones = NULL;
    SortableRingtone *sr;

    /* The list of default ringtones is stored in /etc/ringtones in a
     * "I cannot believe it's not an ini file" file.
     * Sections indicate for what the following ringtones are used
     * (emails, IM, etc.), keys (surrounded by quotes) are the display
     * name, and values are the paths to the ringtones. Keys and values
     * are separated by spaces or tabs.
     * The entries in the file are sorted at the moment, but the defualt
     * UI ensures they are sorted, so we have to do the same.
     * We just care about the "[RINGTONE]" section, used for calls. */

    DEBUG ("Reading ringtones configuration file %s", ringtones_file);

    if (!g_file_get_contents (ringtones_file, &buff, NULL, NULL))
    {
        DEBUG ("    No ringtones configuration file found");
        return;
    }

    start = buff;

    while (start)
    {
        start = strstr (start, ringtone_section);
        if (!start)
            break;
        start += strlen (ringtone_section);

        while (start)
        {
            done = FALSE;

            switch (start[0])
            {
                case '\n':
                    /* Skip empty line */
                    start++;
                    continue;
                case '[':
                    /* New section */
                    done = TRUE;
                    break;
                case '\0':
                    /* End of file */
                    done = TRUE;
                    break;
                case '"':
                    /* Ok */;
                    break;
                default:
                    /* Unknown line, let's skip it */
                    DEBUG ("    Found an unknown line starting with '%c' "
                            "at offset %d", start[0], start - buff);
                    start = strchr (start, '\n');
                    continue;
            }

            if (done)
                break;

            start++; /* Skip the " */
            end = strchr (start, '"');
            tmp = strchr (start, '\n');
            if (!end)
            {
                DEBUG ("    Unterminated quote at the end of file");
                start = NULL;
                break;
            }
            else if (tmp && tmp < end)
            {
                DEBUG ("    Unterminated quote at offset %d", start - buff);
                start = tmp;
                continue;
            }

            display = g_strndup (start, end - start);

            start = end + 1;
            while (*start != '\n' && isspace (*start))
                start++;

            end = strchr (start, '\n');
            if (end)
            {
                filename = g_strndup (start, end - start);
                start = end + 1;
            }
            else
            {
                filename = g_strdup (start);
                start = NULL;
            }

            if (*filename && *display)
            {
                DEBUG ("    Add %s (%s)", display, filename);

                sr = g_new0 (SortableRingtone, 1);
                sr->text = display;
                sr->path = filename;

                ringtones = g_list_prepend (ringtones, sr);
            }
            else
            {
                if (!*filename && !*display)
                    DEBUG ("    Skipping entry with no data");
                else if (!*filename)
                    DEBUG ("    Skipping entry '%s' with no file name",
                            display);
                else
                    DEBUG ("    Skipping entry '%s' with no display name",
                            filename);

                g_free (display);
                g_free (filename);
            }
        }
    }

    ringtones = g_list_sort (ringtones, sortable_ringtone_cmp);

    while (ringtones)
    {
        sr = ringtones->data;

        add_ringtone (self, sr->text, sr->path);

        g_free (sr->text);
        g_free (sr->path);
        g_free (sr);

        ringtones = g_list_delete_link (ringtones, ringtones);
    }

    g_free (buff);
}

static void
fill_store (PCRRingtoneChooser *self)
{
    add_ringtone (self, "Default ringtone", PCR_RINGTONE_DEFAULT);
    add_ringtone (self, "Silent", PCR_RINGTONE_SILENT);

    load_default_ringtones (self);

    /* Select the currently set ringtone, or add it to the list if it's
     * not in the default locations */
    ensure_current_ringtone (self);
}

static void
pcr_ringtone_chooser_init (PCRRingtoneChooser *self)
{
    self->priv = PCR_RINGTONE_CHOOSER_GET_PRIVATE (self);

    gtk_window_set_title (GTK_WINDOW (self), "Ringing tone");

    gtk_dialog_add_buttons (GTK_DIALOG (self),
            "More", RESPONSE_MORE,
            "Done", GTK_RESPONSE_OK,
            NULL);

    g_signal_connect (self, "response", G_CALLBACK (response_cb), NULL);

    self->priv->store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
    self->priv->selector = hildon_touch_selector_new ();
    hildon_touch_selector_append_text_column (
            HILDON_TOUCH_SELECTOR (self->priv->selector),
            GTK_TREE_MODEL (self->priv->store), TRUE);
    gtk_widget_show (self->priv->selector);

    gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area (GTK_DIALOG (self))),
            self->priv->selector, TRUE, TRUE, 0);
} 

static void
pcr_ringtone_chooser_constructed (GObject *object)
{
    PCRRingtoneChooser *self = PCR_RINGTONE_CHOOSER (object);

    fill_store (self);

    /* We do this after the current ringtone was selected to avoid playing
     * anything as soon as we create the dialog */
    g_signal_connect (self->priv->selector, "changed",
            G_CALLBACK (selector_changed_cb), self);

    G_OBJECT_CLASS (pcr_ringtone_chooser_parent_class)->constructed (object);
}

static void
pcr_ringtone_chooser_finalize (GObject *object)
{
    PCRRingtoneChooser *self = PCR_RINGTONE_CHOOSER (object);

    stop_preview (self);

    g_free (self->priv->ringtone);

    G_OBJECT_CLASS (pcr_ringtone_chooser_parent_class)->finalize (object);
}

static void
pcr_ringtone_chooser_get_property (GObject    *object,
                                   guint       param_id,
                                   GValue     *value,
                                   GParamSpec *pspec)
{
    PCRRingtoneChooser *self = PCR_RINGTONE_CHOOSER (object);

    switch (param_id)
    {
        case PROP_RINGTONE:
            g_value_set_string (value, self->priv->ringtone);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
pcr_ringtone_chooser_set_property (GObject      *object,
                                   guint         param_id,
                                   const GValue *value,
                                   GParamSpec   *pspec)
{
    PCRRingtoneChooser *self = PCR_RINGTONE_CHOOSER (object);

    switch (param_id)
    {
        case PROP_RINGTONE:
            self->priv->ringtone = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
pcr_ringtone_chooser_class_init (PCRRingtoneChooserClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

    object_class->constructed = pcr_ringtone_chooser_constructed;
    object_class->finalize = pcr_ringtone_chooser_finalize;
    object_class->get_property = pcr_ringtone_chooser_get_property;
    object_class->set_property = pcr_ringtone_chooser_set_property;

    widget_class->size_request = pcr_ringtone_chooser_size_request;

    g_object_class_install_property (
            object_class,
            PROP_RINGTONE,
            g_param_spec_string (
                "ringtone",
                "Ringtone",
                "The path of the selected ringtone",
                "",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_type_class_add_private (klass, sizeof (PCRRingtoneChooserPrivate));
}
