/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "player.h"
#include <gst/gst.h>
#include <pulse/pulseaudio.h>
#include <string.h>
#include "debug.h"
#include "profile.h"


#define RINGTONED_PLAYER_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RINGTONED_TYPE_PLAYER, \
                                  RingtonedPlayerPrivate))

enum
{
    PROP_0,
    PROP_FILE_PATH,
    PROP_PREVIEW,
};

struct _RingtonedPlayerPrivate
{
    gchar *file_path;
    gboolean preview;
    GstElement *player;
};

G_DEFINE_TYPE (RingtonedPlayer, ringtoned_player, G_TYPE_OBJECT)

RingtonedPlayer *
ringtoned_player_new (const gchar *file_path,
                      gboolean     preview)
{
    return g_object_new (RINGTONED_TYPE_PLAYER,
            "file-path", file_path,
            "preview", preview,
            NULL);
}

static gboolean
bus_watch_cb (GstBus     *bus,
              GstMessage *msg,
              gpointer    user_data)
{
    RingtonedPlayer *self = user_data;

    switch (GST_MESSAGE_TYPE (msg))
    {
        case GST_MESSAGE_EOS:
            if (self->priv->preview)
            {
                DEBUG ("Finished playing %s", self->priv->file_path);
            }
            else
            {
                DEBUG ("Repeating %s", self->priv->file_path);
                gst_element_seek_simple (self->priv->player, GST_FORMAT_TIME,
                        GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, 0);
            }
            break;

        case GST_MESSAGE_ERROR:
        {
            GError *err = NULL;
            gchar *dbg_info = NULL;
            
            gst_message_parse_error (msg, &err, &dbg_info);

            g_printerr ("Error from element %s: %s\n",
                    GST_OBJECT_NAME (msg->src), err->message);
            g_printerr ("Debugging info:\n%s\n",
                    dbg_info ? dbg_info : "none");

            ringtoned_player_stop (self);

            g_error_free (err);
            g_free (dbg_info);

            break;
        }

        default:
            break;
    }
 
    return TRUE;
}
 
void
ringtoned_player_start (RingtonedPlayer *self)
{
    const gchar *event_id;
    const gchar *restore_id;
    pa_proplist *proplist;
    GstElement *sink;
    GstBus *bus;
    gchar *escaped;
    const gchar *playable;
    gchar *uri;
 
    if (self->priv->player)
        /* Already playing */
        return;

    DEBUG ("Starting %s", self->priv->file_path);

    /* Magical bits to make the ringtones ignore the system volume */
    if (self->priv->preview)
    {
        event_id = "ringtone-preview";
        restore_id = "x-maemo-applet-profiles";
    }
    else
    {
        event_id = "phone-incoming-call";
        restore_id = "x-maemo-hildon-notify";
    }

    proplist = pa_proplist_new();
    pa_proplist_sets (proplist, "event.id", event_id);
    pa_proplist_sets (proplist, "module-stream-restore.id", restore_id);

    sink = gst_element_factory_make ("pulsesink", "ringtone-sink");
    g_object_set (sink,
            "proplist", proplist,
            NULL);

    /* If it's a preview you want to play exactly the file that is selected,
     * while if it's a ringtone you want to play something in any case so
     * the user doesn't miss the call */
    if (self->priv->preview)
        playable = self->priv->file_path;
    else
        playable = ringtoned_profile_get_playable_path (self->priv->file_path);

    DEBUG ("Actually playing %s", playable);
    escaped = g_uri_escape_string (playable, "/", FALSE);
    uri = g_strconcat ("file://", escaped, NULL);

    self->priv->player = gst_element_factory_make ("playbin2", "ringtone-player");
    g_object_set (self->priv->player,
            "uri", uri,
            "audio-sink", sink,
            NULL);
 
    bus = gst_pipeline_get_bus (GST_PIPELINE (self->priv->player));
    gst_bus_add_watch (bus, bus_watch_cb, self);
 
    gst_element_set_state (self->priv->player, GST_STATE_PLAYING);

    gst_object_unref (bus);
    g_free (uri);
    g_free (escaped);
}
 
void
ringtoned_player_stop (RingtonedPlayer *self)
{
    if (!self->priv->player)
        /* Already stopped */
        return;

    DEBUG ("Stopping %s", self->priv->file_path);

    gst_element_set_state (self->priv->player, GST_STATE_NULL);
    gst_object_unref (self->priv->player);
    self->priv->player = NULL;
}

static void
ringtoned_player_get_property (GObject    *object,
                               guint       param_id,
                               GValue     *value,
                               GParamSpec *pspec)
{
    RingtonedPlayer *self = RINGTONED_PLAYER (object);

    switch (param_id)
    {
        case PROP_FILE_PATH:
            g_value_set_string (value, self->priv->file_path);
            break;
        case PROP_PREVIEW:
            g_value_set_boolean (value, self->priv->preview);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_player_set_property (GObject      *object,
                               guint         param_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
    RingtonedPlayer *self = RINGTONED_PLAYER (object);

    switch (param_id)
    {
        case PROP_FILE_PATH:
            self->priv->file_path = g_value_dup_string (value);
            break;
        case PROP_PREVIEW:
            self->priv->preview = g_value_get_boolean (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_player_init (RingtonedPlayer *self)
{
    self->priv = RINGTONED_PLAYER_GET_PRIVATE (self);
}

static void
ringtoned_player_finalize (GObject *object)
{
    RingtonedPlayer *self = RINGTONED_PLAYER (object);

    ringtoned_player_stop (self);
        
    g_free (self->priv->file_path);

    G_OBJECT_CLASS (ringtoned_player_parent_class)->finalize (object);
}

static void
ringtoned_player_class_init (RingtonedPlayerClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    object_class->finalize = ringtoned_player_finalize;
    object_class->get_property = ringtoned_player_get_property;
    object_class->set_property = ringtoned_player_set_property;

    g_object_class_install_property (
            object_class,
            PROP_FILE_PATH,
            g_param_spec_string (
                "file-path",
                "File path",
                "The path to the file to play",
                "",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_PREVIEW,
            g_param_spec_boolean (
                "preview",
                "Preview",
                "Whether the ringtone is played for preview purposes",
                FALSE,
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_type_class_add_private (object_class, sizeof (RingtonedPlayerPrivate));
}
