/*************************************************************************}
{ phononplayer.h - Phonon-based audio player                              }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 29 May 2012                                              }
{*************************************************************************/

#ifndef PHONONPLAYER_H
#define PHONONPLAYER_H

#include "includes.h"
#include "equalizer.h"
#include "phononsound.h"

class PhononPlayer : public QObject
{
    Q_OBJECT
public:
    explicit PhononPlayer(QObject *parent = 0);
    ~PhononPlayer();
    inline const QStringList& getKnownExtensions() const {return knownExtensions;}
    inline const QStringList& getSupportedMime() const {return supportedMime;}
    inline const QStringList& getSupportedExtensions() const {return supportedExtensions;}
    inline const QString& getUnzipPath() const {return unzipPath;}
    inline const QString& getCurrentFilename() const {return sound->getCurrentFilename();}
    inline Equalizer* getEqualizer() const {return equalizer;}
    inline PhononSoundState getState() const {return sound->getState();}
    inline PhononSoundSourceType getSourceType() const {return sound->getSourceType();}
    inline const QString& getTitle() const {return sound->getTitle();}
    bool play(const QString& filename, bool doCloseCurrent = true);
    void cache(const QString& filename);
    void clearCache(const QString& filename);
    void clearCache();
    void pause();
    void unpause();
    void stop();
    bool isHeadsetConnected();
    inline bool isEqualizerEnabled() const {return equalizerEnabled;}
    bool setEqualizerEnabled(bool doEnable);
    inline bool isCalling() const {return calling;}
    bool unzip(const QString& filename, QString& tmpFilename);

public slots:
    void updateSupportedMime();

signals:
    void onPlayEnd();
    void onError();
    void onStateChanged();
    void onHeadsetStatusChange(bool isConnected);
    void onCallStart();
    void onCallEnd();
    void onTitleChange();

protected:
    void updateKnownMime();
    void findUnzip();
    void updateNameFilters();
    void killLater(PhononSound *soundObject, bool doClose = true);
    bool loadFromCache(const QString &filename);
    bool cacheExists(const QString &filename);

    PhononSound *sound, *cacheSound;
    Equalizer* equalizer;

    bool equalizerEnabled;
    qint64 lastPos;
    bool calling;

    QHash<QString, QStringList> mimeExtensions;
    QStringList knownExtensions;
    QStringList supportedMime;
    QStringList supportedExtensions;
    QString unzipPath;
    QString tmpFilenameBase;
    QObjectList obsoleteObjects;

protected slots:
    void onHeadsetCondition();
    void onCheckHeadset();
    void onCallStateChange(QString callState);
    void onKillObsoleteObject();
};

#endif // PHONONPLAYER_H
