/*************************************************************************}
{ optionitem.cpp - a graphical representation of a single option          }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 27 Nov 2012                                              }
{*************************************************************************/

#include "optionitem.h"

OptionItem::OptionItem(QWidget *parent) :
    KineticListItem(parent)
{
    type = oit_None;
    minValue = 0;
    maxValue = 100;
}

void OptionItem::setType(OptionItemType theType)
{
    type = theType;
    updateLabel();
    updateIcon();
}

void OptionItem::setLabelText(const QString &newText)
{
    labelText = newText;
    updateLabel();
}

void OptionItem::setString(const QString &newValue)
{
    int i;
    switch(type)
    {
        case oit_SelectString:
        case oit_SelectInt:
            i = stringValues.indexOf(newValue);
            if(i < 0)
                return;
            intValue = i;
            break;

        case oit_String:
            break;

        default:
            return;
    }
    KineticListItem::setString(newValue);
    updateLabel();
}

void OptionItem::setBool(bool newValue)
{
    KineticListItem::setBool(newValue);
    if(type == oit_Check)
        updateIcon();
}

void OptionItem::setInt(int newValue)
{
    switch(type)
    {
        case oit_Int:
            if((newValue < minValue) || (newValue > maxValue))
                return;
            break;

        case oit_SelectInt:
        case oit_SelectString:
            if((newValue < 0) || (newValue >= stringValues.size()))
                return;
            stringValue = stringValues[newValue];
            break;

        default:
            return;
    }

    KineticListItem::setInt(newValue);
    updateLabel();
}

void OptionItem::load()
{
    QVariant v = MAINWINDOW->getIniSettings()->value(fullIniKey, defaultValue);
    bool ok;


    switch(type)
    {
        case oit_Check:
            boolValue = v.toBool();
            break;

        case oit_SelectInt:
        case oit_SelectString:
            if(type == oit_SelectString)
                intValue = stringValues.indexOf(v.toString());
            else
                intValue = v.toInt();
            if((intValue < 0) || (intValue >= stringValues.size()))
            {

                intValue = defaultValue.toInt(&ok);
                if(!ok || (intValue < 0) || (intValue >= stringValues.size()))
                    intValue = 0;
            }
            stringValue = stringValues[intValue];
            break;

        case oit_Int:
            intValue = v.toInt(&ok);
            if(!ok)
            {
                intValue = defaultValue.toInt(&ok);
                if(!ok)
                    intValue = 0;
            }
            if(intValue < minValue)
            {
                intValue = minValue;
            }
            else
            {
                if(intValue > maxValue)
                    intValue = maxValue;
            }
            break;

        case oit_String:
            stringValue = v.toString();
            break;

        default:
            return;
    }

    updateIcon();
    updateLabel();
}

void OptionItem::save()
{
    QVariant v;

    switch(type)
    {
        case oit_Check:
            v = boolValue;
            break;

        case oit_SelectString:
            v = stringValues[intValue];
            break;

        case oit_SelectInt:
        case oit_Int:
            v = intValue;
            break;

        case oit_String:
            v = stringValue;
            break;

        default:
            return;
    }

    MAINWINDOW->getIniSettings()->setValue(fullIniKey, v);
}

void OptionItem::init(OptionItemType theType)
{
    // to not change the icon twice (in setType() and load())
    type = theType;
    load();
}

OptionItem *OptionItem::create(
        QWidget *parent,
        OptionItemType type,
        const QString &iniKey,
        const QVariant &defaultValue,
        const QString &labelText,
        const QStringList &stringValues,
        int minValue,
        int maxValue
)
{
    OptionItem* item = new OptionItem(parent);
    item->setObjectName(iniKey);
    item->setIniKey(iniKey);
    item->setDefaultValue(defaultValue);
    item->setLabelText(labelText);
    item->setStringValues(stringValues);
    item->setMinValue(minValue);
    item->setMaxValue(maxValue);
    item->init(type);
    return item;
}

void OptionItem::updateLabel()
{
    switch(type)
    {
        case oit_Check:
            setRawLabelText(labelText);
            break;

        case oit_SelectInt:
        case oit_SelectString:
            setRawLabelText(labelText+": "+stringValues[getInt()]);
            break;

        case oit_Int:
            setRawLabelText(labelText+": "+QString::number(getInt()));
            break;

        case oit_String:
            setRawLabelText(labelText+": "+getString());
            break;

        default:
            break;
    }
}

void OptionItem::updateIcon()
{
    switch(type)
    {
        case oit_Check:
            if(getBool())
                setIcon(MAINWINDOW->getSkin().icoCheck);
            else
                setIcon(MAINWINDOW->getSkin().icoUncheck);
            break;

        default:
            if(!icon().isNull())
                setIcon(MAINWINDOW->getSkin().icoNone);
    }
}

void OptionItem::click()
{
    switch(type)
    {
        case oit_Check:
            setBool(!getBool());
            break;

        case oit_SelectInt:
        case oit_SelectString:
            setInt(SelectWindow::showSelectDialog(this, stringValues, getInt(), labelText));
            break;

        default:
            break;
    }
}

void OptionItem::setRawLabelText(const QString &newText)
{
    QStringList lines = wordwrap(newText, width() - iconSize().width(), fontMetrics());
    QString line = lines.join("\n");
    setText(line);
}


