/*************************************************************************}
{ coreapp.cpp - QCoreApplication wrapper                                  }
{                                                                         }
{ (c) Alexey Parfenov, 2011                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This library is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This library is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 27 Nov 2012                                              }
{*************************************************************************/

#include "coreapp.h"

#ifndef Q_OS_WIN
void sigHandler(int signo)
{
    switch(signo)
    {
        case SIGINT:
        case SIGKILL:
            ((CoreApp*)qApp)->interrupt(signo);
            break;

        default:
            break;
    }
}
#else
BOOL CtrlHandler(DWORD fdwCtrlType)
{
    switch(fdwCtrlType)
    {
        case CTRL_C_EVENT:
            ((CoreApp*)qApp)->interrupt(fdwCtrlType);
            return true;

        default:
            return false;
    }
}
#endif

CoreApp::CoreApp(int& argc, char** argv):CoreAppParentClass(argc, argv), ErrorHandler()
{
#ifdef APP_NAME
    setApplicationName(APP_NAME);
#endif
#ifdef APP_VERSION
    setApplicationVersion(APP_VERSION);
#endif
#ifdef APP_ORGANIZATION
    setOrganizationName(APP_ORGANIZATION);
#endif
#ifdef APP_ORGANIZATION_DOMAIN
    setOrganizationDomain(APP_ORGANIZATION_DOMAIN);
#endif

    connect(this,
            SIGNAL(onError(const QObject*,const char*,int,int,const void*)),
            SLOT(onGetError(const QObject*,const char*,int,int,const void*)));
    QObject::connect(this, SIGNAL(aboutToQuit()), this, SLOT(_onQuit()));
    QTimer::singleShot(0, this, SLOT(_run()));
    exitAfterMain = true;
#ifdef QT_NETWORK_LIB
    singleInstance = false;
    localServer = NULL;
    localSock = NULL;
#endif
    quitting = false;

#ifndef Q_OS_WIN
    signal(SIGINT, sigHandler);
    signal(SIGKILL, sigHandler);
#else
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)CtrlHandler, true);
#endif
}

QString CoreApp::errorCodeToString(int errorCode)
{
    QString s = ErrorHandler::errorCodeToString(errorCode);
    if(!s.isEmpty())
        return s;
    switch(errorCode)
    {
#ifdef QT_NETWORK_LIB
        case app_err_CreateLocalServer: return "Cannot create a local server";
        case app_err_FreeLocalServer: return "Cannot free the local server";
        case app_err_ReadFromLocalSock: return "Error reading from local socket";
        case app_err_WriteToLocalSock: return "Error writing to local socket";
        case app_err_SendToLocalServer: return "Error sending commands to the local server";
        case app_err_LocalSocketDisconnect: return "Local socket have disconnected unexpentedly";
#endif
    }
    return "";
}

void CoreApp::interrupt(int sigNum)
{
    Q_UNUSED(sigNum);
    quit();
}

void CoreApp::_run()
{
#ifdef QT_NETWORK_LIB
    if(singleInstance)
    {
        updateLocalSockName();
        QLocalSocket s;
        s.connectToServer(localSockName, QIODevice::WriteOnly);
        if(s.waitForConnected())
        {
            if(passCommandsToLocalServer(s))
            {
                quit();
                return;
            }
        }
        if(!startLocalServer())
        {
            exit(-1);
            return;
        }
    }
#endif
    if(exitAfterMain)
    {
        _main();
    }
    else
    {
        int result = main();
        if(result || quitting)
            exit(result);
    }
}

void CoreApp::_main()
{
    try{
        exit(main());
    }catch(...){
        exit(-1);
    }
}

int CoreApp::main()
{
    return 0;
}

#ifdef QT_NETWORK_LIB
bool CoreApp::startLocalServer()
{
    if(!localServer)
    {
        localServer = new QLocalServer();
        connect(localServer, SIGNAL(newConnection()), SLOT(onNewLocalServerConnection()));
        if(!localServer->listen(localSockName))
        {
            CHECK(localServer->serverError() == QAbstractSocket::AddressInUseError, app_err_CreateLocalServer);
            CHECK(localServer->removeServer(localSockName), app_err_FreeLocalServer);
            CHECK(localServer->listen(localSockName), app_err_CreateLocalServer);
        }
    }
    return true;
}

bool CoreApp::passCommandsToLocalServer(QLocalSocket &sock)
{
    QStringList args = arguments();
    if(args.size() <= 1)
        return true;
    args.removeFirst();
    QByteArray data;
    foreach(QString arg, args)
    {
        data.append(arg.toUtf8());
        data.append((char)0);
    }
    data.append((char)0);
    if(sock.write(data) == -1)
    {
        SETERROR(app_err_SendToLocalServer);
        return true;
    }
    if(!sock.waitForBytesWritten())
    {
        SETERROR(app_err_SendToLocalServer);
        return true;
    }
    return true;
}

void CoreApp::closeLocalSocket()
{
    if(localSock)
    {
        localSock->disconnect(this);
        localSock->disconnectFromServer();
        localSockCommands.clear();
        localSockBuffer.clear();
        localSock->deleteLater();
        localSock = NULL;
    }
}

void CoreApp::updateLocalSockName()
{
    if(localSockName.isEmpty())
    {
        localSockName = applicationName();
        if(localSockName.isEmpty())
        {
            QFileInfo i(applicationFilePath());
            localSockName = i.completeBaseName();
        }
    }
}

void CoreApp::onNewLocalServerConnection()
{
    closeLocalSocket();
    localSock = localServer->nextPendingConnection();
    if(!localSock)
        return;
    connect(localSock, SIGNAL(readyRead()), SLOT(onLocalSocketClientData()));
}

void CoreApp::onLocalSocketClientData()
{
    if(!localSock)
        return;
    if(qobject_cast<QLocalSocket*>(sender()) != localSock)
        return;
    int n = localSock->bytesAvailable();
    if(!n)
        return;
    char x;
    for(int a=0; a<n; a++)
    {
        if(!localSock->getChar(&x))
        {
            closeLocalSocket();
            SETERROR(app_err_ReadFromLocalSock);
            return;
        }
        if(x == 0)
        {
            if(localSockBuffer.isEmpty())
            {
                parseLocalSockCommands();
                bool ok;
                if(localSock->bytesToWrite())
                    ok = localSock->waitForBytesWritten();
                else
                    ok = true;
                closeLocalSocket();
                CHECKV(ok, app_err_WriteToLocalSock);
                return;
            }
            localSockCommands.append(QString::fromUtf8(localSockBuffer.constData()));
            localSockBuffer.clear();
        }
        else
        {
            localSockBuffer.append(x);
        }
    }
}

void CoreApp::onLocalSockDisconnect()
{
    closeLocalSocket();
    SETERROR(app_err_LocalSocketDisconnect);
}
#endif

void CoreApp::_onQuit()
{
    quitting = true;
    onQuit();
}
