#!/usr/bin/env python
# -*- coding: utf-8 -*-

#########################################################################
#    Copyright (C) 2010 Sergio Villar Senin <svillar@igalia.com>
#
#    This file is part of ReSiStance
#
#    ReSiStance is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    ReSiStance is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with ReSiStance.  If not, see <http://www.gnu.org/licenses/>.
#########################################################################

import feedparser
import gobject
import constants
import cPickle
import urllib
import urlparse
import os
import base64
import gtk
from sgmllib import SGMLParser
from threading import Thread
from xml.dom.minidom import Document
from xml.dom import minidom
import xmlrpclib

# http://diveintomark.org/archives/2002/05/31/rss_autodiscovery_in_python
def getRSSLink(url):
    BUFFERSIZE = 1024

    try:
        usock = urllib.urlopen(url)
        parser = LinkParser()
        while 1:
            buffer = usock.read(BUFFERSIZE)
            parser.feed(buffer)
            if parser.nomoretags: break
            if len(buffer) < BUFFERSIZE: break
        usock.close()
        return urlparse.urljoin(url, parser.href)
    except IOError:
        print 'Could not establish a connection to ' + url
        return ''


class LinkParser(SGMLParser):
    def reset(self):
        SGMLParser.reset(self)
        self.href = ''

    def do_link(self, attrs):
        if not ('rel', 'alternate') in attrs:
            return
        if not (('type', 'application/rss+xml') or ('type', 'application/atom+xml')) in attrs:
            return
        hreflist = [e[1] for e in attrs if e[0]=='href']
        if hreflist:
            self.href = hreflist[0]
        self.setnomoretags()

    def end_head(self, attrs):
        self.setnomoretags()
    start_body = end_head

class ReSiStanceFeedDict(feedparser.FeedParserDict):

    def __init__(self, feed_data):
        super(ReSiStanceFeedDict, self).__init__()

        self.update(feed_data)

        for entry in self.entries:
            if not 'read' in entry:
                entry['read'] = False

class FeedManager(gobject.GObject):

    def __init__(self):
        super(FeedManager, self).__init__()

        self.feed_data_list = []

    def get_feed_list(self):
        return self.feed_data_list

    def add_feed(self, url, callback, data=None):

        # Create a new thread to update from network
        adding_thread = Thread(target=self._add_feed_in_thread,
                               args=(url, callback, data))
        adding_thread.start()

    def update_feed(self, feed, callback, data=None):

        # Create a new thread to update from network
        updating_thread = Thread(target=self._update_feeds_in_thread,
                                 args=([feed], callback, data))
        updating_thread.start()

    def update_all(self, callback, data):

        # Create a new thread to update from network
        updating_thread = Thread(target=self._update_feeds_in_thread,
                                 args=(self.feed_data_list, callback, data))
        updating_thread.start()

    def remove_feed(self, feed):
        self.feed_data_list.remove(feed)

        # Save to disk
        try:
            self.save(None)
        except IOError:
            pass

    def export_opml(self, file_path, callback, data=None):

        # Create a new thread to update from network
        exporting_thread = Thread(target=self._export_opml_in_thread,
                               args=(file_path, callback, data))
        try:
            exporting_thread.start()
        except IOError:
            callback(False)

    def _export_opml_in_thread(self, file_path, callback, user_data):
        # Create the minidom document
        opml_doc = Document()

        # Add tags and create the body of the document
        root = opml_doc.createElement('opml')
        root.setAttribute('version','1.0')
        root.appendChild(opml_doc.createElement('head'))
        body = opml_doc.createElement('body')
        for feed_data in self.feed_data_list:
            outline = opml_doc.createElement('outline')
            outline.setAttribute('title',feed_data.feed.title)
            outline.setAttribute('text',feed_data.feed.subtitle)
            outline.setAttribute('xmlUrl',feed_data['href'])
            body.appendChild(outline)

        root.appendChild(body)
        opml_doc.appendChild(root)

        try:
            retval = True
            file_opml = open(file_path,"w")
            opml_doc.writexml(file_opml, "    ", "", "\n", "UTF-8")
        except IOError :
            print 'Error exporting OPML, could not write to ' + file_path
            retval = False

        gtk.gdk.threads_enter()
        callback(retval)
        gtk.gdk.threads_leave()

    def import_opml(self, file_path, callback, data=None):

        # Create a new thread to update from network
        importing_thread = Thread(target=self._import_opml_in_thread,
                               args=(file_path, callback, data))
        try:
            importing_thread.start()
        except IOError:
            callback(None, None)

    def _import_opml_in_thread(self, file_path, callback, user_data):
        if (not (os.path.exists(file_path) and os.path.isfile(file_path)) or
            (os.path.splitext(file_path)[1] != '.opml')):
            gtk.gdk.threads_enter()
            callback(None,None)
            gtk.gdk.threads_leave()
            return

        try:
            doc=open(file_path,'r')
            opml_doc = minidom.parse(doc)
            opml_doc.getElementsByTagName('outline')
            for node in opml_doc.getElementsByTagName('outline'):
                if node.getAttribute('xmlUrl') != '':
                    url = node.attributes['xmlUrl'].value
                    # Add feed to manager
                    self._add_feed_in_thread(url, callback, user_data)
        except IOError :
            gtk.gdk.threads_enter()
            callback(None,None)
            gtk.gdk.threads_leave()
            return

        doc.close()

    def find_feed(self, key_words, dialog, callback):
        # Create a new thread to search
        find_thread = Thread(target=self._find_feed_in_thread,
                               args=(key_words, dialog, callback))

        return find_thread.start()

    def _find_feed_in_thread(self, key_words, dialog, callback):
        try:
            server = xmlrpclib.Server('http://www.syndic8.com/xmlrpc.php')
            feedids = server.syndic8.FindFeeds(key_words,'last_pubdate',25,0)
            infolist = server.syndic8.GetFeedInfo(feedids, ['imageurl','sitename','dataurl'])
        except:
            infolist = None
            print 'Error while accessing syndic8.com'

        if callback:
            gtk.gdk.threads_enter()
            callback(key_words, dialog, infolist)
            gtk.gdk.threads_leave()

    def save(self, callback, data=None):
        # TODO: migrate to "with" statement when available
        try:
            db_file = open(constants.RSS_DB_FILE, 'w')
        except IOError:
            print 'Cannot write to', constants.RSS_DB_FILE
            raise
        else:
            # Create a new thread to store in disk
            saving_thread = Thread(target=self._save_in_thread,
                                   args=(db_file, callback, data))
            saving_thread.start()

    def _save_in_thread(self, db_file, callback, data):
        cPickle.dump(self.feed_data_list, db_file)
        if callback:
            gtk.gdk.threads_enter()
            callback(data)
            gtk.gdk.threads_leave()

    def load(self, callback, data=None):
        # TODO: migrate to "with" statement when available
        try:
            db_file = open(constants.RSS_DB_FILE, 'r')
        except IOError:
            print 'Cannot open', constants.RSS_DB_FILE
            raise
        else:
            # Create a new thread to load from disk
            loading_thread = Thread(target=self._load_in_thread,
                                   args=(db_file, callback, data))
            loading_thread.start()

    def _load_in_thread(self, db_file, callback, data):
        self.feed_data_list = cPickle.load(db_file)
        if callback:
            gtk.gdk.threads_enter()
            callback(data)
            gtk.gdk.threads_leave()

    def get_favicon(self, url):
        # Check that user dir exists
        user_path = os.path.join (constants.RSS_CONF_FOLDER, 'icons')

        if os.path.exists(user_path) == False:
            os.makedirs(user_path, 0700)

        file_name = os.path.join (user_path, base64.b64encode(url) + '.favicon.ico')
        if os.path.exists(file_name) == False:
            parsed_url = urlparse.urlsplit(url)
            localfile, headers = urllib.urlretrieve(parsed_url.scheme + '://' +
                                                    parsed_url.netloc + '/favicon.ico',
                                                    file_name)

            # Try with a more general address. If we got a text/html then we most
            # likely requested an invalid address. It's better this than to check
            # for something like "image/" because I noticed that some servers return
            # icons with funny content types like text/plain. No comment
            if headers['Content-type'].startswith('text/html') == True:
                domains = parsed_url.netloc.rsplit('.',2)
                # Do not retry if domains == 2 because it will be
                # the same address we tried before
                if len(domains) > 2:
                    localfile, headers = urllib.urlretrieve(parsed_url.scheme + '://' +
                                                            domains[-2] + '.' + domains[-1] +
                                                            '/favicon.ico', file_name)

                if headers['Content-type'].startswith('image/') == False:
                    os.remove(localfile)
        try:
            pixbuf = gtk.gdk.pixbuf_new_from_file(file_name)
        except:
            return None

        # Scale pixbuf. TODO: do not use hard-coded values
        if (pixbuf.get_width() != 32):
            pixbuf = pixbuf.scale_simple(32,32,gtk.gdk.INTERP_BILINEAR)
            pixbuf.save(file_name, 'png')

        return pixbuf

    def _add_feed_in_thread(self, url, callback, user_data):

        parsed_url = urlparse.urlsplit(url)
        if parsed_url.scheme == '':
            url = 'http://' + url

        if not url.endswith('xml') and not url.endswith('opml'):
            url = getRSSLink(url)

        # Return if we cannot get the feed URL
        if url == '':
            gtk.gdk.threads_enter()
            callback(None, None, user_data)
            gtk.gdk.threads_leave()
            return

        new_feed_data = ReSiStanceFeedDict(feedparser.parse(url))
        self.feed_data_list.append(new_feed_data)

        if 'link' in new_feed_data.feed:
            pixbuf = self.get_favicon(new_feed_data.feed.link)
        else:
            pixbuf = self.get_favicon(new_feed_data.href)

        # Call user callback
        gtk.gdk.threads_enter()
        callback(pixbuf, new_feed_data, user_data)
        gtk.gdk.threads_leave()

        # Save to disk
        try:
            self.save(None)
        except IOError:
            pass

    def _update_feeds_in_thread(self, feed_data_list, callback, user_data):

        for feed_data in feed_data_list:
            updated_feed_data = ReSiStanceFeedDict(feedparser.parse(feed_data.href))

            # In case of network failure
            if updated_feed_data == None or updated_feed_data.entries == None or \
                    len(updated_feed_data.entries) == 0:
                continue

            updated_feed_date = updated_feed_data.feed.get('updated_parsed') or \
                updated_feed_data.entries[0].get('updated_parsed')
            feed_date = feed_data.feed.get('updated_parsed') or \
                feed_data.entries[0].get('updated_parsed')

            if updated_feed_date > feed_date:
                # Merge feed data
                old_feed_entries = feed_data.entries[:]
                for entry in updated_feed_data.entries:
                    if entry in old_feed_entries:
                        updated_feed_data.entries.remove(entry)
                feed_data.update(updated_feed_data)
                feed_data.entries += old_feed_entries

        # Call user callback. Call it just once, if callers want a different behaviour
        # they can always call update_feed() for each one
        gtk.gdk.threads_enter()
        callback(user_data)
        gtk.gdk.threads_leave()

        # Save to disk
        try:
            self.save(None)
        except IOError:
            pass
