#!/usr/bin/env python
# -*- coding: utf-8 -*-

#########################################################################
#    Copyright (C) 2010 Sergio Villar Senin <svillar@igalia.com>
#
#    This file is part of ReSiStance
#
#    ReSiStance is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    ReSiStance is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with ReSiStance.  If not, see <http://www.gnu.org/licenses/>.
#########################################################################

import pango
import feedparser
import locale
import gettext
import gobject
import hildon
import pygtk
pygtk.require('2.0')
import gtk
import webkit
import os
import urllib2
from portrait import FremantleRotation
import constants
from feedmanager import FeedManager
# For unescape()
import re, htmlentitydefs
from threading import Thread
import time
from settings import Settings

_ = gettext.gettext

# Pre-cache colors
color_style = gtk.rc_get_style_by_paths(gtk.settings_get_default() , 'GtkButton', 'osso-logical-colors', gtk.Button)
active_color = color_style.lookup_color(constants.ACTIVE_TEXT_COLOR)
default_color = color_style.lookup_color(constants.DEFAULT_TEXT_COLOR)
secondary_color = color_style.lookup_color(constants.SECONDARY_TEXT_COLOR)
del color_style

manager = FeedManager()
settings = Settings()

# From http://effbot.org/zone/re-sub.htm#unescape-html
def unescape(text):
    def fixup(m):
        text = m.group(0)
        if text[:2] == "&#":
            # character reference
            try:
                if text[:3] == "&#x":
                    return unichr(int(text[3:-1], 16))
                else:
                    return unichr(int(text[2:-1]))
            except ValueError:
                pass
        else:
            # named entity
            try:
                text = unichr(htmlentitydefs.name2codepoint[text[1:-1]])
            except KeyError:
                pass
        return text # leave as is
    return re.sub("&#?\w+;", fixup, text)

def get_author_from_item(item):
    words = item.title.rsplit(":")
    if len(words) > 1:
        author = words[0].lstrip()
    elif 'publisher_detail' in item:
        author = item.publisher_detail.name
    elif 'author' in item:
        author = item.author
    else:
        author = ''

    return author

class FeedsView(hildon.GtkTreeView):

    FEEDS_ICON_COLUMN = 0
    FEEDS_NAME_COLUMN = 1
    FEEDS_DATA_COLUMN = 2
    FEEDS_READ_COLUMN = 3

    DUMMY_FEED_STATUS = -1

    def __init__(self, ui_mode):
        super(FeedsView, self).__init__(ui_mode)

        text_renderer = gtk.CellRendererText()
        text_renderer.set_property('ellipsize', pango.ELLIPSIZE_END)

        pix_renderer = gtk.CellRendererPixbuf()

        # Add columns
        column = gtk.TreeViewColumn('Icon', pix_renderer, pixbuf = 0)
        self.append_column(column);

        column = gtk.TreeViewColumn('Name', text_renderer, markup = self.FEEDS_NAME_COLUMN)
        column.set_expand(True)
        self.append_column(column);

        date_renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Date', date_renderer)
        column.set_cell_data_func(date_renderer, self._unread_view_data_func)
        self.append_column(column);


    def _unread_view_data_func(self, column, renderer, model, feed_iter):
        feed_data, unread_count = model.get(feed_iter, self.FEEDS_DATA_COLUMN,
                                      self.FEEDS_READ_COLUMN)

        if feed_data.status == self.DUMMY_FEED_STATUS:
            renderer.set_property('text', '')
            return

        if unread_count:
            renderer.set_property('markup', '<span size="xx-small">%d '
                                  '%s\n%s</span>' % (unread_count, _('unread'), _('items')))
        else:
            renderer.set_property('markup', '')

    def add_dummy_feed(self, url):

        store = self.get_model()
        feed_iter = store.append()
        dummy_feed_data = feedparser.FeedParserDict()
        dummy_feed_data.status = self.DUMMY_FEED_STATUS
        dummy_feed_data.href = url
        dummy_feed_data.feed = feedparser.FeedParserDict()
        dummy_feed_data.feed.title = _('Retrieving info...')
        store.set(feed_iter, self.FEEDS_ICON_COLUMN, None,
                  self.FEEDS_NAME_COLUMN, self.get_feed_title_markup(dummy_feed_data),
                  self.FEEDS_DATA_COLUMN, dummy_feed_data,
                  self.FEEDS_READ_COLUMN, 0)

        return (dummy_feed_data,feed_iter)

    def get_feed_title_markup(self, feed_data):
        if 'link' in  feed_data.feed:
            url = feed_data.feed.link
        else:
            url = feed_data.href

        return '%s\n<span face="monospace" foreground="%s"' \
            'size="x-small">%s</span>' % \
            (feed_data.feed.title, secondary_color.to_string(), url)

    def sort(self, order):
        if (order == constants.ASCENDING_ORDER):
            self.get_model().set_sort_column_id(self.FEEDS_NAME_COLUMN,
                                                gtk.SORT_ASCENDING)
        else:
            self.get_model().set_sort_column_id(self.FEEDS_NAME_COLUMN,
                                                gtk.SORT_DESCENDING)

class SettingsDialog(gtk.Dialog):

    def __init__(self, parent):
        super(SettingsDialog, self).__init__(_('Settings'), parent,
                                                gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                                (gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))

        # Orientation picker
        self.selector_orientation = hildon.TouchSelector(text=True)
        for caption in FremantleRotation.MODE_CAPTIONS:
            self.selector_orientation.append_text(caption)
        self.picker_orientation = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                      hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.picker_orientation.set_selector(self.selector_orientation)
        self.picker_orientation.set_alignment(0, 0.5, 0, 0)
        self.picker_orientation.set_title(_("Screen orientation"))
        self.vbox.add(self.picker_orientation)
        self.picker_orientation.show()

        # Font size picker
        self.selector_font_size = hildon.TouchSelector(text=True)
        for i in constants.font_size_range:
            self.selector_font_size.append_text(str(i))
        self.picker_font_size = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                                      hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.picker_font_size.set_selector(self.selector_font_size)
        self.picker_font_size.set_alignment(0, 0.5, 0, 0)
        self.picker_font_size.set_title(_("Default font size"))
        self.vbox.add(self.picker_font_size)
        self.picker_font_size.show()

        # Load images check button
        self.auto_load_images = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.auto_load_images.set_label(_('Auto load images'))
        self.vbox.add(self.auto_load_images)
        self.auto_load_images.show()

        # Load settings
        self.auto_load_images.set_active(settings.auto_load_images)
        self.selector_orientation.set_active(0, settings.rotation_mode)
        try:
            font_size_index = constants.font_size_range.index(settings.default_font_size)
        except ValueError:
            # defaults to 16pt
            font_size_index = constants.font_size_range.index(16)
        self.selector_font_size.set_active(0, font_size_index)

class NewFeedDialog(gtk.Dialog):

    def __init__(self, parent):
        super(NewFeedDialog, self).__init__(_('New Feed'), parent,
                                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            (gtk.STOCK_ADD, gtk.RESPONSE_ACCEPT))

        self.entry = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.entry.set_input_mode(gtk.HILDON_GTK_INPUT_MODE_FULL)
        self.entry.connect('changed', self._entry_changed_cb)

        caption = gtk.HBox(False, 16)
        caption.pack_start(gtk.Label(_('Address') + ':'), False, False)
        caption.pack_start(self.entry)
        self.vbox.add(caption)

        self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

    def _entry_changed_cb(self, entry):
        if len(entry.get_text()) > 0:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, True)
        else:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

class FeedsWindow(hildon.StackableWindow):

    def __init__(self):
        super(FeedsWindow, self).__init__()

        # Init i18n stuff
        languages = []
        lc, encoding = locale.getdefaultlocale()
        if lc:
            languages = [lc]
        languages += constants.DEFAULT_LANGUAGES
        gettext.bindtextdomain(constants.RSS_COMPACT_NAME,
                               constants.LOCALE_DIR)
        gettext.textdomain(constants.RSS_COMPACT_NAME)
        language = gettext.translation(constants.RSS_COMPACT_NAME,
                                       constants.LOCALE_DIR,
                                       languages = languages,
                                       fallback = True)
        _ = language.gettext

        # Check that user dir exists
        if os.path.exists(constants.RSS_CONF_FOLDER) == False:
            os.mkdir(constants.RSS_CONF_FOLDER, 0700)
            os.mkdir(os.path.join(constants.RSS_CONF_FOLDER, 'icons'), 0700)

        # Feeds
        self.view = FeedsView(gtk.HILDON_UI_MODE_NORMAL)
        self.view.set_model (gtk.ListStore(gtk.gdk.Pixbuf, str, gobject.TYPE_PYOBJECT, int))
        self.view.connect ("row-activated", self._on_feed_activated)
        self.view.show()

        # Used for osso context in rotation object
        app_name = constants.RSS_NAME
        app_version = constants.RSS_VERSION

        # Auto rotate
        self.rotation_object = FremantleRotation(app_name, self, app_version,
                                                 FremantleRotation.AUTOMATIC)

        # Edit toolbar
        self.edit_toolbar_button_handler = 0
        self.edit_toolbar = hildon.EditToolbar()
        self.edit_toolbar.connect('arrow-clicked', self._restore_normal_mode)

        align = gtk.Alignment(0, 0, 1, 1)
        align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.show()

        align.add(pannable)
        self.add(align)
        align.show()

        self._create_menu()
        self.set_title(app_name)
        self.connect('delete-event', self._on_exit)
        hildon.Program.get_instance().add_window(self)

        # Apply settings
        settings.load()
        self.view.sort(settings.feeds_order)
        if settings.feeds_order == constants.DESCENDING_ORDER:
            self.descending_filter_button.set_active(True)

        # Load Feeds (could be none)
        try:
            manager.load(self._feed_data_loaded)
        except IOError:
            pass


    def _on_feed_activated(self, treeview, path, column):
        feed_iter = self.view.get_model().get_iter(path)
        feed_data = self.view.get_model().get_value(feed_iter, self.view.FEEDS_DATA_COLUMN)

        # Check that we're not trying to open a dummy feed
        if feed_data.status == self.view.DUMMY_FEED_STATUS:
            hildon.hildon_banner_show_information(self, '', _('Wait until feed is refreshed'))
            return

        news_window = EntriesWindow(feed_data)
        hildon.Program.get_instance().add_window(news_window)
        news_window.show()

        # When the news window is destroyed reset the unread cache
        news_window.connect('delete-event', self._on_news_window_destroyed,
                            gtk.TreeRowReference(self.view.get_model(), path))

    def _on_news_window_destroyed(self, news_window, event, row_reference):
        feed_iter = self.view.get_model().get_iter(row_reference.get_path())
        feed_data = self.view.get_model().get_value(feed_iter, self.view.FEEDS_DATA_COLUMN)

        # Reset unread cache (the value have most likely changed).
        unread_count = len([entry for entry in feed_data.entries if entry.read == False])
        self.view.get_model().set(feed_iter, self.view.FEEDS_READ_COLUMN, unread_count)

    def _on_exit(self, window, event):
        # Save the feeds status
        manager.save(self._on_feed_data_saved)
        settings.save()

    def _on_feed_data_saved(self, user_data):
        # Close application
        gtk.main_quit()

    def _create_menu(self):
        menu = hildon.AppMenu()

        # Sorting filter
        self.ascending_filter_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.ascending_filter_button.set_mode(False)
        self.ascending_filter_button.set_label(_('A-Z'))
        self.ascending_filter_button.connect('clicked', self._sort_ascending_cb)
        menu.add_filter(self.ascending_filter_button)
        self.descending_filter_button = \
            hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT,
                                  group = self.ascending_filter_button)
        self.descending_filter_button.set_mode(False)
        self.descending_filter_button.set_label(_('Z-A'))
        self.descending_filter_button.connect('clicked', self._sort_descending_cb)
        menu.add_filter(self.descending_filter_button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('New Feed'))
        button.connect('clicked', self._new_feed_cb)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Remove Feed'))
        button.connect('clicked', self._remove_feed_cb)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Update all'))
        button.connect('clicked', self._update_all_cb)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Settings'))
        button.connect('clicked', self._preferences_cb)
        menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)

    def _new_feed_cb(self, button):
        dialog = NewFeedDialog(self)
        dialog.connect('response', self._new_feed_response_cb)
        dialog.show_all()

    def _remove_feed_cb(self, button):

        # tree view edit mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_EDIT)
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.set_label(_('Select Feeds to remove'))
        self.edit_toolbar.set_button_label(_('Remove'))
        if self.edit_toolbar.handler_is_connected (self.edit_toolbar_button_handler):
            self.edit_toolbar.disconnect(self.edit_toolbar_button_handler)
        self.edit_toolbar_button_handler = \
            self.edit_toolbar.connect('button-clicked', self._remove_button_clicked_cb)

        self.set_edit_toolbar(self.edit_toolbar)
        self.edit_toolbar.show()
        self.fullscreen()

    def _remove_button_clicked_cb(self, button):
        selection = self.view.get_selection()
        selected_rows = selection.get_selected_rows()
        model, paths = selected_rows
        if not paths:
            hildon.hildon_banner_show_information(self, '', _('Please select at least one Feed'))
            return

        removed = []
        for path in paths:
            try:
                manager.remove_feed(model[path][FeedsView.FEEDS_DATA_COLUMN])
            except IOError:
                print 'Could not remove', constants.RSS_DB_FILE
                hildon.hildon_banner_show_information(self, '', _('File error while removing Feed'))
            else:
                removed.append(gtk.TreeRowReference(model, path))

        for reference in removed:
            model.remove(model.get_iter(reference.get_path()))

        # restore normal mode
        self._restore_normal_mode(button)

    def _restore_normal_mode(self, button):

        # tree view normal mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_NORMAL)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.hide()
        self.unfullscreen()

    def _preferences_cb(self, button):
        dialog = SettingsDialog(self)
        dialog.connect('response', self._preferences_response_cb)
        dialog.show_all()

    def _update_all_cb(self, button):
        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        # Ask manager to update feed
        manager.update_all(self._all_feeds_updated_cb, None)

    def _all_feeds_updated_cb(self, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def _new_feed_response_cb(self, dialog, response):

        if response == gtk.RESPONSE_ACCEPT:

            url = dialog.entry.get_text()

            # Quickly show feedback to user
            hildon.hildon_gtk_window_set_progress_indicator(self, True)

            # Insert a dummy row while information is retrieved
            dummy_feed_data, feed_iter = self.view.add_dummy_feed(url)
            path = self.view.get_model().get_path(feed_iter)
            row_reference = gtk.TreeRowReference(self.view.get_model(), path)

            # Add feed to manager
            manager.add_feed(url, self._feed_added_cb, row_reference)

            dialog.destroy()

    def _feed_added_cb(self, pixbuf, new_feed_data, row_reference):

        # Remove progress information
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        store = self.view.get_model()
        feed_iter = store.get_iter(row_reference.get_path())

        # Check that the feed data was retrieved
        if new_feed_data == None:
            dummy_data = store.get_value (feed_iter, self.view.FEEDS_DATA_COLUMN)
            message = _('Cannot retrieve feed from') + ' ' + dummy_data.href
            store.remove(feed_iter)
            hildon.hildon_banner_show_information(self, '', message)
            return

        # Update model
        store.set(feed_iter, self.view.FEEDS_ICON_COLUMN, pixbuf,
                  self.view.FEEDS_NAME_COLUMN, self.view.get_feed_title_markup(new_feed_data),
                  self.view.FEEDS_DATA_COLUMN, new_feed_data,
                  self.view.FEEDS_READ_COLUMN, len(new_feed_data.entries))

        # Take screenshot
        hildon.hildon_gtk_window_take_screenshot(self, False)
        hildon.hildon_gtk_window_take_screenshot(self, True)

    def _preferences_response_cb(self, dialog, response):

        if response == gtk.RESPONSE_ACCEPT:
            settings.auto_load_images = dialog.auto_load_images.get_active()
            settings.rotation_mode = dialog.selector_orientation.get_active(0)
            settings.default_font_size = int(dialog.selector_font_size.get_current_text())
            dialog.destroy()

            # Update rotation
            self.rotation_object.set_mode(settings.rotation_mode)

    def _feed_data_loaded(self, user_data):
        # TODO: migrate to "with" statement when available
        try:
            feed_data_list = manager.get_feed_list()
        except IOError:
            print 'cannot open', constants.RSS_DB_FILE
        else:
            store = self.view.get_model()

            for feed_data in feed_data_list:
                iter = store.append()
                pixbuf = manager.get_favicon(feed_data.feed.link or feed_data.href)
                unread_count = len([entry for entry in feed_data.entries if entry.read == False])
                store.set(iter, self.view.FEEDS_ICON_COLUMN, pixbuf,
                          self.view.FEEDS_NAME_COLUMN, self.view.get_feed_title_markup(feed_data),
                          self.view.FEEDS_DATA_COLUMN, feed_data,
                          self.view.FEEDS_READ_COLUMN, unread_count)
                # Improve responsiveness
#                while gtk.events_pending():
#                    gtk.main_iteration_do()

        # If there is no data then show the new feed dialog
        if len(feed_data_list) == 0:
            dialog = NewFeedDialog(self)
            dialog.connect('response', self._new_feed_response_cb)
            dialog.show_all()

    def _sort_ascending_cb(self, button):
        settings.feeds_order = constants.ASCENDING_ORDER
        self.view.sort(constants.ASCENDING_ORDER)

    def _sort_descending_cb(self, button):
        settings.feeds_order = constants.DESCENDING_ORDER
        self.view.sort(constants.DESCENDING_ORDER)

class EntriesView(hildon.GtkTreeView):

    ENTRIES_ICON_COLUMN = 0
    ENTRIES_NAME_COLUMN = 1
    ENTRIES_DATA_COLUMN = 2
    ENTRIES_DATE_COLUMN = 3

    def __init__(self, feed_title, ui_mode):
        super(EntriesView, self).__init__(ui_mode)

        self.fallback_author = feed_title

        # Add columns
        pix_renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn('Icon', pix_renderer, pixbuf = 0)
        self.append_column(column);

        text_renderer = gtk.CellRendererText()
        text_renderer.set_property('ellipsize', pango.ELLIPSIZE_END)
        column = gtk.TreeViewColumn('Name', text_renderer)
        column.set_cell_data_func(text_renderer, self._view_data_func)
        column.set_expand(True)
        self.append_column(column);

        date_renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn('Date', date_renderer, markup = self.ENTRIES_DATE_COLUMN)
        column.set_expand(False)
        self.append_column(column);

    def _view_data_func(self, column, renderer, model, iter):
        entry = model.get_value(iter, self.ENTRIES_DATA_COLUMN)

        # TODO: show entry.summary ?
        author = get_author_from_item(entry)
        if author == '':
            author = self.fallback_author

        words = entry.title.rsplit(":")
        if len(words) > 1:
            title = words[1].lstrip()
        else:
            title = entry.title

        if 'read' in entry and entry['read'] == True:
            color = secondary_color.to_string()
        else:
            color = active_color.to_string()

        renderer.set_property('markup',
                              '<span foreground="%s">%s</span>\n<span size="small">%s</span>'
                              % (color,
                                 gobject.markup_escape_text(author),
                                 gobject.markup_escape_text(title)))

    def get_visual_entry_date(self, entry):

        now = time.localtime()

        # Today
        if now.tm_yday == entry.updated_parsed.tm_yday:
            return '<span size="xx-small">%s</span>' % \
                time.strftime('%X', entry.updated_parsed)
        # This week
        elif now.tm_yday - entry.updated_parsed.tm_yday < 7 and \
                entry.updated_parsed.tm_wday < now.tm_wday:
            return '<span size="xx-small">%s</span>' % \
                time.strftime('%A', entry.updated_parsed)
        # This year
        elif now.tm_year == entry.updated_parsed.tm_year:
            return '<span size="xx-small">%s</span>' % \
                time.strftime('%d %B', entry.updated_parsed)
        # Anything else
        else:
            return '<span size="xx-small">%s</span>' % \
                time.strftime('%x', entry.updated_parsed)

class EntriesWindow(hildon.StackableWindow):

    def __init__(self, feed_data):
        super(EntriesWindow, self).__init__()

        self.feed_data = feed_data
        self.set_title(unescape(self.feed_data.feed.title))

        # Feeds
        self.view = EntriesView(feed_data.feed.title, gtk.HILDON_UI_MODE_NORMAL)
        self.view.set_model (gtk.ListStore(gtk.gdk.Pixbuf, str, gobject.TYPE_PYOBJECT, str))
        self.view.connect ("row-activated", self._on_entry_activated)
        self.view.show()

        model = self.view.get_model()

        for entry in self.feed_data.entries:
            entry_iter = model.append()
            model.set(entry_iter, 1, entry.title, 2, entry,
                      3, self.view.get_visual_entry_date(entry))

        align = gtk.Alignment(0, 0, 1, 1)
        align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.show()

        align.add(pannable)
        self.add(align)
        align.show()

        self._create_menu()

    def _create_menu(self):
        menu = hildon.AppMenu()

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Update Feed'))
        button.connect('clicked', self._update_feed_cb)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Mark all as Read'))
        button.connect('clicked', self._mark_all_read_cb)
        menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)

    def _mark_all_read_cb(self, button):
        for entry in self.feed_data.entries:
            entry['read'] = True

    def _on_entry_activated(self, treeview, path, column):
        item_window = ItemWindow(self.feed_data, treeview.get_model(), path)
        hildon.Program.get_instance().add_window(item_window)
        item_window.show()

    def _update_feed_cb(self, button):
        url = self.feed_data.href

        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        # Ask manager to update feed
        manager.update_feed(self.feed_data, self._feed_updated_cb, None)

    def _feed_updated_cb(self, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        # No need to update self.feed_data as it is automatically updated.
        # The same happens with the reference stored in the FeedsWindow :-)
        model = self.view.get_model()

        model.clear()
        for entry in self.feed_data.entries:
            entry_iter = model.append()
            model.set(entry_iter, 1, entry.title, 2, entry,
                      3, self.view.get_visual_entry_date(entry))

        return

class ItemWindowHeader(gtk.HBox):

    def __init__(self, homogeneous, spacing, item):
        super(ItemWindowHeader, self).__init__(homogeneous, spacing)

        self.item = item

        # Header
        header_vbox = gtk.VBox()
        navigation_hbox = gtk.HBox(False, 0)

        # Author and URL
        self.author = gtk.Label('')
        self.author.set_alignment(0,0.5)
        self.author.set_ellipsize(pango.ELLIPSIZE_END)
        # self.url = gtk.LinkButton('', _('View in Browser'))
        # self.url.set_alignment(0,0.5)
        header_vbox.pack_start(self.author, True, False)
#        header_vbox.pack_start(self.url, True, False)
        self.author.show()
#        self.url.show()
        self.set_item(item)
        header_vbox.show()

        # Navigation buttons. Sizes are important. We set the size of the icon but
        # we let the button grow as needed
        self.button_up = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button_up_img = gtk.Image()
        button_up_img.set_from_icon_name(constants.ICON_UP_NAME,
                                              gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.button_up.set_image(button_up_img)
        self.button_down = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button_down_img = gtk.Image()
        button_down_img.set_from_icon_name(constants.ICON_DOWN_NAME,
                                                gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.button_down.set_image(button_down_img)

        navigation_hbox.pack_start(self.button_up, False, False, 0)
        navigation_hbox.pack_start(self.button_down, False, False, 0)
        navigation_hbox.show_all()

        self.pack_start(header_vbox, True, True)
        self.pack_start(navigation_hbox, False, False)

        # Signals
        #self.url.connect('clicked', self._link_button_clicked)

    def _link_button_clicked(self, link_button):
        print 'Open ' + link_button.get_uri()

    def set_item(self, item):

        self.item = item

        # Mark item as read
        item['read'] = True

        self.author.set_text(get_author_from_item(item))
        #self.url.set_uri(item.link)

class ItemWindow(hildon.StackableWindow):

    def __init__(self, feed_data, model, path):
        super(ItemWindow, self).__init__()

        item_iter = model.get_iter(path)
        item = model.get_value(item_iter, EntriesView.ENTRIES_DATA_COLUMN)

        self.feed_data = feed_data
        self.model = model
        self.model_iter = model.get_iter(path)

        # Header
        self.item_window_header = ItemWindowHeader(False, 16, item)
        self.item_window_header.button_up.connect("clicked", self._up_button_clicked)
        self.item_window_header.button_down.connect("clicked", self._down_button_clicked)

        # HTML renderer
        self.view = webkit.WebView()
        wbk_settings = self.view.get_settings()
        # TODO: read this from settings
        wbk_settings.set_property('default_font_size', settings.default_font_size)
        wbk_settings.set_property('auto-load-images', settings.auto_load_images)
        wbk_settings.set_property('auto-shrink-images', True)
        self.set_item(item)

        # Pannable for showing content
        pannable = hildon.PannableArea()
        pannable.add(self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
        self.view.show()

        # Header Alignment
        header_align = gtk.Alignment(0, 0, 1, 1)
        header_align.set_padding(4, 0 , 16, 16)
        header_align.add(self.item_window_header)
        header_align.show()

        vbox = gtk.VBox(False, 8)
        vbox.pack_start(header_align, False, False)
        vbox.pack_start(pannable, True, True)
        self.item_window_header.show()
        pannable.show()

        self.add(vbox)
        vbox.show()

        self._create_menu()

    def _on_request_url(self, object, url, stream):
        if (url.lower().startswith("http")):
            request_thread = Thread(target=self._request_url_in_thread,
                                    args=(url, stream))
            request_thread.start()

    def _request_url_in_thread(self, url, stream):
        f = urllib2.urlopen(url)

        gtk.gdk.threads_enter()
        stream.write(f.read())
        stream.close()
        gtk.gdk.threads_leave()

    def _create_menu(self):
        menu = hildon.AppMenu()

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Item Details'))
        button.connect('clicked', self._item_details_cb)
        menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)


    def _add_to_table (self, table, color, key, value, y_coord):
        label = gtk.Label('')
        label.set_markup('<span foreground="%s">%s:</span>' % (color, key))
        label.set_alignment(1, 0)
        table.attach(label, 0, 1, y_coord, y_coord + 1)

        value_label = gtk.Label('')
        value_label.set_markup(value)
        value_label.set_alignment(0, 0)
        table.attach(gtk.Label(value), 1, 2, y_coord, y_coord + 1)

    def _item_details_cb(self, button):
        dialog = gtk.Dialog()
        dialog.set_title(_('Item Details'))

        pannable = hildon.PannableArea()
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)

        item = self.item_window_header.item

        row = 0;
        table = gtk.Table()
        table.set_col_spacing(0, 12)
        self._add_to_table (table, secondary_color.to_string(), _('Title'), item.title, row)
        row += 1
        self._add_to_table (table, secondary_color.to_string(), _('Published'),
                            time.strftime('%x', item.updated_parsed), row)

        pannable.add_with_viewport(table)

        table.show_all()
        pannable.show()
        dialog.vbox.add(pannable)

        dialog.show()

    def _up_button_clicked(self, button):

        # http://faq.pygtk.org/index.py?req=show&file=faq13.051.htp
        path = self.model.get_path(self.model_iter)
        position = path[-1]
        if position == 0:
            return
        prev_path = list(path)[:-1]
        prev_path.append(position - 1)
        self.model_iter = self.model.get_iter(tuple(prev_path))

        if self.model_iter == None:
            return

        item = self.model.get_value(self.model_iter, EntriesView.ENTRIES_DATA_COLUMN)
        self.set_item(item)

    def _down_button_clicked(self, button):

        if self.model_iter == None:
            return

        self.model_iter = self.model.iter_next(self.model_iter)

        item = self.model.get_value(self.model_iter, EntriesView.ENTRIES_DATA_COLUMN)
        self.set_item(item)

    def set_item(self, item):

        self.set_title(unescape(item.title))

        # Set item in header
        self.item_window_header.set_item(item)

        # Get content type and body
        content_type = 'text/html'
        if 'summary' in item:
            if 'summary_detail' in item and 'type' in item.summary_detail:
                content_type = item.summary_detail.type
            body = item.summary
        elif 'content' in item:
            # We are only considering the first content. TODO check more?
            if 'type' in item.content[0]:
                content_type = item.content[0].type
            body = item.content[0].value
        else:
            # Should never happen
            body = _('No text')

        # Write HTML
        self.view.load_string(body, content_type, self.feed_data.encoding, '')

        # Update button sensitiviness
        path = self.model.get_path(self.model_iter)
        position = path[-1]
        if position == 0:
            self.item_window_header.button_up.set_sensitive(False)
        elif position == len(self.model) - 1:
            self.item_window_header.button_down.set_sensitive(False)
        else:
            self.item_window_header.button_up.set_sensitive(True)
            self.item_window_header.button_down.set_sensitive(True)
