#!/usr/bin/env python
# -*- coding: utf-8 -*-

#########################################################################
#    Copyright (C) 2010 Sergio Villar Senin <svillar@igalia.com>
#
#    This file is part of ReSiStance
#
#    ReSiStance is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    ReSiStance is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with ReSiStance.  If not, see <http://www.gnu.org/licenses/>.
#########################################################################

import pango
import feedparser
import locale
import gettext
import gobject
import hildon
import pygtk
pygtk.require('2.0')
import gtk
import gtkhtml2
import os
from portrait import FremantleRotation
import constants
from feedmanager import FeedManager
# For unescape()
import re, htmlentitydefs
_ = gettext.gettext

manager = FeedManager()

# From http://effbot.org/zone/re-sub.htm#unescape-html
def unescape(text):
    def fixup(m):
        text = m.group(0)
        if text[:2] == "&#":
            # character reference
            try:
                if text[:3] == "&#x":
                    return unichr(int(text[3:-1], 16))
                else:
                    return unichr(int(text[2:-1]))
            except ValueError:
                pass
        else:
            # named entity
            try:
                text = unichr(htmlentitydefs.name2codepoint[text[1:-1]])
            except KeyError:
                pass
        return text # leave as is
    return re.sub("&#?\w+;", fixup, text)

def get_author_from_item(item):
    words = item.title.rsplit(":")
    if len(words) > 1:
        author = words[0].lstrip()
    elif 'publisher_detail' in item:
        author = item.publisher_detail.name
    elif 'author' in item:
        author = item.author
    else:
        author = ''

    return author

class FeedsView(hildon.GtkTreeView):

    FEEDS_ICON_COLUMN = 0
    FEEDS_NAME_COLUMN = 1
    FEEDS_DATA_COLUMN = 2

    DUMMY_FEED_STATUS = -1

    def __init__(self, ui_mode):
        super(FeedsView, self).__init__(ui_mode)

        text_renderer = gtk.CellRendererText()
        text_renderer.set_property('ellipsize', pango.ELLIPSIZE_END)

        pix_renderer = gtk.CellRendererPixbuf()

        # Add columns
        column = gtk.TreeViewColumn('Icon', pix_renderer, pixbuf = 0)
        self.append_column(column);

        column = gtk.TreeViewColumn('Name', text_renderer, text = self.FEEDS_NAME_COLUMN)
        column.set_cell_data_func(text_renderer, self._view_data_func)

        self.append_column(column);

    def _view_data_func(self, column, renderer, model, iter):
        name = model.get_value(iter, self.FEEDS_NAME_COLUMN)
        feed_data = model.get_value(iter, self.FEEDS_DATA_COLUMN)

        if 'link' in  feed_data.feed:
            url = feed_data.feed.link
        else:
            url = feed_data.href

        renderer.set_property('markup',
                              '%s\n<span size="x-small" style="italic">'
                              '%s</span>' % \
                              (name, url))

    def add_dummy_feed(self, url):

        store = self.get_model()
        feed_iter = store.append()
        dummy_feed_data = feedparser.FeedParserDict()
        dummy_feed_data.status = self.DUMMY_FEED_STATUS
        dummy_feed_data.href = url
        dummy_feed_data.feed = feedparser.FeedParserDict()
        dummy_feed_data.feed.title = _('Retrieving info...')
        store.set(feed_iter, 0, None, 1, dummy_feed_data.feed.title, 2, dummy_feed_data)

        return (dummy_feed_data,feed_iter)

class PreferencesDialog(gtk.Dialog):

    def __init__(self, parent):
        super(PreferencesDialog, self).__init__(_('Settings'), parent,
                                                gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                                (gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))


        self.selector_orientation = hildon.TouchSelector(text=True)
        for caption in FremantleRotation.MODE_CAPTIONS:
            self.selector_orientation.append_text(caption)
        self.selector_orientation.set_active(0, FremantleRotation.AUTOMATIC)
        self.picker_orientation = hildon.PickerButton(gtk.HILDON_SIZE_AUTO,
                                                      hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.picker_orientation.set_selector(self.selector_orientation)
        self.picker_orientation.set_alignment(0, 0.5, 0, 0)
        self.picker_orientation.set_title(_("Screen orientation"))
#        self.picker_orientation.connect("value-changed", on_picker_value_changed)

        self.vbox.add(self.picker_orientation)
        self.picker_orientation.show()

class NewFeedDialog(gtk.Dialog):

    def __init__(self, parent):
        super(NewFeedDialog, self).__init__(_('New Feed'), parent,
                                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            (gtk.STOCK_ADD, gtk.RESPONSE_ACCEPT))

        self.entry = hildon.Entry(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.entry.set_input_mode(gtk.HILDON_GTK_INPUT_MODE_FULL)
        self.entry.connect('changed', self._entry_changed_cb)

        caption = gtk.HBox(False, 16)
        caption.pack_start(gtk.Label(_('Address') + ':'), False, False)
        caption.pack_start(self.entry)
        self.vbox.add(caption)

        self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

    def _entry_changed_cb(self, entry):
        if len(entry.get_text()) > 0:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, True)
        else:
            self.set_response_sensitive(gtk.RESPONSE_ACCEPT, False)

class FeedsWindow(hildon.StackableWindow):

    def __init__(self):
        super(FeedsWindow, self).__init__()

        # Init i18n stuff
        languages = []
        lc, encoding = locale.getdefaultlocale()
        if lc:
            languages = [lc]
        languages += constants.DEFAULT_LANGUAGES
        gettext.bindtextdomain(constants.RSS_COMPACT_NAME,
                               constants.LOCALE_DIR)
        gettext.textdomain(constants.RSS_COMPACT_NAME)
        language = gettext.translation(constants.RSS_COMPACT_NAME,
                                       constants.LOCALE_DIR,
                                       languages = languages,
                                       fallback = True)
        _ = language.gettext

        # Check that user dir exists
        if os.path.exists(constants.RSS_CONF_FOLDER) == False:
            os.mkdir(constants.RSS_CONF_FOLDER, 0700)
            os.mkdir(os.path.join(constants.RSS_CONF_FOLDER, 'icons'), 0700)

        # Load Feeds (could be none)
        try:
            manager.load()
        except IOError:
            pass

        # Used for osso context in rotation object
        app_name = constants.RSS_NAME
        app_version = constants.RSS_VERSION

        # Auto rotate
        rotation_object = FremantleRotation(app_name, self,
                                                app_version, FremantleRotation.AUTOMATIC)

        # Edit toolbar
        self.edit_toolbar_button_handler = 0
        self.edit_toolbar = hildon.EditToolbar()
        self.edit_toolbar.connect('arrow-clicked', self._restore_normal_mode)

        # Feeds
        self.view = FeedsView(gtk.HILDON_UI_MODE_NORMAL)
        self.view.set_model (gtk.ListStore(gtk.gdk.Pixbuf, str, gobject.TYPE_PYOBJECT))
        self.view.connect ("row-activated", self._on_feed_activated)
        self.view.show()

        align = gtk.Alignment(0, 0, 1, 1)
        align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.show()

        align.add(pannable)
        self.add(align)
        align.show()

        self._create_menu()
        self.set_title(app_name)
        self.connect('delete-event', self._on_exit)
        hildon.Program.get_instance().add_window(self)

    def _on_feed_activated(self, treeview, path, column):
        feed_iter = self.view.get_model().get_iter(path)
        feed_data = self.view.get_model().get_value(feed_iter, self.view.FEEDS_DATA_COLUMN)

        # Check that we're not trying to open a dummy feed
        if feed_data.status == self.view.DUMMY_FEED_STATUS:
            hildon.hildon_banner_show_information(self, '', _('Wait until feed is refreshed'))
            return

        news_window = EntriesWindow(feed_data)
        hildon.Program.get_instance().add_window(news_window)
        news_window.show()

    def _on_exit(self, window, event):
        # Save the feeds status
        manager.save()

        # Close application
        gtk.main_quit()

    def _create_menu(self):
        menu = hildon.AppMenu()

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('New Feed'))
        button.connect('clicked', self._new_feed_cb)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Remove Feed'))
        button.connect('clicked', self._remove_feed_cb)
        menu.append(button)

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Update all'))
        button.connect('clicked', self._update_all_cb)
        menu.append(button)

        # Disabled for the moment as we don't handle settings yet
        # button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        # button.set_label(_('Settings'))
        # button.connect('clicked', self._preferences_cb)
        # menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)

    def _new_feed_cb(self, button):
        dialog = NewFeedDialog(self)
        dialog.connect('response', self._new_feed_response_cb)
        dialog.show_all()

    def _remove_feed_cb(self, button):

        # tree view edit mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_EDIT)
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.set_label(_('Select Feeds to remove'))
        self.edit_toolbar.set_button_label(_('Remove'))
        if self.edit_toolbar.handler_is_connected (self.edit_toolbar_button_handler):
            self.edit_toolbar.disconnect(self.edit_toolbar_button_handler)
        self.edit_toolbar_button_handler = \
            self.edit_toolbar.connect('button-clicked', self._remove_button_clicked_cb)

        self.set_edit_toolbar(self.edit_toolbar)
        self.edit_toolbar.show()
        self.fullscreen()

    def _remove_button_clicked_cb(self, button):
        selection = self.view.get_selection()
        selected_rows = selection.get_selected_rows()
        model, paths = selected_rows
        if not paths:
            hildon.hildon_banner_show_information(self, '', _('Please select at least one Feed'))
            return
        for path in paths:
            try:
                manager.remove_feed(model[path][FeedsView.FEEDS_DATA_COLUMN])
            except IOError:
                print 'Could not remove', constants.RSS_DB_FILE
                hildon.hildon_banner_show_information(self, '', _('File error while removing Feed'))
            else:
                model.remove(model.get_iter(path))

        # restore normal mode
        self._restore_normal_mode(button)

    def _restore_normal_mode(self, button):

        # tree view normal mode
        hildon.hildon_gtk_tree_view_set_ui_mode(self.view, gtk.HILDON_UI_MODE_NORMAL)
        self.view.get_selection().unselect_all()

        self.edit_toolbar.hide()
        self.unfullscreen()

    def _preferences_cb(self, button):
        dialog = PreferencesDialog(self)
        dialog.connect('response', self._preferences_response_cb)
        dialog.show_all()

    def _update_all_cb(self, button):
        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        # Ask manager to update feed
        manager.update_all(self._all_feeds_updated_cb, None)

    def _all_feeds_updated_cb(self, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def _new_feed_response_cb(self, dialog, response):

        if response == gtk.RESPONSE_ACCEPT:

            url = dialog.entry.get_text()

            # Quickly show feedback to user
            hildon.hildon_gtk_window_set_progress_indicator(self, True)

            # Insert a dummy row while information is retrieved
            dummy_feed_data, feed_iter = self.view.add_dummy_feed(url)
            path = self.view.get_model().get_path(feed_iter)
            row_reference = gtk.TreeRowReference(self.view.get_model(), path)

            # Add feed to manager
            manager.add_feed(url, self._feed_added_cb, row_reference)

            dialog.destroy()

    def _feed_added_cb(self, pixbuf, new_feed_data, row_reference):

        # Remove progress information
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        store = self.view.get_model()
        feed_iter = store.get_iter(row_reference.get_path())

        # Check that the feed data was retrieved
        if new_feed_data == None:
            dummy_data = store.get_value (feed_iter, self.view.FEEDS_DATA_COLUMN)
            message = _('Cannot retrieve feed from') + ' ' + dummy_data.href
            store.remove(feed_iter)
            hildon.hildon_banner_show_information(self, '', message)
            return

        # Update model
        store.set(feed_iter, 0, pixbuf, 1, new_feed_data.feed.title, 2, new_feed_data)

        # Take screenshot
        hildon.hildon_gtk_window_take_screenshot(self, False)
        hildon.hildon_gtk_window_take_screenshot(self, True)

    def _preferences_response_cb(self, dialog, response):

        if response == gtk.RESPONSE_ACCEPT:
            print 'save preferences'
            dialog.destroy()

    def load_data(self):
        # TODO: migrate to "with" statement when available
        try:
            feed_data_list = manager.get_feed_list()
        except IOError:
            print 'cannot open', constants.RSS_DB_FILE
        else:
            store = self.view.get_model()

            for feed_data in feed_data_list:
                iter = store.append()
                pixbuf = manager.get_favicon(feed_data.feed.link or feed_data.href)
                store.set(iter, 0, pixbuf, 1, feed_data.feed.title, 2, feed_data)

        # If there is no data then show the new feed dialog
        if len(feed_data_list) == 0:
            dialog = NewFeedDialog(self)
            dialog.connect('response', self._new_feed_response_cb)
            dialog.show_all()

class EntriesView(hildon.GtkTreeView):

    ENTRIES_ICON_COLUMN = 0
    ENTRIES_NAME_COLUMN = 1
    ENTRIES_DATA_COLUMN = 2

    def __init__(self, feed_title, ui_mode):
        super(EntriesView, self).__init__(ui_mode)

        self.fallback_author = feed_title
        text_renderer = gtk.CellRendererText()
        text_renderer.set_property('ellipsize', pango.ELLIPSIZE_END)

        pix_renderer = gtk.CellRendererPixbuf()

        # Add columns
        column = gtk.TreeViewColumn('Icon', pix_renderer, pixbuf = 0)
        self.append_column(column);

        column = gtk.TreeViewColumn('Name', text_renderer, text = self.ENTRIES_NAME_COLUMN)
        column.set_cell_data_func(text_renderer, self._view_data_func)
        self.append_column(column);

    def _view_data_func(self, column, renderer, model, iter):
        entry = model.get_value(iter, self.ENTRIES_DATA_COLUMN)

        # TODO: show entry.summary ?
        author = get_author_from_item(entry)
        if author == '':
            author = self.fallback_author

        words = entry.title.rsplit(":")
        if len(words) > 1:
            title = words[1].lstrip()
        else:
            title = entry.title

        if 'read' in entry and entry['read'] == True:
            color = self.style.lookup_color(constants.SECONDARY_TEXT_COLOR).to_string()
        else:
            color = self.style.lookup_color(constants.ACTIVE_TEXT_COLOR).to_string()

        renderer.set_property('markup',
                              '<span foreground="%s">%s</span>\n<span size="small">%s</span>'
                              % (color, author, title))


class EntriesWindow(hildon.StackableWindow):

    def __init__(self, feed_data):
        super(EntriesWindow, self).__init__()

        self.feed_data = feed_data
        self.set_title(unescape(self.feed_data.feed.title))

        # Feeds
        self.view = EntriesView(feed_data.feed.title, gtk.HILDON_UI_MODE_NORMAL)
        self.view.set_model (gtk.ListStore(gtk.gdk.Pixbuf, str, gobject.TYPE_PYOBJECT))
        self.view.connect ("row-activated", self._on_entry_activated)
        self.view.show()

        model = self.view.get_model()

        for entry in self.feed_data.entries:
            entry_iter = model.append()
            model.set(entry_iter, 1, entry.title, 2, entry)

        align = gtk.Alignment(0, 0, 1, 1)
        align.set_padding(4, 0 , 16, 16)

        pannable = hildon.PannableArea()
        pannable.add (self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        pannable.show()

        align.add(pannable)
        self.add(align)
        align.show()

        self._create_menu()

    def _create_menu(self):
        menu = hildon.AppMenu()

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Update Feed'))
        button.connect('clicked', self._update_feed_cb)

        menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)

    def _on_entry_activated(self, treeview, path, column):
        item_window = ItemWindow(self.feed_data, treeview.get_model(), path)
        hildon.Program.get_instance().add_window(item_window)
        item_window.show()

    def _update_feed_cb(self, button):
        url = self.feed_data.href

        # Quickly show feedback to user
        hildon.hildon_gtk_window_set_progress_indicator(self, True)

        # Ask manager to update feed
        manager.update_feed(self.feed_data, self._feed_updated_cb, None)

    def _feed_updated_cb(self, data):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

        # No need to update self.feed_data as it is automatically updated.
        # The same happens with the reference stored in the FeedsWindow :-)
        model = self.view.get_model()

        model.clear()
        for entry in self.feed_data.entries:
            entry_iter = model.append()
            model.set(entry_iter, 1, entry.title, 2, entry)

        return

class ItemWindowHeader(gtk.HBox):

    def __init__(self, homogeneous, spacing, item):
        super(ItemWindowHeader, self).__init__(homogeneous, spacing)

        self.item = item

        # Header
        header_vbox = gtk.VBox()
        navigation_hbox = gtk.HBox(False, 0)

        # Author and URL
        self.author = gtk.Label('')
        self.author.set_alignment(0,0.5)
        self.author.set_ellipsize(pango.ELLIPSIZE_END)
        self.url = gtk.LinkButton('', _('View in Browser'))
        self.url.set_alignment(0,0.5)
        header_vbox.pack_start(self.author, True, False)
        header_vbox.pack_start(self.url, True, False)
        self.author.show()
        self.url.show()
        self.set_item(item)
        header_vbox.show()

        # Navigation buttons. Sizes are important. We set the size of the icon but
        # we let the button grow as needed
        self.button_up = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button_up_img = gtk.Image()
        button_up_img.set_from_icon_name(constants.ICON_UP_NAME,
                                              gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.button_up.set_image(button_up_img)
        self.button_down = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button_down_img = gtk.Image()
        button_down_img.set_from_icon_name(constants.ICON_DOWN_NAME,
                                                gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.button_down.set_image(button_down_img)

        navigation_hbox.pack_start(self.button_up, False, False, 0)
        navigation_hbox.pack_start(self.button_down, False, False, 0)
        navigation_hbox.show_all()

        self.pack_start(header_vbox, True, True)
        self.pack_start(navigation_hbox, False, False)

        # Signals
        self.url.connect('clicked', self._link_button_clicked)

    def _link_button_clicked(self, link_button):
        print 'Open ' + link_button.get_uri()

    def set_item(self, item):
        # Mark item as read
        item['read'] = True

        self.author.set_text(get_author_from_item(item))
        self.url.set_uri(item.link)

class ItemWindow(hildon.StackableWindow):

    def __init__(self, feed_data, model, path):
        super(ItemWindow, self).__init__()

        item_iter = model.get_iter(path)
        item = model.get_value(item_iter, EntriesView.ENTRIES_DATA_COLUMN)

        self.feed_data = feed_data
        self.model = model
        self.model_iter = model.get_iter(path)

        # Header
        self.item_window_header = ItemWindowHeader(False, 16, item)
        self.item_window_header.button_up.connect("clicked", self._up_button_clicked)
        self.item_window_header.button_down.connect("clicked", self._down_button_clicked)

        # HTML renderer
        self.view = gtkhtml2.View()
        self.document = gtkhtml2.Document()
        self.view.set_document(self.document)
        self.set_item(item)

        # Alignment for padding
        align = gtk.Alignment(0, 0, 1, 1)
        align.set_padding(4, 0 , 16, 16)

        # Pannable for showing content
        pannable = hildon.PannableArea()
        pannable.add(self.view)
        pannable.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        self.view.show()

        vbox = gtk.VBox(False, 8)
        vbox.pack_start(self.item_window_header, False, False)
        vbox.pack_start(pannable, True, True)
        self.item_window_header.show()
        pannable.show()

        align.add(vbox)
        vbox.show()

        self.add(align)
        align.show()

#        self._create_menu()

    def _create_menu(self):
        menu = hildon.AppMenu()

        button = hildon.GtkButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        button.set_label(_('Item Window Empty'))
        menu.append(button)

        menu.show_all()
        self.set_app_menu(menu)

    def _up_button_clicked(self, button):

        # http://faq.pygtk.org/index.py?req=show&file=faq13.051.htp
        path = self.model.get_path(self.model_iter)
        position = path[-1]
        if position == 0:
            return
        prev_path = list(path)[:-1]
        prev_path.append(position - 1)
        self.model_iter = self.model.get_iter(tuple(prev_path))

        if self.model_iter == None:
            return

        item = self.model.get_value(self.model_iter, EntriesView.ENTRIES_DATA_COLUMN)
        self.set_item(item)

    def _down_button_clicked(self, button):

        if self.model_iter == None:
            return

        self.model_iter = self.model.iter_next(self.model_iter)

        item = self.model.get_value(self.model_iter, EntriesView.ENTRIES_DATA_COLUMN)
        self.set_item(item)

    def set_item(self, item):

        self.set_title(unescape(item.title))

        # Set item in header
        self.item_window_header.set_item(item)

        # Get content type and body
        content_type = 'text/html'
        if 'summary' in item:
            if 'summary_detail' in item and 'type' in item.summary_detail:
                content_type = item.summary_detail.type
            body = item.summary
        elif 'content' in item:
            # We are only considering the first content. TODO check more?
            if 'type' in item.content[0]:
                content_type = item.content[0].type
            body = item.content[0].value
        else:
            # Should never happen
            body = _('No text')

        # unicode -> encoding -> latin-1. In case of characters not belonging to
        # latin-1 do xmlcharrefreplace
        body = body.decode(self.feed_data.encoding).encode('latin-1', 'xmlcharrefreplace')

        # Write HTML
        self.document.clear()
        self.document.open_stream(content_type)
        self.document.write_stream("<html><body>%s</body></html>" % body)
        self.document.close_stream()
