/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "resources.h"
#include "printer.h"
#include "resourcesrequest.h"

namespace QYouTube {

Resources::Resources(QObject *parent) :
    QObject(parent),
    m_request(0),
    m_format("tsv")
{
}

QString Resources::apiKey() const {
    return m_key;
}

void Resources::setApiKey(const QString &key) {
    m_key = key;
}

QString Resources::accessToken() const {
    return m_token;
}

void Resources::setAccessToken(const QString &token) {
    m_token = token;
}

QString Resources::refreshToken() const {
    return m_refresh;
}

void Resources::setRefreshToken(const QString &token) {
    m_refresh = token;
}

QString Resources::outputFormat() const {
    return m_format;
}

void Resources::setOutputFormat(const QString &format) {
    m_format = format.toLower();
}

void Resources::del(const QString &id, const QString &resourcePath) {
    initRequest();

    if (m_request->status() == ResourcesRequest::Loading) {
        return;
    }

    m_request->del(id, resourcePath);
}

void Resources::insert(const QVariantMap &resource, const QString &resourcePath, const QStringList &part,
                       const QVariantMap &params) {
    initRequest();

    if (m_request->status() == ResourcesRequest::Loading) {
        return;
    }

    m_request->insert(resource, resourcePath, part, params);
}

void Resources::list(const QString &resourcePath, const QStringList &part, const QVariantMap &filters,
                     const QVariantMap &params) {
    initRequest();
    
    if (m_request->status() == ResourcesRequest::Loading) {
        return;
    }
    
    m_request->list(resourcePath, part, filters, params);
}

void Resources::update(const QString &resourcePath, const QVariantMap &resource, const QStringList &part) {
    initRequest();

    if (m_request->status() == ResourcesRequest::Loading) {
        return;
    }

    m_request->update(resourcePath, resource, part);
}

void Resources::initRequest() {
    if (m_request) {
        return;
    }

    m_request = new ResourcesRequest(this);
    m_request->setApiKey(apiKey());
    m_request->setAccessToken(accessToken());
    m_request->setRefreshToken(refreshToken());
    connect(m_request, SIGNAL(finished()), this, SLOT(onRequestFinished()));
}

void Resources::printResult() const {
    if (outputFormat() == "html") {
        Printer::printHtml(m_result);
    }
    else if (outputFormat() == "json") {
        Printer::printJson(m_result);
    }
    else if (outputFormat() == "xml") {
        Printer::printXml(m_result);
    }
    else {
        if (m_result.contains("items")) {
            const QVariantList items = m_result.value("items").toList();
            
            if (!items.isEmpty()) {
                Printer::printTsvHeader(items.first());
                Printer::printTsv(items);
            }
        }
        else {
            Printer::printTsvHeader(m_result);
            Printer::printTsv(m_result);
        }
    }
}

void Resources::onRequestFinished() {
    m_result = m_request->result().toMap();
    printResult();
    emit finished();
}

}
