/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "RestoreAccountStateWindow.h"

#include "CellularRadio.h"
#include "ModeWindow.h"

#include <QtDBus>
#include <QtGui>

using namespace Windows;

RestoreAccountStateWindow::RestoreAccountStateWindow(Settings &settings, QWidget *parent) :
	BaseWindow(settings, parent)
{
	if(CellularRadio::CurrentState() == false && NumEnabledAccounts() == 0)
		CreateContents();
	else
		Advance();
}

void RestoreAccountStateWindow::CreateContents()
{
	switch(CurrentSettings().AppMode())
	{
		case Settings::APPMODE_GUI:
		{
			QVBoxLayout *layout = new QVBoxLayout();
			{
				layout->setMargin(40);

				QLabel *lblNotice1 = new QLabel(tr("You have no enabled accounts and the cellular modem is disabled. If this was the result of the last crash of this program, please restore them as you wish."));
				lblNotice1->setAlignment(Qt::AlignCenter);
				lblNotice1->setWordWrap(true);
				layout->addWidget(lblNotice1, 1);

				QLabel *lblNotice2 = new QLabel(tr("Once you have enabled things as you want them, tap next to proceed."));
				lblNotice2->setWordWrap(true);
				lblNotice2->setAlignment(Qt::AlignCenter);
				layout->addWidget(lblNotice2, 1);

				QHBoxLayout *layoutButtons = new QHBoxLayout();
				{
					QPushButton* btnToggleModem = new QPushButton(tr("Toggle Cellular Modem"));
					QObject::connect(btnToggleModem, SIGNAL(clicked()), this, SLOT(ToggleModem()));
					layoutButtons->addWidget(btnToggleModem);
					layoutButtons->setAlignment(btnToggleModem, Qt::AlignHCenter);

					QLabel *lblSpacer = new QLabel();
					layoutButtons->addWidget(lblSpacer);
					layoutButtons->setAlignment(lblSpacer, Qt::AlignHCenter | Qt::AlignVCenter);

					QPushButton* btnSetAccountStatus = new QPushButton(tr("Set Account Status"));
					QObject::connect(btnSetAccountStatus, SIGNAL(clicked()), this, SLOT(SetAccountStatus()));
					layoutButtons->addWidget(btnSetAccountStatus);
					layoutButtons->setAlignment(btnSetAccountStatus, Qt::AlignHCenter);
				}
				layout->addLayout(layoutButtons);

				QPushButton* btnProceed = new QPushButton(tr("Next"));
				QObject::connect(btnProceed, SIGNAL(clicked()), this, SLOT(Proceed()));
				layout->addWidget(btnProceed);
				layout->setAlignment(btnProceed, Qt::AlignHCenter);

				centralWidget()->setLayout(layout);
			}

			break;
		}

		case Settings::APPMODE_CONSOLE:
		{
			// Process args.

			Advance();
			break;
		}
	}
}

void RestoreAccountStateWindow::ToggleModem()
{
	CellularRadio::Toggle();
}

void RestoreAccountStateWindow::SetAccountStatus()
{
	const char * NOKIA_SERVICE_ACCOUNTS_UI("com.nokia.AccountsUI");
	const char * NOKIA_PATH_ACCOUNTS_UI("/com/nokia/AccountsUI");
	const char * NOKIA_IFACE_ACCOUNTS_UI("com.nokia.Accounts");

	QDBusMessage msg(QDBusMessage::createMethodCall(NOKIA_SERVICE_ACCOUNTS_UI, NOKIA_PATH_ACCOUNTS_UI, NOKIA_IFACE_ACCOUNTS_UI, "OpenAccountsList"));
	msg << (uint)winId();
	QDBusConnection::sessionBus().call(msg);
}

void RestoreAccountStateWindow::Proceed()
{
	Advance();
}

void RestoreAccountStateWindow::Advance()
{
	QWidget* next = new Windows::ModeWindow(CurrentSettings());
	next->show();
	close();
}

const uint RestoreAccountStateWindow::NumEnabledAccounts()
{
	const char * TP_ACCOUNT_PHONE("/org/freedesktop/Telepathy/Account/ring/tel/ring");
	const char * TP_IFACE_ACCOUNT("org.freedesktop.Telepathy.Account");
	const char * TP_IFACE_ACCOUNT_MANAGER("org.freedesktop.Telepathy.AccountManager");
	const char * TP_PATH_ACCOUNT_MANAGER("/org/freedesktop/Telepathy/AccountManager");
	const char * TP_IFACE_DBUS_PROPERTIES("org.freedesktop.DBus.Properties");

	uint numEnabledAccounts(0);

	QDBusInterface tpAccountManager(TP_IFACE_ACCOUNT_MANAGER, TP_PATH_ACCOUNT_MANAGER, TP_IFACE_DBUS_PROPERTIES);
	if (tpAccountManager.isValid())
	{
		// Grab state
		QDBusMessage accountList = tpAccountManager.call("Get", TP_IFACE_ACCOUNT_MANAGER, "ValidAccounts");
		foreach (QVariant arg, accountList.arguments())
		{
			const QDBusArgument argument(qvariant_cast<QDBusVariant>(arg).variant().value<QDBusArgument>());
			argument.beginArray();
			while (!argument.atEnd())
			{
				// Grab the account path
				QDBusObjectPath accountPath;
				argument >> accountPath;

				// Ignore the phone account
				if(accountPath.path() == TP_ACCOUNT_PHONE)
					continue;
				else
				{
					// Grab the enabled property of each account
					QDBusInterface tpAccount(TP_IFACE_ACCOUNT_MANAGER, accountPath.path(), TP_IFACE_DBUS_PROPERTIES);
					if (tpAccount.isValid())
					{
						QDBusReply<QVariant> isEnabledProp = tpAccount.call("Get", TP_IFACE_ACCOUNT, "Enabled");

						if(isEnabledProp.value().toBool())
							++numEnabledAccounts;
					}
				}
			}
			argument.endArray();
		}
	}

	return numEnabledAccounts;
}
