/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "SMS.h"

#include "Attachment.h"
#include "DBBackends/RtcomEventLogger.h"
#include "EventParsers/VMGEntities/VMessage.h"
#include "NumberToNameLookup.h"
#include "Settings.h"

#include <QDateTime>
#include <QDir>
#include <QFile>
#include <QList>
#include <QRegExp>
#include <QString>
#include <QTextStream>

#include <utime.h>

#include <rtcom-eventlogger/event.h>
#include <rtcom-eventlogger/eventlogger-attach-iter.h>

using namespace EventTypes;

const DBBackends::iDBBackend &SMS::DB() const
{
	return DBBackends::RtcomEventLogger(CurrentSettings(), *this);
}

SMS::SMS(const Settings &settings) :
	m_Settings(settings)
{
}

SMS::~SMS()
{
	foreach(QSharedPointer<Attachment> attachment, m_Attachments)
	{
		attachment.clear();
	}
}

SMS::SMS(const Settings &settings, const RTComElEvent &event, const QList<RTComElAttachment*> attachments) :
	m_Settings(settings)
{
	setVersion(2.1);
	setIsRead(event.fld_is_read);
	setDestination(event.fld_outgoing ? SENT : INBOX);
	setTimestamp(QDateTime::fromTime_t(event.fld_start_time).toUTC());
	setTel(event.fld_remote_uid);
	if(Tel().indexOf("0") == 0)
		setTel(QString(Tel()).replace(QRegExp("^0"), "+44"));
	//setContents(QString(event.fld_free_text).replace("\n", QChar(0x2029)));
	setContents(QString::fromUtf8(event.fld_free_text));

	// We directly access the m_Attachments member variable here rather than the
	// accessor as the accessor is const
	if(attachments.count() > 0)
		foreach(RTComElAttachment *attachment, attachments)
			m_Attachments.append(new Attachment(*attachment));
}
#include <QDebug>
const uint SMS::HashCode() const
{
//	qDebug() << Timestamp().toUTC().toTime_t() << ", " << qHash(Tel()) << ", " << qHash(Destination()) << ", " << qHash(Contents()) << ", " << Attachments().HashCode();

//	foreach(QChar c, Contents().toUtf8())
//	{
//		qDebug() << c.unicode();
//	}

	return
		Timestamp().toUTC().toTime_t() ^
		qHash(Tel()) ^
		qHash(Destination()) ^
		qHash(Contents()) ^
		Attachments().HashCode();
}

RTComElEvent * SMS::toRTComEvent(const NumberToNameLookup &numberToNameLookup) const
{
	QString groupId((Tel().length() < 7 || Tel().indexOf(QRegExp("[:alpha:]+")) > -1)
		? Tel()
		: Tel().right(7));

	RTComElEvent *event(rtcom_el_event_new());
	memset(event, 0, sizeof(RTComElEvent));

	RTCOM_EL_EVENT_SET_FIELD (event, service, g_strdup("RTCOM_EL_SERVICE_SMS"));
	RTCOM_EL_EVENT_SET_FIELD (event, event_type, g_strdup("RTCOM_EL_EVENTTYPE_SMS_MESSAGE"));
	RTCOM_EL_EVENT_SET_FIELD (event, storage_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, start_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, end_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, is_read, IsRead() ? 1 : 0);
	RTCOM_EL_EVENT_SET_FIELD (event, outgoing, Destination() == SMS::SENT ? 1 : 0);
	RTCOM_EL_EVENT_SET_FIELD (event, local_uid, g_strdup("ring/tel/ring"));
	//RTCOM_EL_EVENT_SET_FIELD (&event, local_name, g_strdup("<SelfHandle>"));
	RTCOM_EL_EVENT_SET_FIELD (event, remote_uid, g_strdup(Tel().toUtf8()));
	//RTCOM_EL_EVENT_SET_FIELD (&event, remote_name, g_strdup(QString::number(numberToNameLookup.ContactDetails().value(Tel()).second).toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, remote_ebook_uid, g_strdup(QString::number(numberToNameLookup.ContactDetails().value(Tel()).first).toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, group_uid, g_strdup(groupId.toUtf8()));
	//RTCOM_EL_EVENT_SET_FIELD (event, free_text, g_strdup(Contents().replace(0x2029, "\n").toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, free_text, g_strdup(Contents().toUtf8()));

	return event;
}

void SMS::Export(const QString &baseDirectory) const
{
	// Build the path and ensure it exists...
	QString eventFilename(baseDirectory);
	eventFilename += Destination() == EventTypes::SMS::SENT ? "/Sent/" : "/Inbox/";
	eventFilename += QString::number(Timestamp().toUTC().date().year()) + "/";
	QDir().mkpath(eventFilename);

	// ...then build the filename and open it.
	eventFilename += QString::number(Timestamp().toUTC().toTime_t()) + ".vmg";
	QFile data(eventFilename);
	if (data.open(QFile::WriteOnly | QFile::Truncate))
	{
		QTextStream stream(&data);

		QTextCodec *oldCodec = stream.codec();
		stream.setAutoDetectUnicode(false);
		stream.setCodec("UTF-16LE");

		EventParsers::VMGEntities::VMessage writer(NULL, 1.1);
		writer.Write(stream, *this);
//stream << "Test";
		//stream.setCodec(oldCodec);
		stream.flush();
		data.close();

		utimbuf fileTimes;
		fileTimes.modtime = Timestamp().toUTC().toTime_t();
		utime(eventFilename.toAscii(), &fileTimes);
	}
}

QDebug operator<<(QDebug dbg, SMS& event)
{
	dbg.nospace() << "\tFolder:\t\t" << (event.Destination() == SMS::SENT ? "Sent" : "Inbox") << "\n";
	dbg.nospace() << "\tTimestamp:\t" << event.Timestamp().toUTC() << "\n";
	dbg.nospace() << "\tRemote-Tel:\t" << event.Tel() << "\n";
	//dbg.nospace() << "\tremote-name:\t" << event.fld_remote_name << "\n";
	dbg.nospace() << "\tIs-read:\t\t" << (event.IsRead() ? "true" : "false") << "\n";
	dbg.nospace() << "\tContents:\t" << event.Contents() << "\n";

	return dbg;
}
