/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "VBody.h"

#include "Attachment.h"
#include "Factory.h"
#include "EventTypes/SMS.h"

#include <QDateTime>
#include <QDir>
#include <QTextStream>

#include <QDebug>

using namespace EventParsers::VMGEntities;

VBody::VBody(const SMSEntity* parent) :
	SMSEntity(parent)
{
}

//VBody::VBody(QTextStream& stream)
//{
//}

VBody::~VBody()
{
}

void VBody::Write(QTextStream &stream, const EventTypes::SMS &event)
{
	stream << "BEGIN:" << getTagName() << endl;

	// First, the event's date field.
	stream << "Date:" << event.Timestamp().toString("d.M.yyyy hh:mm:ss") << endl;

	// ...next, the event's attachments
	foreach(QSharedPointer<Attachment> attachment, event.Attachments())
		stream << attachment->Stream().readAll() << endl;

	// ...and now the event's contents
	stream << event.Contents() << endl;

	stream << "END:" << getTagName() << endl;
}

bool VBody::Read(const QString &initialLine, QTextStream &stream, EventTypes::SMS &event)
{
	bool hasEnded(false);
	float version(0);
	QString text;

	// Stream may or may not have a 'BEGIN' present. Swallow it if it's ours.
	uint linePos(stream.pos());
	QString lineData(initialLine.length() > 0 ? initialLine : stream.readLine());
	if(lineData.startsWith("BEGIN:"))
	{
		if(lineData != QString("BEGIN:") + getTagName())
		{
			qDebug() << "Invalid stream";
			return false;
		}
		else
		{
			if(isAttachment())
			{
			}
			else
			{
				// ...discard this line
				lineData = stream.readLine();
			}
		}
	}

	do
	{
		if(lineData.startsWith("Date:"))
		{
			version = lineData.mid(lineData.indexOf(":")+1).toFloat();
		}
		else if(lineData.startsWith("BEGIN:"))
		{
			iReader* reader = Factory::Instantiate(lineData, this);
			bool valid(NULL != reader && reader->Read(lineData, stream, event));
			delete reader;

			// Quit processing if the nested content is not valid
			if(!valid)
				return valid;
		}
		else if(lineData.startsWith("END:"))
		{
			if(lineData != QString("END:") + getTagName())
			{
				qDebug() << getTagName() << " parser mismatch error: " << lineData;
				return false;
			}
			else
			{
				hasEnded = true;
				break;
			}
		}
		else
		{
			if(text.isEmpty() && !lineData.isEmpty() && lineData.at(0) == 1)
			{
				// It's a binary SMS. Oh joy.
				if(lineData.count() >= 5)
				{
					int wspMIMETypeID = lineData.at(5).unicode() & 0x7F;
					if(BinaryMIMETypes().contains(wspMIMETypeID))
					{
						QString mimeType(BinaryMIMETypes().value(wspMIMETypeID));
						qDebug() << "Attachment is: " << mimeType;
						Attachment *binaryAttachment(new Attachment(
							(QDir::tempPath() + "/attachment-" + QString::number(event.Timestamp().toTime_t()) + "-" + QString::number(event.Attachments().count()) + ".bin").toUtf8(),
							mimeType));

						// Grab the content

						// Alias the stream's device so we can asily access it
						// directly in binary mode
						QIODevice &binfile(*stream.device());

						// Were going to ignore this data for the moment, but
						// get an approximation of the range in the file.
						int startPos(linePos);
						stream.readLine();
						int endPos(stream.pos());
						binfile.seek(startPos);

						qDebug() << "Binary offsets in file: " << startPos << ", " << endPos;

						// Prepare our pointers for storing the data
						int binDataLength((endPos - startPos) / sizeof(quint16));
						char binaryData[binDataLength];
						memset(&binaryData, 0, binDataLength);
						char *binaryDataByte(binaryData);

						// Grab the content from the file a byte at a time,
						// skipping over every alternate byte (VMGs are UTF16LE)
						// ...yes, even the binary data :(
						int curpos(startPos);
						while(binfile.pos() < endPos)
						{
							binfile.seek(curpos);
							binfile.read(binaryDataByte++, 1);

							// Look at the byte just stored. If it's an EOL,
							// we're done. Even binary data ends on a valid line.
							if(*(binaryDataByte - 1) == 0x0A)
							{
								// Note where the data actually ended.
								binDataLength = (binfile.pos() - startPos) / sizeof(quint16);
								break;
							}

							qDebug() << hex << (int)binfile.pos() << "/" << hex << endPos << ": " << hex << (int)*(binaryDataByte - 1);

							// Advance, skipping over every other byte.
							curpos += sizeof(quint16);
						}
						// Move back a bit so the stream is ready to continue in
						// text mode once we're done with the attachment.
						stream.seek(endPos - 1);

						// Directly access the output device and dump the binary
						// data into it.
						// NOTE: This is why the QFile needs to be unbuffered
						binaryAttachment->Stream().device()->write(binaryData, binDataLength);

						// Save attachment
						event.Attachments().append(binaryAttachment);
					}
					else
					{
						qDebug() << "Unrecognised binary mime type: " << hex << wspMIMETypeID;
						return false; // Not supported for now
					}
				}
				else
				{
					qDebug() << "Binary attachment too short (" << lineData.count() << " bytes). Not supported.";
					return false; // Not supported for now
				}
			}
			else
			{
				// If this isn't the first line, add a newline.
				if(!text.isEmpty())
					text.append(0x0A);
				text.append(lineData);
			}
		}

		linePos = stream.pos();
		lineData = stream.readLine();
	}while(!hasEnded && !stream.atEnd());

	if(hasEnded)
	{
		event.setContents(text);
		//event.fld_storage_time
	}

	return true;
}
