/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "Attachment.h"

#include <QCryptographicHash>
#include <QDir>
#include <QFile>
#include <QTextStream>

#include <stdexcept>

#include <sys/mman.h>
#include <fcntl.h>

#include <rtcom-eventlogger/eventlogger-attach-iter.h>

Attachment::Attachment(QString path, QString description) :
	m_Path(path),
	m_Description(description),
	m_FileHandle(m_Path),
	m_DeleteFileOnDestruction(true)
{
	if(m_FileHandle.open(QFile::WriteOnly | QFile::Unbuffered))
	{
		m_Stream = new QTextStream(&m_FileHandle);
		if(m_Stream != NULL)
		{
			m_Stream->setCodec("UTF8");
		}
		else
			throw std::runtime_error(QString("Unable to create attachment stream: '%1'").arg(m_Path).toStdString());
	}
	else
		throw std::runtime_error(QString("Unable to open attachment file: '%1'. The reason was: %2").arg(m_Path).arg(m_FileHandle.errorString()).toStdString());
}

Attachment::~Attachment()
{
	if(m_Stream != NULL)
		delete m_Stream;

	m_FileHandle.close();

	if(m_DeleteFileOnDestruction && m_Path.length() > 0)
		QFile::remove(m_Path);
}

Attachment::Attachment(const _RTComElAttachment& attachment) :
	m_Path(attachment.path),
	m_Description(attachment.desc),
	m_FileHandle(m_Path),
	m_DeleteFileOnDestruction(attachment.id < 1) // Don't delete the file if we have a valid attachment ID
{
	if(m_FileHandle.open(QFile::ReadWrite))
	{
		m_Stream = new QTextStream(&m_FileHandle);
		if(m_Stream != NULL)
		{
			m_Stream->setCodec("UTF8");
		}
		else
			throw std::runtime_error(QString("Unable to create attachment stream: '%1'").arg(m_Path).toStdString());
	}
	else
		throw std::runtime_error(QString("Unable to open attachment file: '%1'. The reason was: %2").arg(m_Path).arg(m_FileHandle.errorString()).toStdString());
}

RTComElAttachment * Attachment::toRTComAttachment() const
{
	RTComElAttachment *attachment(
		rtcom_el_attachment_new(
			g_strdup(Path().toUtf8()),
			g_strdup(Description().toUtf8())));
	return attachment;
}

void Attachment::freeRTComContents(RTComElAttachment &attachment)
{
	g_free(attachment.path);
	attachment.path = NULL;

	g_free(attachment.desc);
	attachment.desc = NULL;
}

const uint Attachment::HashCode() const
{
	qDebug() << "Hashing attachment: " << Path();

	QFile file(Path());
	if(file.open(QFile::ReadOnly))
	{
		uchar *memory(file.map(0, file.size()));
		if (memory)
		{
			QByteArray data;
			data.fromRawData((char*)memory, file.size());
			uint hashcode = QCryptographicHash::hash(data, QCryptographicHash::Md5).toUInt();

			file.unmap(memory);

			return hashcode;
		}
		else
			throw std::runtime_error(QString("Unable to map attachment file: %1").arg(Path()).toStdString());
	}
	else
		throw std::runtime_error(QString("Unable to open attachment file: '%1'. The reason was: %2").arg(m_Path).arg(m_FileHandle.errorString()).toStdString());
}

QDebug operator<<(QDebug dbg, Attachment& attachment)
{
	dbg.nospace() << "Path:\t" << attachment.Path() << "\n";
	dbg.nospace() << "Desc:\t" << attachment.Description() << "\n";

	return dbg;
}
